<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Traits\HasPermissions;
use App\Traits\ApiResponse;
use App\Helpers\PublicHelper;
use App\Http\Resources\UserResource;
use Illuminate\Support\Facades\Log;
use App\Helpers\PermissionHelper;

class UserController extends Controller
{
    use HasPermissions, ApiResponse;

    /**
     * Display a listing of users
     */
    public function index(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $owenerId = $tokenModel['owenerId'] ?? null;
        $userId = $tokenModel['userId'] ?? null;

        $users = \DB::table('vts_users as u')
            ->leftJoin('vts_member_types as mt', 'u.typ', '=', 'mt.id')
            ->select('u.*', 'mt.name as member_type_name')
            ->where('u.mid', $owenerId)
            ->where('u.id', '!=', $userId)
            ->get();

        // Add member_type_id to each user
        $usersWithMemberTypes = $users->map(function ($user) {
            $userData = (array) $user;
            $userData['member_type_id'] = $user->typ;
            return $userData;
        });

        //Log::info('UserController>index: ', ['users' => $usersWithMemberTypes]);

        return response()->json([
            'success' => true,
            'data' => $usersWithMemberTypes,
            'message' => 'Users retrieved successfully'
        ]);
    }

    /**
     * Store a newly created user
     */
    public function store(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $userId = $tokenModel['userId'] ?? null;
        $owenerId = $tokenModel['owenerId'] ?? null;
        $sid = $tokenModel['storeId'] ?? null;


        $request->validate([
            'nm' => 'required|string|max:255',
            'email' => 'required|email|unique:vts_users,ml',  // Database column is 'ml'
            'password' => 'required|string|min:6'
        ]);

        $user = User::create([
            'nm' => $request->nm,
            'ml' => $request->email,  // Map 'email' request to 'ml' database column
            'psw' => $request->password,  // Map 'password' to 'psw' database column
            'mid' => $owenerId,
            'sid' => $sid,
            'typ'=> 4,
            'created_at' => now(),
            'created_by' => $userId
        ]);

        return response()->json([
            'success' => true,
            'data' => $user,
            'message' => 'User created successfully'
        ]);
    }

    /**
     * Display the specified user
     */
    public function show(User $user)
    {
        $userWithMemberType = \DB::table('vts_users as u')
            ->leftJoin('vts_member_types as mt', 'u.typ', '=', 'mt.id')
            ->select('u.*', 'mt.name as member_type_name')
            ->where('u.id', $user->id)
            ->first();

        if (!$userWithMemberType) {
            return $this->error('User not found', 404);
        }

        // Convert to array and add member_type_id
        $userData = (array) $userWithMemberType;
        $userData['member_type_id'] = $userWithMemberType->typ;

        return response()->json([
            'success' => true,
            'data' => $userData,
            'message' => 'User retrieved successfully'
        ]);
    }

    /**
     * Update the specified user
     */
    public function update(Request $request, User $user)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        $userId = $tokenModel['userId'] ?? null;



        $request->validate([
            'nm' => 'required|string|max:255',
            'email' => 'required|email|unique:vts_users,ml,' . $user->id,  // Database column is 'ml'
            'password' => 'nullable|string|min:6'
        ]);

        $userData = [
            'nm' => $request->nm,
            'ml' => $request->email,  // Map 'email' request to 'ml' database column
            'updated_at' => now(),
            'updated_by' => $userId
        ];

        if ($request->filled('password')) {
            $userData['psw'] = $request->password;  // Map 'password' to 'psw' database column
        }

        $user->update($userData);

        return response()->json([
            'success' => true,
            'data' => $user,
            'message' => 'User updated successfully'
        ]);
    }

    /**
     * Remove the specified user
     */
    public function destroy(User $user)
    {
        // Prevent self-deletion
        $currentUser = $this->getCurrentUser();
        if ($currentUser && $currentUser->id === $user->id) {
            return $this->error('Cannot delete your own account', 400);
        }

        $user->delete();

        $user->directPermissions()->delete();

        return response()->json([
            'success' => true,
            'data' => null,
            'message' => 'User deleted successfully'
        ]);
    }

    /**
     * Get current user's information
     */
    public function me()
    {
        $tokenModel = PublicHelper::DecodeJWT(request());
        $userId = $tokenModel['userId'] ?? null;

        if (!$userId) {
            return $this->error('User not found', 404);
        }

        $user = \DB::table('vts_users as u')
            ->leftJoin('vts_member_types as mt', 'u.typ', '=', 'mt.id')
            ->select('u.*', 'mt.name as member_type_name')
            ->where('u.id', $userId)
            ->first();

        if (!$user) {
            return $this->error('User not found', 404);
        }

        // Convert to array and add member_type_id
        $userData = (array) $user;
        $userData['member_type_id'] = $user->typ;

        return response()->json([
            'success' => true,
            'data' => $userData,
            'message' => 'User profile retrieved successfully'
        ]);
    }

    /**
     * Update current user's profile
     */
    public function updateProfile(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT(request());
        $userId = $tokenModel['userId'] ?? null;

        if (!$userId) {
            return $this->error('User not found', 404);
        }

        $user = User::find($userId);
        if (!$user) {
            return $this->error('User not found', 404);
        }

        $request->validate([
            'nm' => 'required|string|max:255',
            'email' => 'required|email|unique:vts_users,ml,' . $user->id,  // Database column is 'ml'
            'password' => 'nullable|string|min:6',
        ]);

        $userData = [
            'nm' => $request->nm,
            'ml' => $request->email,  // Map 'email' request to 'ml' database column
            'updated_at' => now(),
            'updated_by' => $userId
        ];

        if ($request->filled('password')) {
            $userData['psw'] = $request->password;  // Map 'password' to 'psw' database column
        }

        $user->update($userData);

        return response()->json([
            'success' => true,
            'data' => $user,
            'message' => 'Profile updated successfully'
        ]);
    }

    /**
     * Get all available permissions for user assignment
     */
    public function getAvailablePermissions()
    {
        // add manual permission
        $permissions = PermissionHelper::getAvailablePermissions();

        return response()->json([
            'success' => true,
            'data' => $permissions,
            'message' => 'Permissions retrieved successfully'
        ]);
    }

    /**
     * Get user's direct permissions
     */
    public function getUserDirectPermissions(User $user)
    {


        $permissions = $user->getDirectPermissionsArray();

        return response()->json([
            'success' => true,
            'data' => $permissions,
            'message' => 'User permissions retrieved successfully'
        ]);
    }

    /**
     * Update user's direct permissions
     */
    public function updateUserDirectPermissions(Request $request, User $user)
    {
                    
        $request->validate([
            'permissions' => 'required|array'
        ]);
        $user->syncDirectPermissions($request->permissions);

        return response()->json([
            'success' => true,
            'data' => $user->getDirectPermissionsArray(),
            'message' => 'User permissions updated successfully'
        ]);
    }

    /**
     * Upload profile photo for current user
     */
    public function uploadProfilePhoto(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT(request());
        $userId = $tokenModel['userId'] ?? null;

        if (!$userId) {
            return $this->error('User not found', 404);
        }

        $user = User::find($userId);
        if (!$user) {
            return $this->error('User not found', 404);
        }

        $request->validate([
            'profile_photo' => 'required|image|mimes:jpeg,png,jpg,gif|max:5120' // 5MB max
        ]);

        try {
            // Delete old profile photo if exists
            if ($user->img_url && file_exists(public_path($user->img_url))) {
                unlink(public_path($user->img_url));
            }

            // Upload new photo
            $file = $request->file('profile_photo');
            $fileName = 'profile_' . $userId . '_' . time() . '.' . $file->getClientOriginalExtension();
            $filePath = 'uploads/profiles/' . $fileName;
            
            // Create directory if it doesn't exist
            $uploadDir = public_path('uploads/profiles');
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            // Move uploaded file
            $file->move($uploadDir, $fileName);

            // Update user record
            $user->update([
                'img_url' => $filePath,
                'updated_at' => now(),
                'updated_by' => $userId
            ]);

            return response()->json([
                'success' => true,
                'data' => [
                    'img_url' => $filePath
                ],
                'message' => 'Profile photo uploaded successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Profile photo upload error: ' . $e->getMessage());
            return $this->error('Error uploading profile photo', 500);
        }
    }



    /**
     * Get current user for permission checks
     */
    protected function getCurrentUser()
    {
        $tokenModel = PublicHelper::DecodeJWT(request());
        $userId = $tokenModel['userId'] ?? null;

        if (!$userId) {
            return null;
        }

        return User::find($userId);
    }
} 