<?php

namespace App\Traits;

use App\Models\User;
use App\Helpers\PublicHelper;

trait HasPermissions
{
    /**
     * Get current user from JWT token
     */
    protected function getCurrentUser()
    {
        // First try to get user from request attributes (set by middleware)
        $userData = request()->attributes->get('user');
        if ($userData && isset($userData['userId'])) {
            return User::find($userData['userId']);
        }

        // Fallback to decoding JWT token directly
        $token = request()->cookie(config('jwt.public_key'));
        if (!$token) {
            return null;
        }

        $tokenModel = PublicHelper::DecodeJWT(request());
        if (!$tokenModel) {
            return null;
        }

        $userId = $tokenModel['userId'] ?? null;
        if (!$userId) {
            return null;
        }

        return User::find($userId);
    }

    /**
     * Check if current user has direct permission for a specific action
     */
    protected function userHasDirectPermission($permission, $action = 'access')
    {
        $user = $this->getCurrentUser();
        if (!$user) {
            return false;
        }

        return $user->hasDirectPermission($permission, $action);
    }

    /**
     * Check if current user has any direct permission for a specific permission
     */
    protected function userHasAnyDirectPermission($permission)
    {
        $user = $this->getCurrentUser();
        if (!$user) {
            return false;
        }

        return $user->hasAnyDirectPermission($permission);
    }

    /**
     * Check if current user can access a module
     */
    protected function userCanAccess($permission)
    {
        return $this->userHasDirectPermission($permission, 'access');
    }

    /**
     * Check if current user can view data for a module
     */
    protected function userCanView($permission)
    {
        return $this->userHasDirectPermission($permission, 'view');
    }

    /**
     * Check if current user can edit data for a module
     */
    protected function userCanEdit($permission)
    {
        return $this->userHasDirectPermission($permission, 'edit');
    }

    /**
     * Check if current user can delete data for a module
     */
    protected function userCanDelete($permission)
    {
        return $this->userHasDirectPermission($permission, 'delete');
    }

    /**
     * Get current user's direct permissions
     */
    protected function getUserDirectPermissions()
    {
        $user = $this->getCurrentUser();
        if (!$user) {
            return [];
        }

        return $user->getDirectPermissionsArray();
    }

    /**
     * Check permission and return error response if not authorized
     */
    protected function requirePermission($permission, $action = 'access')
    {
        if (!$this->userHasDirectPermission($permission, $action)) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient permissions'
            ], 403);
        }

        return null;
    }

    /**
     * Check access permission and return error response if not authorized
     */
    protected function requireAccess($permission)
    {
        return $this->requirePermission($permission, 'access');
    }

    /**
     * Check view permission and return error response if not authorized
     */
    protected function requireView($permission)
    {
        return $this->requirePermission($permission, 'view');
    }

    /**
     * Check edit permission and return error response if not authorized
     */
    protected function requireEdit($permission)
    {
        return $this->requirePermission($permission, 'edit');
    }

    /**
     * Check delete permission and return error response if not authorized
     */
    protected function requireDelete($permission)
    {
        return $this->requirePermission($permission, 'delete');
    }
} 