<?php
/**
 * Step 5: Finish Installation
 */

$error = '';
$installLog = [];
$installComplete = false;

// Debug info
$debugInfo = [
    'POST Method' => $_SERVER['REQUEST_METHOD'] === 'POST' ? 'Yes' : 'No',
    'run_installation set' => isset($_POST['run_installation']) ? 'Yes' : 'No',
    'Step 2 completed' => !empty($_SESSION['install']['completed_steps'][2]) ? 'Yes' : 'No',
    'Step 3 completed' => !empty($_SESSION['install']['completed_steps'][3]) ? 'Yes' : 'No',
    'Step 4 completed' => !empty($_SESSION['install']['completed_steps'][4]) ? 'Yes' : 'No',
    'DB Config exists' => !empty($_SESSION['install']['db_config']) ? 'Yes' : 'No',
    'Settings exists' => !empty($_SESSION['install']['settings']) ? 'Yes' : 'No',
];

// Check if installation should run
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['run_installation'])) {
    
    // Disable time limit
    set_time_limit(300);
    
    $installLog[] = "Starting installation...";
    
    try {
        // Get all configuration from session
        $dbConfig = $_SESSION['install']['db_config'] ?? [];
        $settings = $_SESSION['install']['settings'] ?? [];
        
        if (empty($dbConfig)) {
            throw new Exception("Missing database configuration. Please go back and complete all steps.");
        }
        
        // Step 1: Create .env file
        $installLog[] = "Creating .env file...";
        $envContent = file_get_contents('../../env.example.txt');
        
        // Replace placeholders
        $envContent = str_replace('APP_NAME="Fleet Tracker Pro"', 'APP_NAME="' . ($settings['app_name'] ?? 'Fleet Tracker Pro') . '"', $envContent);
        $envContent = str_replace('APP_URL=http://localhost', 'APP_URL=' . ($settings['app_url'] ?? 'http://localhost'), $envContent);
        $envContent = str_replace('APP_ENV=production', 'APP_ENV=local', $envContent);
        $envContent = str_replace('APP_DEBUG=false', 'APP_DEBUG=true', $envContent);
        $envContent = str_replace('APP_TIMEZONE=UTC', 'APP_TIMEZONE=' . ($settings['timezone'] ?? 'UTC'), $envContent);
        $envContent = str_replace('APP_LOCALE=en', 'APP_LOCALE=' . ($settings['locale'] ?? 'en'), $envContent);
        
        $envContent = str_replace('SESSION_DRIVER=database', 'SESSION_DRIVER=database', $envContent);
        $envContent = str_replace('SESSION_TABLE=vts_sessions', 'SESSION_TABLE=vts_sessions', $envContent);
        
        $envContent = str_replace('DB_HOST=127.0.0.1', 'DB_HOST=' . $dbConfig['host'], $envContent);
        $envContent = str_replace('DB_PORT=3306', 'DB_PORT=' . $dbConfig['port'], $envContent);
        $envContent = str_replace('DB_DATABASE=fleet_tracker', 'DB_DATABASE=' . $dbConfig['database'], $envContent);
        $envContent = str_replace('DB_USERNAME=root', 'DB_USERNAME=' . $dbConfig['username'], $envContent);
        $envContent = str_replace('DB_PASSWORD=', 'DB_PASSWORD=' . $dbConfig['password'], $envContent);
        
        $envContent = str_replace('INSTALLED=false', 'INSTALLED=true', $envContent);
        
        file_put_contents('../../.env', $envContent);
        $installLog[] = "✓ .env file created";
        
        // Step 2: Generate APP_KEY
        $installLog[] = "Generating application key...";
        $appKey = 'base64:' . base64_encode(random_bytes(32));
        
        $envContent = file_get_contents('../../.env');
        $envContent = preg_replace('/APP_KEY=.*/', 'APP_KEY=' . $appKey, $envContent);
        file_put_contents('../../.env', $envContent);
        $installLog[] = "✓ APP_KEY generated";
        
        // Step 3: Generate JWT Secret
        $installLog[] = "Generating JWT secret...";
        $jwtSecret = base64_encode(random_bytes(64));
        
        $envContent = file_get_contents('../../.env');
        $envContent = preg_replace('/JWT_SECRET=.*/', 'JWT_SECRET=' . $jwtSecret, $envContent);
        file_put_contents('../../.env', $envContent);
        $installLog[] = "✓ JWT_SECRET generated";
        
        // Step 4: Connect to database and run migrations
        $installLog[] = "Connecting to database...";
        $dsn = "mysql:host={$dbConfig['host']};port={$dbConfig['port']};dbname={$dbConfig['database']}";
        $pdo = new PDO($dsn, $dbConfig['username'], $dbConfig['password']);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $installLog[] = "✓ Database connected";
        
        // Step 5: Import database structure
        $installLog[] = "Importing database structure...";
        $sqlPath = __DIR__ . '/../../../vtc_database_table.sql';
        
        if (file_exists($sqlPath)) {
            $sql = file_get_contents($sqlPath);
            $sql = preg_replace('/^--.*$/m', '', $sql);
            $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);
            
            $statements = array_filter(explode(';', $sql), function($stmt) {
                $stmt = trim($stmt);
                return !empty($stmt) && stripos($stmt, 'SET') !== 0;
            });
            
            foreach ($statements as $statement) {
                $statement = trim($statement);
                if (!empty($statement)) {
                    try {
                        $pdo->exec($statement);
                    } catch (PDOException $e) {
                        // Continue on error (table might exist)
                    }
                }
        }
        $installLog[] = "✓ Database structure imported";
        }
        
        // Step 6: Import demo data (if selected)
        if ($settings['install_demo'] ?? false) {
            $installLog[] = "Importing demo data...";
            $demoSqlPath = __DIR__ . '/../../../vtc_database_all.sql';
            
            if (file_exists($demoSqlPath)) {
                $sql = file_get_contents($demoSqlPath);
                
                // Remove MySQL control statements
                $sql = preg_replace('/^--.*$/m', '', $sql);
                $sql = preg_replace('/\/\*!.*?\*\//s', '', $sql);
                
                $statements = explode(';', $sql);
                $imported = 0;
                $errors = 0;
                
                foreach ($statements as $statement) {
                    $statement = trim($statement);
                    
                    // Skip empty and control statements
                    if (empty($statement) || 
                        stripos($statement, 'SET ') === 0 ||
                        stripos($statement, 'CREATE TABLE') === 0 ||
                        stripos($statement, 'ALTER TABLE') === 0) {
                        continue;
                    }
                    
                    // Import INSERT statements
                    if (stripos($statement, 'INSERT INTO') === 0) {
                        try {
                            $pdo->exec($statement);
                            $imported++;
                        } catch (PDOException $e) {
                            $errors++;
                            // Log specific errors for debugging
                            if ($errors <= 3) { // Only log first 3 errors
                                $installLog[] = "⚠ Import error: " . substr($e->getMessage(), 0, 100);
                            }
                        }
                    }
                }
                $installLog[] = "✓ Demo data imported ({$imported} records" . ($errors > 0 ? ", {$errors} skipped)" : ")");
            } else {
                $installLog[] = "⚠ Demo data file not found at: {$demoSqlPath}";
            }
        } else {
            $installLog[] = "⊘ Demo data installation skipped (not selected)";
        }
        
        // Step 8: Create installed lock file
        $installLog[] = "Finalizing installation...";
        file_put_contents('../../storage/installed', date('Y-m-d H:i:s'));
        $installLog[] = "✓ Installation lock created";
        
        // Step 9: Clear caches
        if (file_exists('../../bootstrap/cache/config.php')) {
            unlink('../../bootstrap/cache/config.php');
        }
        if (file_exists('../../bootstrap/cache/routes.php')) {
            unlink('../../bootstrap/cache/routes.php');
        }
        $installLog[] = "✓ Caches cleared";
        
        $installComplete = true;
        $_SESSION['install']['completed'] = true;
        
    } catch (Exception $e) {
        $error = $e->getMessage();
        $installLog[] = "✗ Error: " . $e->getMessage();
    }
}

// Check if ready to install
$canInstall = !empty($_SESSION['install']['completed_steps'][2]) && 
              !empty($_SESSION['install']['completed_steps'][3]);
?>

<h2 class="card-title mb-4">
    <i class="ti ti-rocket me-2"></i> Ready to Install!
</h2>

<?php if ($installComplete): ?>
    
    <div class="alert alert-success">
        <h3 class="alert-title">
            <i class="ti ti-confetti me-2"></i> Installation Completed Successfully!
        </h3>
        <div class="text-muted">
            Your Fleet Tracker Pro installation is ready to use.
        </div>
    </div>

    <div class="card bg-light">
        <div class="card-body">
            <h4 class="mb-3">Installation Log:</h4>
            <div style="max-height: 300px; overflow-y: auto; font-family: monospace; font-size: 0.875rem;">
                <?php foreach ($installLog as $log): ?>
                    <div><?= htmlspecialchars($log) ?></div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>

    <div class="card bg-primary text-white mt-3">
        <div class="card-body">
            <h4>🔐 Demo Login Credentials:</h4>
            <table class="table table-transparent text-white">
                <tr>
                    <td><strong>Admin Email:</strong></td>
                    <td><code class="text-white">admin@mail.com</code></td>
                </tr>
                <tr>
                    <td><strong>Password:</strong></td>
                    <td><code class="text-white">123456</code></td>
                </tr>
                <tr>
                    <td colspan="2" class="pt-3 border-top border-white-50">
                        <strong>Demo User Email:</strong> <code class="text-white">burak@mail.com</code><br>
                        <strong>Password:</strong> <code class="text-white">123456</code>
                    </td>
                </tr>
            </table>
            <p class="mb-0 small">
                <i class="ti ti-alert-triangle me-1"></i>
                Change all passwords immediately after first login!
            </p>
        </div>
    </div>

    <div class="alert alert-warning mt-3">
        <h4 class="alert-title">
            <i class="ti ti-shield-lock me-2"></i> Important Security Steps
        </h4>
        <ol class="mb-0">
            <li>Delete the <code>/install</code> folder immediately for security</li>
            <li>Change your admin password after first login</li>
            <li>Set <code>APP_DEBUG=false</code> in .env for production</li>
            <li>Configure SSL certificate for HTTPS</li>
        </ol>
    </div>

    <div class="d-grid mt-4">
        <a href="../../" class="btn btn-lg btn-success">
            <i class="ti ti-login me-2"></i> Go to Application
        </a>
    </div>

<?php else: ?>

    <?php if ($error): ?>
        <div class="alert alert-danger alert-dismissible">
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            <h4 class="alert-title"><i class="ti ti-alert-circle"></i> Installation Error</h4>
            <div class="text-muted"><?= htmlspecialchars($error) ?></div>
            
            <?php if (!empty($installLog)): ?>
                <div class="mt-3">
                    <strong>Installation Log:</strong>
                    <div class="mt-2" style="max-height: 200px; overflow-y: auto; font-family: monospace; font-size: 0.875rem;">
                        <?php foreach ($installLog as $log): ?>
                            <div><?= htmlspecialchars($log) ?></div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <p class="text-muted mb-4">
        Review your configuration and click "Install Now" to complete the installation.
    </p>

    <!-- Debug Info (remove after testing) -->
    <div class="card bg-info text-white mb-3">
        <div class="card-body">
            <h5>🔍 Debug Information:</h5>
            <small>
                <?php foreach ($debugInfo as $key => $value): ?>
                    <div><strong><?= $key ?>:</strong> <?= $value ?></div>
                <?php endforeach; ?>
            </small>
        </div>
    </div>

    <div class="card bg-light mb-3">
        <div class="card-body">
            <h4 class="card-title">Installation Summary</h4>
            <table class="table table-borderless">
                <tr>
                    <td><strong>Database:</strong></td>
                    <td><?= htmlspecialchars($_SESSION['install']['db_config']['database'] ?? 'N/A') ?></td>
                </tr>
                <tr>
                    <td><strong>App Name:</strong></td>
                    <td><?= htmlspecialchars($_SESSION['install']['settings']['app_name'] ?? 'Fleet Tracker Pro') ?></td>
                </tr>
                <tr>
                    <td><strong>Demo Data:</strong></td>
                    <td><?= ($settings['install_demo'] ?? true) ? 'Yes' : 'No' ?></td>
                </tr>
            </table>
        </div>
    </div>

    <?php if (!$canInstall): ?>
        <div class="alert alert-warning">
            <i class="ti ti-alert-triangle me-2"></i>
            Please complete all previous steps before installing.
        </div>
    <?php endif; ?>

    <form method="POST" id="installForm">
        <div class="d-flex justify-content-between mt-4">
            <a href="?step=4" class="btn">
                <i class="ti ti-arrow-left me-2"></i> Back
            </a>
            <button type="submit" name="run_installation" value="1" class="btn btn-success btn-lg" 
                    <?= !$canInstall ? 'disabled' : '' ?>>
                <i class="ti ti-rocket me-2"></i> Install Now
            </button>
        </div>
    </form>

    <div id="progress" class="mt-4" style="display: none;">
        <div class="progress">
            <div class="progress-bar progress-bar-striped progress-bar-animated" 
                 style="width: 100%">
                Installing...
            </div>
        </div>
    </div>

    <script>
    document.getElementById('installForm').addEventListener('submit', function(e) {
        // Show progress bar
        document.getElementById('progress').style.display = 'block';
        
        // Disable button AFTER form submits (use setTimeout)
        setTimeout(function() {
            document.querySelector('[name="run_installation"]').disabled = true;
            document.querySelector('[name="run_installation"]').innerHTML = '<i class="ti ti-loader me-2"></i> Installing...';
        }, 100);
    });
    </script>

<?php endif; ?>

