@php
use App\Helpers\Base64Helper;

$get_data = Base64Helper::decode($data ?? '');
$did = $get_data != null ? $get_data['did'] : '';
$date = $get_data != null ? $get_data['date'] : date('Y-m-d');
$start_date = $get_data != null ? $get_data['date'] : date('Y-m-d', strtotime('-1 day'));
$end_date = $get_data != null ? $get_data['date'] : date('Y-m-d');
@endphp

@extends('layouts.app')

@section('title', __('Zone Violations Report') )

@section('content')
<div class="page-wrapper">
    <!-- Page header -->
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted">{{ __('Reports') }}</a> / {{ __('Operational Reports') }}
                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-map-pin me-2"></i>
                        {{ __('Zone Violations Report') }}
                    </h2>
                </div>
                <div class="col-auto ms-auto d-print-none">
                    <div class="btn-list">
                        <button class="btn btn-primary" onclick="printReport()">
                            <i class="ti ti-printer me-2"></i>
                            {{ __('Print') }}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Page body -->
    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">{{ __('Vehicle Selection') }}</label>
                            <div class="input-group">
                            <select class="form-select" id="vehicleSelect">
                                    <option value="">{{ __('Select Vehicle') }}</option>
                            </select>
                                <button type="button" class="btn btn-outline-secondary d-none" id="vehicleSettingsLink"
                                    onclick="goToVehicleSettings()" title="{{ __('Go to Vehicle Settings') }}">
                                    <i class="icon ti ti-settings"></i>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">{{ __('Start Date') }}</label>
                            <input type="date" class="form-control" id="startDate" value="{{ $start_date }}">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">{{ __('End Date') }}</label>
                            <input type="date" class="form-control" id="endDate" value="{{ $end_date }}">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">{{ __('Event Type') }}</label>
                            <select class="form-select" id="eventType">
                                <option value="all">{{ __('All Events') }}</option>
                                <option value="entry">{{ __('Entry Only') }}</option>
                                <option value="exit">{{ __('Exit Only') }}</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">&nbsp;</label>
                            <button class="btn btn-primary w-100" onclick="loadRegionData()">
                                <i class="ti ti-search me-2"></i>
                                {{ __('Update Report') }}
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="row row-cards mb-3" id="summaryCards">
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">{{ __('Total Violations') }}</div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-alert-triangle ms-1 text-danger" style="font-size: 1.5rem;"></i>
                            </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="totalViolations">-</div>
                            </div>
                            <div id="apex-total-violations" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted" id="violationBreakdown">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">{{ __('Entry Events') }}</div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-login ms-1 text-success" style="font-size: 1.5rem;"></i>
                            </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="entryEvents">-</div>
                            </div>
                            <div id="apex-entry-events" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted" id="entryTrend">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">{{ __('Exit Events') }}</div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-logout ms-1 text-warning" style="font-size: 1.5rem;"></i>
                            </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="exitEvents">-</div>
                            </div>
                            <div id="apex-exit-events" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted" id="exitTrend">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader">{{ __('Avg. Duration') }}</div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-clock ms-1 text-info" style="font-size: 1.5rem;"></i>
                            </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="avgDuration">-</div>
                            </div>
                            <div id="apex-avg-duration" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted" id="durationTrend">-</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Region Map -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title">{{ __('Region Violations') }}</h3>
                    <div class="card-actions">
                        <div class="btn-list">
                            <button class="btn btn-sm" onclick="toggleRegionLayers()">
                                <i class="ti ti-layers"></i> {{ __('Layers') }}
                            </button>
                            <button class="btn btn-sm" onclick="toggleMapView()">
                                <i class="ti ti-maximize"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div id="regionMap" style="height: 500px;"></div>
                </div>
            </div>

            <!-- Charts Row -->
            <div class="row mb-3">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Violation Trends') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="violationTrendChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Most Violated Regions') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="regionChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Violation Details Table -->
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">{{ __('Violation Details') }}</h3>
                    <div class="card-actions">
                        <div class="btn-list">
                            <button class="btn btn-sm" onclick="table.download('csv', 'zone-violations.csv')">
                                <i class="ti ti-download"></i> {{ __('CSV') }}
                            </button>
                            <button class="btn btn-sm" onclick="toggleColumns()">
                                <i class="ti ti-columns"></i> {{ __('Columns') }}
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div id="violation-table"></div>
                </div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-map-pin text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title">{{ __('No zone violations found') }}</p>
                    <p class="empty-subtitle text-muted">
                        {{ __('No zone violations were detected for the selected period and criteria.') }}
                    </p>
                </div>
            </div>

            <!-- No Vehicle Selected State -->
            <div class="text-center py-4 d-none" id="noVehicleState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-car text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title">{{ __('No Vehicle Selected') }}</p>
                    <p class="empty-subtitle text-muted">
                        {{ __('Please select a vehicle from the dropdown above to generate the zone violations analysis report.') }}
                    </p>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">{{ __('Loading...') }}</span>
                </div>
                <div class="mt-2">{{ __('Generating report...') }}</div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('styles')
<style>
    #regionMap {
        height: 500px;
    }

    .violation-marker .map-icon-label {
        background-color: transparent !important;
        font-size: 15px !important;
        font-weight: 500 !important;
        color: #fff !important;
        padding: 0 !important;
        margin: 0 !important;
        border-radius: 0 !important;
        border: 0 !important;
    }

    .tabulator-cell[tabulator-field="duration_minutes"] {
        font-weight: 600;
    }

    .event-badge {
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 11px;
        font-weight: 600;
    }

    .event-entry {
        background-color: #10b981;
        color: white;
    }

    .event-exit {
        background-color: #ef4444;
        color: white;
    }

    .location-region {
        background-color: #8b5cf6;
        color: white;
    }

    .temp-location-marker {
        background: transparent !important;
        border: none !important;
    }

    .temp-location-marker i {
        text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.5);
        animation: pulse 1s infinite;
    }

    @keyframes pulse {
        0% {
            transform: scale(1);
        }

        50% {
            transform: scale(1.2);
        }

        100% {
            transform: scale(1);
        }
    }
</style>
@endsection

@section('scripts')
<script>
    // --- Unit Variables ---
    var speedUnit = "";
    var lengthUnit = "";
    var currency = "";
    
    let table;
    let myMap;

    $('#summaryCards').addClass('d-none');
    $('#regionMap').closest('.card').addClass('d-none');
    $('.row.mb-3').addClass('d-none'); // Charts row
    $('#violation-table').closest('.card').addClass('d-none');
    $('#loadingState').addClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#noVehicleState').removeClass('d-none');

    onAppReady(function() { 
        loadVehicles()
        .then(initializeMap)
        .then(initializeTable)
        .then(loadRegionData);
    });

    // Initialize map
    async function initializeMap() {
        myMap = new LeafletMap();
        myMap.initMap("regionMap", {
            
        });
    }
    
    // Refresh map function
    function refreshMap() {
        if (myMap && myMap.mapLet) {
            myMap.mapLet.invalidateSize();
        }
    }

    // Initialize table
    async function initializeTable() {
        table = createTabulator("#violation-table", {
            initialSort: [
                {column: "event_time", dir: "desc"}
            ],
            columns: [
                {title: "#", field: "id", hozAlign: "center", width: 50},
                {
                    title: "{{ __('Vehicle') }}", 
                    field: "vehicle_name"
                },
                {
                    title: "{{ __('Region') }}", 
                    field: "region_name"
                },
                {
                    title: "{{ __('Region Status') }}", 
                    field: "region_control", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const status = cell.getValue();
                        if (!status) return '-';
                        const badgeClass = status === 'no_entry' ? 'bg-red' : 'bg-orange';
                        const text = status === 'no_entry' ? '{{ __("no_entry") }}' : '{{ __("no_exit") }}';
                        return `<span class="badge text-white ${badgeClass}">${text}</span>`;
                    }
                },
                {
                    title: "{{ __('Price') }}", 
                    field: "region_price", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const price = cell.getValue();
                        if (!price) return '-';
                        return `${price} ${currency}`;
                    }
                },
                {
                    title: "{{ __('Event') }}", 
                    field: "event_type", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const event = cell.getValue();
                        const badgeClass = event === 'entry' ? 'event-entry' : 'event-exit';
                        const text = event === 'entry' ? '{{ __("Entry") }}' : '{{ __("Exit") }}';
                        return `<span class="event-badge ${badgeClass}">${text}</span>`;
                    }
                },
                {
                    title: "{{ __('Event Time') }}", 
                    field: "event_time",
                    sorter: "string",
                    formatter: function(cell) {
                        const value = cell.getValue();
                        if (!value) return '-';
                        // --- Use single argument for dateTimeFormater ---
                        return dateTimeFormatter(value);
                    }
                },
                {
                    title: "{{ __('Visit Duration') }}", 
                    field: "duration_minutes", 
                    hozAlign: "center",
                    headerTooltip: "{{ __('Time spent inside the region. Shows duration for exit events only.') }}",
                    formatter: function(cell) {
                        const minutes = cell.getValue();
                        if (!minutes || minutes === 0) return '-';
                        return formatDuration(minutes);
                    }
                },
                {
                    title: "{{ __('Status') }}", 
                    field: "status", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const status = cell.getValue();
                        if (status === 'active') {
                            return `<span class="badge text-white bg-yellow">{{ __("Active") }}</span>`;
                        } else {
                            return `<span class="badge text-white bg-green">{{ __("Completed") }}</span>`;
                        }
                    }
                },
                {
                    title: "{{ __('Actions') }}", 
                    field: "actions", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const data = cell.getData();
                        let buttons = '';
                        
                        // Show on map button
                        if (data.lat && data.lng) {
                            buttons += `<button class="btn btn-sm btn-primary me-1" onclick="showLocationOnMap(${data.lat}, ${data.lng})" title="{{ __('Show on map') }}">
                                <i class="ti ti-map-pin"></i>
                                <span class="d-none d-md-inline">{{ __("Map") }}</span>
                            </button>`;
                        } else {
                            buttons += `<button class="btn btn-sm btn-secondary me-1" disabled title="{{ __('No coordinates available') }}">
                                <i class="ti ti-map-pin"></i>
                                <span class="d-none d-md-inline">{{ __("Map") }}</span>
                            </button>`;
                        }
                        
                        // Vehicle history button (if we have vehicle_id and event_time)
                        if (data.vehicle_id && data.event_time) {
                            const eventDate = new Date(data.event_time).toISOString().split('T')[0];
                            const urlEncoded = btoa(JSON.stringify({
                                did: data.vehicle_id,
                                date: eventDate,
                                start_time: "00:01",
                                end_time: "23:59"
                            }));
                            buttons += `<a href="/vehicle-map/${urlEncoded}" class="btn btn-sm btn-info me-1" title="{{ __('Vehicle History') }}">
                                <i class="ti ti-history"></i>
                                <span class="d-none d-md-inline">{{ __("History") }}</span>
                            </a>`;
                        }
                        
                        return buttons;
                    }
                }
            ]
        });
    }

    // Load vehicles
    async function loadVehicles() {
        http.post({
            url: '/api/vehicles-list-with-region',
            data: {},
            loading: false
        }).then(function(response) {
            if (response.success && response.data.vehicles) {

                const select = $('#vehicleSelect');
                select.empty();
                select.append('<option value="">{{ __("Please select a vehicle") }}</option>');
                
                // Separate vehicles by region count
                const vehiclesWithRegions = [];
                const vehiclesWithoutRegions = [];
                
                response.data.vehicles.forEach(vehicle => {
                    const regionCount = parseInt(vehicle.count_region) || 0;
                    if (regionCount > 0) {
                        vehiclesWithRegions.push({...vehicle, regionCount});
                    } else {
                        vehiclesWithoutRegions.push({...vehicle, regionCount});
                    }
                });
                
                // Add vehicles with regions first
                if (vehiclesWithRegions.length > 0) {
                    select.append('<optgroup label="{{ __("Vehicles with Regions") }}">');
                    vehiclesWithRegions.forEach(vehicle => {
                        const regionText = vehicle.regionCount === 1 ? '{{ __("region") }}' : '{{ __("regions") }}';
                        select.append(`<option value="${vehicle.device_did}">${vehicle.device_name} (${vehicle.regionCount} ${regionText})</option>`);
                    });
                    select.append('</optgroup>');
                }
                
                // Add vehicles without regions
                if (vehiclesWithoutRegions.length > 0) {
                    select.append('<optgroup label="{{ __("Vehicles without Regions") }}">');
                    vehiclesWithoutRegions.forEach(vehicle => {
                        select.append(`<option value="${vehicle.device_did}">${vehicle.device_name}</option>`);
                    });
                    select.append('</optgroup>');
                }
                
                // Add settings link if there are vehicles without regions
                if (vehiclesWithoutRegions.length > 0) {
                    $('#vehicleSettingsLink').removeClass('d-none');
                } else {
                    $('#vehicleSettingsLink').addClass('d-none');
                }
                
                // Show warning if no vehicles have regions
                if (vehiclesWithRegions.length === 0) {
                    showConfirmDialog({
                        title: '{{ __("Warning") }}',
                        message: '{{ __("No vehicles have regions configured. Please configure regions for at least one vehicle to use this report.") }}',
                        confirmText: '{{ __("Go to Settings") }}',
                        cancelText: '{{ __("Cancel") }}',
                        type: 'warning',
                        onConfirm: function() {
                            window.location.href = '/vehicles-list';
                        },
                        onCancel: function() {}
                    });
                }
            }
        }).catch(function(error) {
            console.error('{{ __("Error loading vehicles") }}:', error);
        });
    }

    // Load region data
    async function loadRegionData() {
        const vehicleId = document.getElementById('vehicleSelect').value;
        const startDate = document.getElementById('startDate').value;
        const endDate = document.getElementById('endDate').value;
        const eventType = document.getElementById('eventType').value;

        // Check if vehicle is selected
        if (!vehicleId) {
            showConfirmDialog({
                title: '{{ __("Warning") }}',
                message: '{{ __("Please select a vehicle") }}',
                confirmText: '{{ __("OK") }}',
                showCancelButton: false,
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Check if selected vehicle has regions configured
        const selectedOption = $('#vehicleSelect option:selected');
        const selectedText = selectedOption.text();
        if (selectedText.includes('(0 regions)')) {
            const vehicleName = selectedText.replace(' (0 regions)', '');
            showConfirmDialog({
                title: '{{ __("Error") }}',
                message: `{{ __("No regions are configured for") }} "${vehicleName}". {{ __("Please configure regions in vehicle settings to use this report.") }}`,
                confirmText: '{{ __("Go to Settings") }}',
                cancelText: '{{ __("Cancel") }}',
                type: 'danger',
                onConfirm: function() {
                    window.location.href = '/vehicles-list';
                },
                onCancel: function() {}
            });
            return;
        }

        http.post({
            url: '/api/reports/region-data',
            data: {
                vehicle: vehicleId,
                start_date: startDate,
                end_date: endDate,
                location_type: 'regions'
            },
            loading: true
        }).then(function(response) {
    
            // --- Set units after data is loaded ---
                    speedUnit = sessionSettings?.speedUnit || 'km/h';
        lengthUnit = sessionSettings?.lengthUnit || 'km';
            currency = __(sessionSettings?.currency || 'TRY');
            // ---
            if (response.success) {
                const rawData = response.data.raw_data || [];
                const regionsList = response.data.regions_list || [];
                const queryInfo = response.data.query_info || {};
                
                var message = __('#name# used #value# records').replace('#name#', __('Zone Violations Report')).replace('#value#', rawData.length);
                showReportDataProcessingInfo(message);
                
                // Process raw data in JavaScript
                processRawRegionData(rawData, regionsList, eventType);
            } else {
                showConfirmDialog({
                    title: '{{ __("Error") }}',
                    message: '{{ __("Data not found") }}',
                    confirmText: '{{ __("OK") }}',
                    cancelText: '',
                    type: 'danger',
                    onConfirm: function() {},
                    onCancel: function() {}
                });
            }
        }).catch(function(error) {
            showConfirmDialog({
                title: '{{ __("Error") }}',
                message: '{{ __("Error fetching data") }}',
                confirmText: '{{ __("OK") }}',
                cancelText: '',
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
        });
    }

    // Process raw data
    function processRawRegionData(rawData, regionsList, eventType) {
        // Refresh map to ensure proper display when container becomes visible
        if (myMap && myMap.mapLet) {
            myMap.mapLet.invalidateSize();
        }
        
        // Clear existing map elements
        myMap.removeMarkers();
        myMap.clearAllPolygons();
        myMap.removeCircles();
        
        // 1. Add regions to map
        if (regionsList && regionsList.length > 0) {
            regionsList.forEach(region => {
                if (region.polygon_json) {
                    try {
                        const polygon = JSON.parse(region.polygon_json);
                        myMap.addPolygon({
                            id: `region-${region.id}`,
                            coords: polygon,
                            color: "purple",
                            fillColor: "purple",
                            fillOpacity: 0.2,
                            label: region.name
                        });
                    } catch (e) {}
                }
            });
        }
        
        // 2. Process region events
        const events = [];
        const currentVisits = {};
        let eventId = 1;
        
        if (rawData && rawData.length > 0) {
            rawData.forEach(record => {
                if (record.region_control !== '-') {
                    const vehicleId = record.did;
                    const visitKey = `region_${vehicleId}_${record.region_id}`;
                    
                    if (record.region_control === 'entry' && !currentVisits[visitKey]) {
                        // Entry event
                        const event = {
                            id: eventId++,
                            vehicle_id: vehicleId,
                            vehicle_name: record.device_name,
                            region_id: record.region_id,
                            region_name: record.region_name,
                            event_type: 'entry',
                            event_time: record.event_time,
                            lat: parseFloat(record.latitude),
                            lng: parseFloat(record.longitude),
                            status: 'active',
                            region_control: record.region_status, // 'no_entry' or 'no_exit'
                            violation_type: record.region_control_int,
                            region_price: record.region_price,
                            mobile_notification: record.mobile_notification,
                            log_id: record.log_id
                        };
                        events.push(event);
                        currentVisits[visitKey] = {
                            start_time: record.event_time,
                            event_id: event.id
                        };
                    } else if (record.region_control === 'exit' && currentVisits[visitKey]) {
                        // Exit event
                        const visit = currentVisits[visitKey];
                        const duration = calculateDurationMinutes(visit.start_time, record.event_time);
                        const event = {
                            id: eventId++,
                            vehicle_id: vehicleId,
                            vehicle_name: record.device_name,
                            region_id: record.region_id,
                            region_name: record.region_name,
                            event_type: 'exit',
                            event_time: record.event_time,
                            lat: parseFloat(record.latitude),
                            lng: parseFloat(record.longitude),
                            duration_minutes: duration,
                            status: 'completed',
                            region_control: record.region_status,
                            violation_type: record.region_control_int,
                            region_price: record.region_price,
                            mobile_notification: record.mobile_notification,
                            log_id: record.log_id
                        };
                        events.push(event);
                        delete currentVisits[visitKey];
                    }
                }
            });
        }
        
        // Filter events by event type
        const filteredEvents = filterEventsByType(events, eventType);
        
        // Add markers to map for filtered events only
        addEventMarkersToMap(filteredEvents);
        
        // 3. Update table with events
        updateTable(filteredEvents);
        
        // 4. Update charts
        updateCharts(filteredEvents);
        
        // 5. Update summary statistics
        updateSummaryFromEvents(filteredEvents);
        
        // 6. Fit map to show all markers
        if (filteredEvents.length > 0) {
            setTimeout(() => {
                myMap.fitBounds();
            }, 200);
        }
        
        // Show message if no events found
        if (filteredEvents.length === 0) {
            showNoDataState();
        } else {
            hideAllStates();
            // Refresh map after showing all states to ensure proper display
            setTimeout(() => {
                if (myMap && myMap.mapLet) {
                    myMap.mapLet.invalidateSize();
                }
            }, 100);
        }
    }

    // Filter events by event type
    function filterEventsByType(events, eventType) {
        if (!eventType || eventType === 'all') {
            return events;
        }
        
        return events.filter(event => event.event_type === eventType);
    }

    // Generate daily data for sparklines
    function generateDailyData(events) {
        // Get date range from selected dates
        const startDate = new Date(document.getElementById('startDate').value);
        const endDate = new Date(document.getElementById('endDate').value);
        
        // Initialize daily arrays
        const dailyTotalViolations = [];
        const dailyEntryEvents = [];
        const dailyExitEvents = [];
        const dailyAvgDurations = [];
        const dailyDurations = [];
        const dailyDates = [];
        
        // Generate date range
        const dates = [];
        const currentDate = new Date(startDate);
        while (currentDate <= endDate) {
            dates.push(new Date(currentDate));
            currentDate.setDate(currentDate.getDate() + 1);
        }
        
        // Process each date
        dates.forEach(date => {
            const dateStr = date.toISOString().split('T')[0];
            const dayEvents = events.filter(event => {
                const eventDate = new Date(event.event_time).toISOString().split('T')[0];
                return eventDate === dateStr;
            });
            
            // Store date for tooltips
            dailyDates.push(dateStr);
            
            // Count total violations
            dailyTotalViolations.push(dayEvents.length);
            
            // Count entry events
            const dayEntryEvents = dayEvents.filter(e => e.event_type === 'entry');
            dailyEntryEvents.push(dayEntryEvents.length);
            
            // Count exit events
            const dayExitEvents = dayEvents.filter(e => e.event_type === 'exit');
            dailyExitEvents.push(dayExitEvents.length);
            
            // Calculate average duration for the day
            const dayCompletedVisits = dayExitEvents.filter(e => e.duration_minutes > 0);
            const dayAvgDuration = dayCompletedVisits.length > 0 
                ? dayCompletedVisits.reduce((sum, e) => sum + e.duration_minutes, 0) / dayCompletedVisits.length
                : 0;
            dailyAvgDurations.push(Math.round(dayAvgDuration));
        });
        
        return {
            totalViolations: dailyTotalViolations,
            entryEvents: dailyEntryEvents,
            exitEvents: dailyExitEvents,
            avgDurations: dailyAvgDurations,
            dates: dailyDates
        };
    }

    // Create ApexCharts sparkline
    function makeApexSparkline(divId, data, color, label, dates = []) {

        $("#" + divId).html('');

        var options = {
            chart: {
                type: 'line',
                height: 40,
                sparkline: { enabled: true }
            },
            series: [{
                name: label,
                data: data
            }],
            stroke: { curve: 'smooth', width: 2 },
            colors: [color],
            tooltip: {
                enabled: true,
                x: { show: true },
                y: {
                    title: {
                        formatter: function() { return label; }
                    },
                    formatter: function(value, { dataPointIndex }) {
                        let tooltipText = '';
                        
                        // Add date if available
                        if (dates && dates[dataPointIndex]) {
                            const date = new Date(dates[dataPointIndex]);
                            tooltipText += `<strong>${date.toLocaleDateString()}</strong><br>`;
                        }
                        
                        // Add value with proper formatting
                        if (label === '{{ __("Duration") }}') {
                            tooltipText += formatDuration(value);
                        } else {
                            tooltipText += value;
                        }
                        
                        return tooltipText;
                    }
                },
                custom: function({ series, seriesIndex, dataPointIndex, w }) {
                    const value = series[seriesIndex][dataPointIndex];
                    const date = dates && dates[dataPointIndex] ? new Date(dates[dataPointIndex]) : null;
                    
                    let tooltipContent = '';
                    if (date) {
                        tooltipContent += `<div style="font-weight: bold; margin-bottom: 5px;">${date.toLocaleDateString()}</div>`;
                    }
                    tooltipContent += `<div>${label}: ${label === '{{ __("Duration") }}' ? formatDuration(value) : value}</div>`;
                    
                    return tooltipContent;
                }
            }
        };
        new ApexCharts(document.querySelector("#" + divId), options).render();
    }

    // Initialize sparklines
    function initializeSparklines(totalViolations, entryEvents, exitEvents, avgDurations, dates = []) {
        makeApexSparkline('apex-total-violations', totalViolations, '#ef4444', '{{ __("Violations") }}', dates);
        makeApexSparkline('apex-entry-events', entryEvents, '#10b981', '{{ __("Entries") }}', dates);
        makeApexSparkline('apex-exit-events', exitEvents, '#f59e0b', '{{ __("Exits") }}', dates);
        makeApexSparkline('apex-avg-duration', avgDurations, '#3b82f6', '{{ __("Duration") }}', dates);
    }

    // Generate distinct colors for polylines
    function generateDistinctColors(count) {
        const colors = [];
        
        for (let i = 0; i < count; i++) {
            // Use HSL color space for better color distribution
            const hue = (i * 137.508) % 360; // Golden angle approximation for better distribution
            const saturation = 70 + (i % 20); // Vary saturation between 70-90%
            const lightness = 45 + (i % 15); // Vary lightness between 45-60%
            
            // Convert HSL to hex
            const color = hslToHex(hue, saturation, lightness);
            colors.push(color);
        }
        
        return colors;
    }
    
    // Convert HSL to hex color
    function hslToHex(h, s, l) {
        h /= 360;
        s /= 100;
        l /= 100;
        
        const c = (1 - Math.abs(2 * l - 1)) * s;
        const x = c * (1 - Math.abs((h * 6) % 2 - 1));
        const m = l - c / 2;
        
        let r, g, b;
        
        if (h < 1/6) {
            r = c; g = x; b = 0;
        } else if (h < 2/6) {
            r = x; g = c; b = 0;
        } else if (h < 3/6) {
            r = 0; g = c; b = x;
        } else if (h < 4/6) {
            r = 0; g = x; b = c;
        } else if (h < 5/6) {
            r = x; g = 0; b = c;
        } else {
            r = c; g = 0; b = x;
        }
        
        const rHex = Math.round((r + m) * 255).toString(16).padStart(2, '0');
        const gHex = Math.round((g + m) * 255).toString(16).padStart(2, '0');
        const bHex = Math.round((b + m) * 255).toString(16).padStart(2, '0');
        
        return `#${rHex}${gHex}${bHex}`;
    }
    
    // Pair entry and exit events for the same vehicle and region
    function pairEntryExitSessions(events) {
        const sessions = {};
        
        events.forEach(event => {
            const key = `${event.vehicle_id}-${event.region_id}`;
            
            if (!sessions[key]) {
                sessions[key] = {
                    vehicle_id: event.vehicle_id,
                    region_id: event.region_id,
                    vehicle_name: event.vehicle_name,
                    region_name: event.region_name,
                    entry: null,
                    exit: null
                };
            }
            
            if (event.event_type === 'entry') {
                sessions[key].entry = event;
            } else if (event.event_type === 'exit') {
                sessions[key].exit = event;
            }
        });
        
        return Object.values(sessions);
    }
    
    // Add event markers to map
    function addEventMarkersToMap(events) {
        // Group events by vehicle and region to pair entry/exit
        const violationSessions = pairEntryExitSessions(events);
        
        // Generate distinct colors for different sessions
        const groupColors = generateDistinctColors(violationSessions.length);
        
        violationSessions.forEach((session, idx) => {
            // Entry marker
            if (session.entry) {
                const markerColor = session.entry.violation_type === 1 ? 'red' : 'orange';
                const regionControlText = session.entry.region_control === 'no_entry' ? '{{ __("no_entry") }}' : '{{ __("no_exit") }}';
                const eventTime = dateTimeFormatter(session.entry.event_time);
                const popupText = `${session.entry.region_name} : ${regionControlText} {{ __("set as") }} <br> {{ __("Entry date") }}: ${eventTime}`;
                const label = session.entry.region_name +": "+ session.entry.vehicle_name;

                myMap.pinMarkers([{
                    id: `entry-${session.entry.id}`,
                    label: label,
                    coord: [session.entry.lat, session.entry.lng],
                    icon: { name: "fa-solid fa-exclamation-triangle", color: markerColor, className: "violation-marker" },
                    popup: `${session.entry.vehicle_name}<br>${popupText}`
                }]);
            }
            
            // Exit marker
            if (session.exit) {
                const markerColor = session.exit.violation_type === 1 ? 'green' : 'blue';
                const eventTime = dateTimeFormatter(session.exit.event_time);
                const regionControlText = session.exit.region_control === 'no_entry' ? '{{ __("no_entry") }}' : '{{ __("no_exit") }}';
                const popupText = `${session.exit.region_name} : ${regionControlText}  {{ __("set as") }}
                <br> {{ __("Exit date") }}: ${eventTime}`;
                const label = session.exit.region_name +": "+ session.exit.vehicle_name;
                
                myMap.pinMarkers([{
                    id: `exit-${session.exit.id}`,
                    label: label,
                    coord: [session.exit.lat, session.exit.lng],
                    icon: { name: "fa-solid fa-check-circle", color: markerColor, className: "exit-marker" },
                    popup: `${session.exit.vehicle_name}<br>${popupText}`
                }]);
            }
            
            // Add polyline between entry and exit if both exist
            if (session.entry && session.exit) {
                const entryLat = parseFloat(session.entry.lat);
                const entryLng = parseFloat(session.entry.lng);
                const exitLat = parseFloat(session.exit.lat);
                const exitLng = parseFloat(session.exit.lng);
                
                if (!isNaN(entryLat) && !isNaN(entryLng) && !isNaN(exitLat) && !isNaN(exitLng)) {
                    myMap.addPolyline({
                        coords: [[entryLat, entryLng], [exitLat, exitLng]],
                        color: groupColors[idx],
                        weight: 5,
                        opacity: 0.7,
                        smoothFactor: 1,
                    });
                }
            }
        });
    }

    // Helper function to calculate duration in minutes
    function calculateDurationMinutes(startTime, endTime) {
        const start = new Date(startTime);
        const end = new Date(endTime);
        const diffMs = end - start;
        return Math.round(diffMs / (1000 * 60));
    }

    // Update summary from processed events
    function updateSummaryFromEvents(events) {
        if (!events || events.length === 0) {
            $('#totalViolations').text('0');
            $('#entryEvents').text('0');
            $('#exitEvents').text('0');
            $('#avgDuration').text('-');
            $('#violationBreakdown').text('0 {{ __("regions") }}');
            $('#entryTrend').text('{{ __("Total entries") }}');
            $('#exitTrend').text('{{ __("Total exits") }}');
            $('#durationTrend').text('{{ __("Average per visit") }}');
            
            // Initialize empty sparklines
            initializeSparklines([], [], [], []);
            return;
        }
        
        const entryEvents = events.filter(e => e.event_type === 'entry');
        const exitEvents = events.filter(e => e.event_type === 'exit');
        const regionEvents = events.filter(e => e.region_id);
        
        const completedVisits = exitEvents.filter(e => e.duration_minutes > 0);
        const avgDuration = completedVisits.length > 0 
            ? Math.round(completedVisits.reduce((sum, e) => sum + e.duration_minutes, 0) / completedVisits.length)
            : 0;
        
        $('#totalViolations').text(events.length);
        $('#entryEvents').text(entryEvents.length);
        $('#exitEvents').text(exitEvents.length);
        $('#avgDuration').text(formatDuration(avgDuration));
        
        $('#violationBreakdown').text(`${regionEvents.length} {{ __("regions") }}`);
        $('#entryTrend').text('{{ __("Total entries") }}');
        $('#exitTrend').text('{{ __("Total exits") }}');
        $('#durationTrend').text('{{ __("Average per visit") }}');
        
        // Generate daily data for sparklines
        const dailyData = generateDailyData(events);
        initializeSparklines(dailyData.totalViolations, dailyData.entryEvents, dailyData.exitEvents, dailyData.avgDurations, dailyData.dates);
    }

    // Update charts
    function updateCharts(events) {
        // Prepare data as before...
            const violationData = events.reduce((acc, event) => {
                const date = dateTimeFormatter(event.event_time);
            if (!acc[date]) acc[date] = { entry: 0, exit: 0 };
            if (event.event_type === 'entry') acc[date].entry++;
            else acc[date].exit++;
                return acc;
            }, {});

            const dates = Object.keys(violationData).sort();
            const entryData = dates.map(date => violationData[date].entry);
            const exitData = dates.map(date => violationData[date].exit);

        // Call ApexCharts render function
        renderViolationTrendChart(dates, entryData, exitData);

        // Region chart
            const regionData = events.reduce((acc, event) => {
                acc[event.region_name] = (acc[event.region_name] || 0) + 1;
                return acc;
            }, {});
            const topRegions = Object.entries(regionData)
                .sort(([,a], [,b]) => b - a)
                .slice(0, 5);

        renderRegionChart(
            topRegions.map(([name]) => name),
            topRegions.map(([,count]) => count)
        );
    }

    // Update table
    function updateTable(violations) {
        if (violations && violations.length > 0) {
            table.setData(violations);
        } else {
            table.setData([]);
        }
    }

    // Helper functions
    function formatDuration(minutes) {
        if (minutes < 60) {
            return `${minutes} min`;
        } else {
            const hours = Math.floor(minutes / 60);
            const mins = minutes % 60;
            if (mins === 0) {
                return `${hours} hour`;
            } else {
                return `${hours} hour ${mins} min`;
            }
        }
    }

    function showLocationOnMap(lat, lng) {
        if (!lat || !lng || isNaN(lat) || isNaN(lng)) {
            showConfirmDialog({
                title: '{{ __("Warning") }}',
                message: '{{ __("Invalid coordinates") }}',
                confirmText: '{{ __("OK") }}',
                cancelText: '',
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }
        
        // Ensure map container is visible first
        $('#regionMap').closest('.card').removeClass('d-none');
        
        // Refresh map to ensure proper display
        if (myMap && myMap.mapLet) {
            myMap.mapLet.invalidateSize();
        }
        
        // Center map on the location
        myMap.mapLet.setView([parseFloat(lat), parseFloat(lng)], 16);
        
        // Add a temporary marker to highlight the location
        const tempMarker = L.marker([lat, lng], {
            icon: L.divIcon({
                className: 'temp-location-marker',
                html: '<i class="fa-solid fa-crosshairs" style="color: red; font-size: 20px;"></i>',
                iconSize: [20, 20],
                iconAnchor: [10, 10]
            })
        }).addTo(myMap.mapLet);
        
        // Remove the temporary marker after 3 seconds
        setTimeout(() => {
            myMap.mapLet.removeLayer(tempMarker);
        }, 3000);
    }

    function toggleRegionLayers() {
        showConfirmDialog({
            title: '{{ __("Info") }}',
            message: '{{ __("Layer toggle feature will be added") }}',
            confirmText: '{{ __("OK") }}',
            cancelText: '',
            type: 'info',
            onConfirm: function() {},
            onCancel: function() {}
        });
    }

    function toggleMapView() {
        document.getElementById('regionMap').requestFullscreen();
    }

    function toggleColumns() {
        showConfirmDialog({
            title: '{{ __("Info") }}',
            message: '{{ __("Column selector feature will be added") }}',
            confirmText: '{{ __("OK") }}',
            cancelText: '',
            type: 'info',
            onConfirm: function() {},
            onCancel: function() {}
        });
    }

    function printReport() {
        window.print();
    }

    // State management functions
    function showLoadingState() {
        $('#loadingState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#regionMap').closest('.card').addClass('d-none');
        $('.row.mb-3').addClass('d-none'); // Charts row
        $('#violation-table').closest('.card').addClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
    }

    function hideLoadingState() {
        $('#loadingState').addClass('d-none');
    }

    function showNoDataState() {
        $('#noDataState').removeClass('d-none');
        $('#noVehicleState').addClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#regionMap').closest('.card').addClass('d-none');
        $('.row.mb-3').addClass('d-none'); // Charts row
        $('#violation-table').closest('.card').addClass('d-none');
    }

    function showNoVehicleState() {
        $('#noVehicleState').removeClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#regionMap').closest('.card').addClass('d-none');
        $('.row.mb-3').addClass('d-none'); // Charts row
        $('#violation-table').closest('.card').addClass('d-none');
    }

    function hideAllStates() {
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
        $('#summaryCards').removeClass('d-none');
        $('#regionMap').closest('.card').removeClass('d-none');
        $('.row.mb-3').removeClass('d-none'); // Charts row
        $('#violation-table').closest('.card').removeClass('d-none');
        
        // Refresh map after showing all states to ensure proper display
        setTimeout(() => {
            if (myMap && myMap.mapLet) {
                myMap.mapLet.invalidateSize();
            }
        }, 100);
    }

    // Go to vehicle settings page
    function goToVehicleSettings() {
        window.location.href = '/vehicles-list';
    }

    function renderViolationTrendChart(labels, entryData, exitData) {
        // Clear existing chart
        $("#violationTrendChart").html('');
        
        var options = {
            chart: {
                type: 'line',
                height: 300,
                toolbar: { show: false }
            },
            series: [
                { name: '{{ __("Entry Events") }}', data: entryData },
                { name: '{{ __("Exit Events") }}', data: exitData }
            ],
            xaxis: {
                categories: labels,
                title: { text: '{{ __("Date") }}' }
            },
            yaxis: {
                title: { text: '{{ __("Events") }}' }
            },
            colors: ['#10b981', '#ef4444'],
            stroke: { curve: 'smooth', width: 2 },
            markers: { size: 4 }
        };
        new ApexCharts(document.querySelector("#violationTrendChart"), options).render();
    }

    function renderRegionChart(labels, data) {
        // Clear existing chart
        $("#regionChart").html('');
        
        var options = {
            chart: {
                type: 'donut',
                height: 300,
                toolbar: { show: false }
            },
            series: data,
            labels: labels,
            colors: [
                '#8b5cf6', '#10b981', '#f59e0b', '#ef4444', '#3b82f6'
            ],
            legend: { position: 'bottom' }
        };
        new ApexCharts(document.querySelector("#regionChart"), options).render();
    }

</script>
@endsection