<?php
use App\Helpers\Base64Helper;

$get_data = Base64Helper::decode($data ?? '');
$did = $get_data != null ? $get_data['did'] : '';
$date = $get_data != null ? $get_data['date'] : date('Y-m-d');
$start_date = $get_data != null ? $get_data['date'] : date('Y-m-d', strtotime('-7 days'));
$end_date = $get_data != null ? $get_data['date'] : date('Y-m-d');
?>



<?php $__env->startSection('title', __('Working Hours Utilization Report') ); ?>

<?php $__env->startSection('content'); ?>
<div class="page-wrapper">
    <!-- Page header -->
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted"><?php echo e(__('Reports')); ?></a> / <?php echo e(__('Operational Reports')); ?>

                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-briefcase me-2"></i>
                        <?php echo e(__('Working Hours Utilization Report')); ?>

                    </h2>
                    <div class="text-muted mt-1">
                        <?php echo e(__('Analyze vehicle usage and productivity during business hours')); ?>

                    </div>
                </div>
                <div class="col-auto ms-auto d-print-none">
                    <div class="btn-list">
                        <button class="btn btn-primary" onclick="printReport()">
                            <i class="ti ti-printer me-2"></i>
                            <?php echo e(__('Print')); ?>

                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Page body -->
    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('Vehicle Selection')); ?></label>
                            <div class="input-group">
                                <select class="form-select" id="vehicleSelect">
                                    <option value=""><?php echo e(__('Select Vehicle')); ?></option>
                                </select>
                                <button type="button" class="btn btn-outline-secondary d-none" id="vehicleSettingsLink"
                                    onclick="goToVehicleSettings()" title="<?php echo e(__('Go to Vehicle Settings')); ?>">
                                    <i class="icon ti ti-settings"></i>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label"><?php echo e(__('Start Date')); ?></label>
                            <input type="date" class="form-control" id="startDate" value="<?php echo e($start_date); ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label"><?php echo e(__('End Date')); ?></label>
                            <input type="date" class="form-control" id="endDate" value="<?php echo e($end_date); ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label"><?php echo e(__('Analysis Type')); ?></label>
                            <select class="form-select" id="analysisType">
                                <option value="all"><?php echo e(__('All Working Hours')); ?></option>
                                <option value="moving"><?php echo e(__('Moving Only')); ?></option>
                                <option value="operational"><?php echo e(__('Operational Only')); ?></option>
                                <option value="idle"><?php echo e(__('Idle During Hours')); ?></option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">&nbsp;</label>
                            <button class="btn btn-primary w-100" onclick="loadWorkingHoursData()">
                                <i class="ti ti-search me-2"></i>
                                <?php echo e(__('Update Report')); ?>

                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="row row-cards mb-3" id="summaryCards">
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader"><?php echo e(__('Working Sessions')); ?></div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-briefcase ms-1 text-primary" style="font-size: 1.5rem;"></i>
                                </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="workingSessions">-</div>
                            </div>
                            <div id="apex-working-sessions" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted" id="sessionsBreakdown">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader"><?php echo e(__('Working Hours')); ?></div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-clock ms-1 text-success" style="font-size: 1.5rem;"></i>
                                </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="workingHours">-</div>
                            </div>
                            <div id="apex-working-hours" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted" id="hoursBreakdown">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader"><?php echo e(__('Utilization Rate')); ?></div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-percentage ms-1 text-warning" style="font-size: 1.5rem;"></i>
                                </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="utilizationRate">-</div>
                            </div>
                            <div id="apex-utilization-rate" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted" id="utilizationBreakdown">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader"><?php echo e(__('Most Productive')); ?></div>
                                <div class="ms-auto lh-1">
                                    <i class="ti ti-trending-up ms-1 text-info" style="font-size: 1.5rem;"></i>
                                </div>
                            </div>
                            <div class="d-flex align-items-baseline">
                                <div class="h1 mb-0 me-2" id="mostProductiveDay">-</div>
                            </div>
                            <div id="apex-efficiency-score" style="width: 100%; height: 40px;"></div>
                            <div class="text-muted" id="productivityBreakdown">-</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Vehicle Efficiency Comparison Chart -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Vehicle Efficiency Comparison')); ?></h3>
                </div>
                <div class="card-body">
                    <div id="vehicleComparisonChart" style="height: 400px;"></div>
                </div>
            </div>

            <!-- Charts Row -->
            <div class="row mb-3">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Daily Working Hours Trend')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="workingHoursTrendChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Hourly Utilization')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="hourlyUtilizationChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Business Hours Settings -->
            <div class="card mb-3 d-none" id="businessHoursCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Business Hours Settings')); ?></h3>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-vcenter" id="businessHoursTable">
                            <thead>
                                <tr>
                                    <th><?php echo e(__('Day')); ?></th>
                                    <th><?php echo e(__('Start Time')); ?></th>
                                    <th><?php echo e(__('End Time')); ?></th>
                                    <th><?php echo e(__('Status')); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Working Hours Details Table -->
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Working Hours Details')); ?></h3>
                    <div class="card-actions">
                        <div class="btn-list">
                            <button class="btn btn-sm"
                                onclick="tableWorkingHours.download('csv', 'working-hours-report.csv')">
                                <i class="ti ti-download"></i> <?php echo e(__('CSV')); ?>

                            </button>
                            <button class="btn btn-sm" onclick="toggleColumns()">
                                <i class="ti ti-columns"></i> <?php echo e(__('Columns')); ?>

                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div id="working-hours-table"></div>
                </div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-briefcase text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No working hours data found')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('No vehicle activity was detected during business hours for the selected period.')); ?>

                    </p>
                </div>
            </div>

            <!-- No Vehicle Selected State -->
            <div class="text-center py-4 d-none" id="noVehicleState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-car text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No Vehicle Selected')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('Please select a vehicle from the dropdown above to generate the working hours analysis
                        report.')); ?>

                    </p>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden"><?php echo e(__('Loading...')); ?></span>
                </div>
                <div class="mt-2"><?php echo e(__('Generating working hours report...')); ?></div>
            </div>
        </div>
    </div>
</div>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('styles'); ?>
<style>
    .efficiency-badge {
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 11px;
        font-weight: 600;
    }

    .efficiency-high {
        background-color: #10b981;
        color: white;
    }

    .efficiency-medium {
        background-color: #f59e0b;
        color: white;
    }

    .efficiency-low {
        background-color: #ef4444;
        color: white;
    }

    .status-badge {
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 11px;
        font-weight: 600;
    }

    .status-moving {
        background-color: #10b981;
        color: white;
    }

    .status-parked {
        background-color: #f59e0b;
        color: white;
    }

    .status-idle {
        background-color: #ef4444;
        color: white;
    }
</style>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('scripts'); ?>
<script>
    // --- Unit Variables ---
    var speedUnit = "";
    var lengthUnit = "";
    var currency = "";
    var vehicleList = [];
    let tableWorkingHoursData;
    let tableWorkingHours;
    let raw_logs = [];

    $('#summaryCards').addClass('d-none');
    $('.row.mb-3').addClass('d-none'); // Charts row
    $('#vehicleComparisonChart').closest('.card').addClass('d-none');
    $('#working-hours-table').closest('.card').addClass('d-none');
    $('#loadingState').addClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#noVehicleState').removeClass('d-none');

    onAppReady(function() {      
        loadVehicles()
        .then(initializeTable)
        .then(loadWorkingHoursData);
    });
    
    // Initialize table
    async function initializeTable() {
        tableWorkingHoursData  = createTabulator("#working-hours-table", {
            initialSort: [
                {column: "date", dir: "desc"}
            ],
            columns: [
                {title: "#", field: "id", hozAlign: "center", width: 50},
                {
                    title: "<?php echo e(__('Vehicle')); ?>", 
                    field: "vehicle_name"
                },
                {
                    title: "<?php echo e(__('Date')); ?>", 
                    field: "date",
                    sorter: "string",
                    formatter: function(cell) {
                        const value = cell.getValue();
                        if (!value) return '-';
                        return dateTimeFormatter(value);
                    }
                },
                {
                    title: "<?php echo e(__('Working Hours')); ?>", 
                    field: "working_hours", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const hours = cell.getValue();
                        if (!hours || hours === 0) return '-';
                        return formatDuration(hours * 60); // Convert hours to minutes for formatting
                    }
                },
                {
                    title: "<?php echo e(__('Utilization (%)')); ?>", 
                    field: "utilization_rate", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const rate = cell.getValue();
                        if (!rate || rate === 0) return '-';
                        return `${rate.toFixed(1)}%`;
                    }
                },
                {
                    title: "<?php echo e(__('Avg Speed')); ?>", 
                    field: "avg_speed", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const speed = cell.getValue();
                        if (!speed || speed === 0) return '-';
                        return `${speed.toFixed(1)} ${speedUnit}`;
                    }
                },
                {
                    title: "<?php echo e(__('Actions')); ?>", 
                    field: "actions", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const data = cell.getData();
                        let buttons = '';
                        if (data.vehicle_id && data.date) {
                            const urlEncoded = btoa(JSON.stringify({
                                did: data.vehicle_id,
                                date: data.date,
                                start_time: "00:01",
                                end_time: "23:59"
                            }));
                            buttons += `<a href="/vehicle-map/${urlEncoded}" class="btn btn-sm btn-info me-1" title="<?php echo e(__('Vehicle History')); ?>">
                                <i class="ti ti-history"></i>
                                <span class="d-none d-md-inline"><?php echo e(__("History")); ?></span>
                            </a>`;
                        }
                        return buttons;
                    }
                }
            ]
        });
    }

    // Load vehicles
    async function loadVehicles() {
        http.post({
            url: '/api/vehicles-list-with-alarm',
            data: {
                alarm_id: 307
            },
            loading: false
        }).then(function(response) {
            if (response.success && response.data.vehicles) {
                const select = $('#vehicleSelect');
                select.empty();
                select.append('<option value=""><?php echo e(__("Please select a vehicle")); ?></option>');
                
                // Separate vehicles with and without business hours monitoring
                const vehiclesWithMonitoring = [];
                const vehiclesWithoutMonitoring = [];
                
                response.data.vehicles.forEach(vehicle => {
                    if (vehicle.has_alarm) {
                        vehiclesWithMonitoring.push(vehicle);
                    } else {
                        vehiclesWithoutMonitoring.push(vehicle);
                    }
                });
                
                // Add vehicles with monitoring first
                if (vehiclesWithMonitoring.length > 0) {
                    select.append('<optgroup label="<?php echo e(__("Vehicles with Business Hours Monitoring")); ?>">');
                    vehiclesWithMonitoring.forEach(vehicle => {
                        select.append(`<option value="${vehicle.did}">${vehicle.name} ✓</option>`);
                    });
                    select.append('</optgroup>');
                }
                
                // Add vehicles without monitoring
                if (vehiclesWithoutMonitoring.length > 0) {
                    select.append('<optgroup label="<?php echo e(__("Vehicles without Business Hours Monitoring")); ?>">');
                    vehiclesWithoutMonitoring.forEach(vehicle => {
                        select.append(`<option value="${vehicle.did}" disabled>${vehicle.name} ✗</option>`);
                    });
                    select.append('</optgroup>');
                }
                
                // Add settings link if there are vehicles without monitoring
                if (vehiclesWithoutMonitoring.length > 0) {
                    $('#vehicleSettingsLink').removeClass('d-none');
                } else {
                    $('#vehicleSettingsLink').addClass('d-none');
                }
                
                // Show warning if no vehicles have monitoring enabled
                if (vehiclesWithMonitoring.length === 0) {
                    showConfirmDialog({
                        title: '<?php echo e(__("Warning")); ?>',
                        message: '<?php echo e(__("No vehicles have business hours monitoring enabled. Please enable business hours monitoring for at least one vehicle to use this report.")); ?>',
                        confirmText: '<?php echo e(__("Go to Settings")); ?>',
                        cancelText: '<?php echo e(__("Cancel")); ?>',
                        type: 'warning',
                        onConfirm: function() {
                            window.location.href = '/vehicles-list';
                        },
                        onCancel: function() {}
                    });
                }
            }
        }).catch(function(error) {
            console.error('<?php echo e(__("Error loading vehicles")); ?>:', error);
        });
    }

    // Load working hours data
    async function loadWorkingHoursData() {
        const vehicleId = document.getElementById('vehicleSelect').value;
        const startDate = document.getElementById('startDate').value;
        const endDate = document.getElementById('endDate').value;
        const analysisType = document.getElementById('analysisType').value;

        // Check if vehicle is selected
        if (!vehicleId) {
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("Please select a vehicle")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Check if selected vehicle has business hours monitoring enabled
        const selectedOption = $('#vehicleSelect option:selected');
        if (selectedOption.is(':disabled') || selectedOption.text().includes('✗')) {
            const vehicleName = selectedOption.text().replace(' ✗', '');
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: `<?php echo e(__("Business hours monitoring is not enabled for")); ?> "${vehicleName}". <?php echo e(__("Please enable business hours monitoring in vehicle settings to use this report.")); ?>`,
                confirmText: '<?php echo e(__("Go to Settings")); ?>',
                cancelText: '<?php echo e(__("Cancel")); ?>',
                type: 'danger',
                onConfirm: function() {
                    window.location.href = '/vehicles-list';
                },
                onCancel: function() {}
            });
            return;
        }

        // Show loading state
        showLoadingState();

        const requestData = {
                device_id: vehicleId,
                start_date: startDate,
                end_date: endDate
        };

        http.post({
            url: '/api/reports/working-hours',
            data: {
                data: Base64Helper.encode(requestData)
            },
            loading: false
        }).then(function(response) {
    
            // --- Set units after data is loaded ---
            speedUnit = sessionSettings?.speedUnit || 'km/h';
            lengthUnit = sessionSettings?.lengthUnit || 'km';
            currency = __(sessionSettings?.currency || 'TRY');
            // ---
            if (response.success) {
                const reportData = response.data;
                raw_logs = reportData.raw_logs;
                
                var message = __('#name# used #value# records').replace('#name#', __('Working Hours Report')).replace('#value#', reportData.raw_logs ? reportData.raw_logs.length : 0);
                showReportDataProcessingInfo(message);
                
                // Process raw data in JavaScript
                processWorkingHoursData(reportData, analysisType);
            } else {
                showConfirmDialog({
                    title: '<?php echo e(__("Error")); ?>',
                    message: response.message || '<?php echo e(__("Data not found")); ?>',
                    confirmText: '<?php echo e(__("OK")); ?>',
                    cancelText: '',
                    type: 'danger',
                    onConfirm: function() {},
                    onCancel: function() {}
                });
            }
            hideLoadingState();
        }).catch(function(error) {
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: '<?php echo e(__("Error fetching data")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                cancelText: '',
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
            hideLoadingState();
        });
    }

    // Process working hours data
    function processWorkingHoursData(reportData, analysisType) {
        if (!reportData || !reportData.raw_logs || reportData.raw_logs.length === 0) {
            showNoDataState();
            return;
        }

        // Parse business hours from settings
        const businessHours = parseBusinessHours(reportData.settings);
        
        // Analyze working hours usage
        const workingHoursUsage = analyzeWorkingHoursUsage(reportData.raw_logs, businessHours);
        
        if (workingHoursUsage.length === 0) {
            showNoDataState();
            return;
        }

        // Calculate summary statistics
        const summary = calculateSummary(workingHoursUsage);
        
        // Display the processed data
        displayReport({
            device: reportData.device,
            settings: businessHours,
            report_period: reportData.report_period,
            summary: summary,
            detailed_usage: workingHoursUsage
        });
    }

    // Calculate summary statistics
    function calculateSummary(sessions) {
        const totalSessions = sessions.length;
        const totalWorkingHours = sessions.reduce((sum, session) => sum + session.working_hours, 0);
        const totalDistance = sessions.reduce((sum, session) => sum + session.distance_km, 0);
        
        // Most productive day
        const dayCounts = {};
        sessions.forEach(session => {
            dayCounts[session.day_of_week] = (dayCounts[session.day_of_week] || 0) + session.working_hours;
        });
        
        let mostProductiveDay = null;
        if (Object.keys(dayCounts).length > 0) {
            const maxDay = Object.keys(dayCounts).reduce((a, b) => dayCounts[a] > dayCounts[b] ? a : b);
            mostProductiveDay = {
                day: maxDay,
                hours: dayCounts[maxDay]
            };
        }
        
        // Average utilization rate
        const avgUtilizationRate = sessions.length > 0 ? 
            sessions.reduce((sum, session) => sum + session.utilization_rate, 0) / sessions.length : 0;
        
        return {
            total_working_sessions: totalSessions,
            total_working_hours: {
                hours: totalWorkingHours,
                formatted: formatDuration(totalWorkingHours * 60)
            },
            total_distance: Math.round(totalDistance * 100) / 100,
            avg_utilization_rate: avgUtilizationRate,
            most_productive_day: mostProductiveDay
        };
    }

    // Display report data
    function displayReport(data) {
        // Show all sections
        $('#summaryCards').removeClass('d-none');
        $('.row.mb-3').removeClass('d-none'); // Charts row
        $('#businessHoursCard').removeClass('d-none');
        $('#vehicleComparisonChart').closest('.card').removeClass('d-none');
        $('#working-hours-table').closest('.card').removeClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');

        // Update summary cards
        $('#workingSessions').text(data.summary.total_working_sessions);
        $('#workingHours').text(data.summary.total_working_hours.formatted);
        $('#utilizationRate').text(data.summary.avg_utilization_rate.toFixed(1) + '%');
        
        if (data.summary.most_productive_day) {
            $('#mostProductiveDay').text(__(data.summary.most_productive_day.day));
        }

        // Display business hours
        displayBusinessHours(data.settings);

        // Display detailed usage
        displayDetailedUsage(data.detailed_usage);

        // Update charts
        updateCharts(data.detailed_usage);
        
        // Generate daily data for sparklines
        const dailyData = generateDailyData(data.detailed_usage);
        initializeSparklines(dailyData.workingSessions, dailyData.workingHours, dailyData.utilizationRates, dailyData.dates);
    }

    // Display business hours table
    function displayBusinessHours(settings) {
        const tbody = document.querySelector('#businessHoursTable tbody');
        tbody.innerHTML = '';

        const days = [__('Day_1'), __('Day_2'), __('Day_3'), __('Day_4'), __('Day_5'), __('Day_6'), __('Day_7')];
        
        days.forEach((day, index) => {
            const dayNumber = index + 1;
            const setting = settings[dayNumber];
            
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${day}</td>
                <td>${setting ? setting.start : '-'}</td>
                <td>${setting ? setting.end : '-'}</td>
                <td>
                    <span class="badge text-white ${setting ? 'bg-success' : 'bg-danger'}">
                        ${setting ? '<?php echo e(__("Seted")); ?>' : '<?php echo e(__("Not Set")); ?>'}
                    </span>
                </td>
            `;
            tbody.appendChild(row);
        });
    }

    // Display detailed usage table
    function displayDetailedUsage(usage) {
        if (usage && usage.length > 0) {
            tableWorkingHoursData.setData(usage); 
        } else {
            table.setData([]);
        }
    }

    // Generate daily data for sparklines
    function generateDailyData(sessions) {
        // Get date range from selected dates
        const startDate = new Date(document.getElementById('startDate').value);
        const endDate = new Date(document.getElementById('endDate').value);
        
        // Initialize daily arrays
        const dailyWorkingSessions = [];
        const dailyWorkingHours = [];
        const dailyUtilizationRates = [];
        const dailyDates = [];
        
        // Generate date range
        const dates = [];
        const currentDate = new Date(startDate);
        while (currentDate <= endDate) {
            dates.push(new Date(currentDate));
            currentDate.setDate(currentDate.getDate() + 1);
        }
        
        // Process each date
        dates.forEach(date => {
            const dateStr = date.toISOString().split('T')[0];
            const daySessions = sessions.filter(session => {
                const sessionDate = new Date(session.start_time).toISOString().split('T')[0];
                return sessionDate === dateStr;
            });
            
            // Store date for tooltips
            dailyDates.push(dateStr);
            
            // Count working sessions for the day
            dailyWorkingSessions.push(daySessions.length);
            // Calculate total working hours for the day
            const dayWorkingHours = daySessions.reduce((sum, session) => sum + session.working_hours, 0);
            dailyWorkingHours.push(Math.round(dayWorkingHours * 100) / 100);
            // Calculate average utilization rate for the day
            const dayUtilizationRate = daySessions.length > 0 ? 
                daySessions.reduce((sum, session) => sum + session.utilization_rate, 0) / daySessions.length : 0;
            dailyUtilizationRates.push(Math.round(dayUtilizationRate * 100) / 100);
        });
        
        return {
            workingSessions: dailyWorkingSessions,
            workingHours: dailyWorkingHours,
            utilizationRates: dailyUtilizationRates,
            dates: dailyDates
        };
    }

    // Parse business hours from alarm settings (similar to after-hours report)
    function parseBusinessHours(settings) {
        const businessHours = {};
        
        for (let i = 1; i <= 7; i++) {
            const startTime = settings[`start_time_${i}`] || null;
            const endTime = settings[`end_time_${i}`] || null;
            
            if (startTime && endTime) {
                businessHours[i] = {
                    start: startTime,
                    end: endTime,
                    day_name: getDayName(i)
                };
            }
        }
        
        return businessHours;
    }

    // Get day name by number
    function getDayName(dayNumber) {
        const days = {
            1: '<?php echo e(__("Day_1")); ?>', // Monday
            2: '<?php echo e(__("Day_2")); ?>', // Tuesday
            3: '<?php echo e(__("Day_3")); ?>', // Wednesday
            4: '<?php echo e(__("Day_4")); ?>', // Thursday
            5: '<?php echo e(__("Day_5")); ?>', // Friday
            6: '<?php echo e(__("Day_6")); ?>', // Saturday
            7: '<?php echo e(__("Day_7")); ?>' // Sunday
        };
        
        return days[dayNumber] || `Day_${dayNumber}`;
    }

    // Check if given time is during business hours
    function isBusinessHoursTime(dayOfWeek, timeOfDay, businessHours) {
        // Adjust day of week (JavaScript uses 0=Sunday, we use 1=Monday)
        const adjustedDay = dayOfWeek === 0 ? 7 : dayOfWeek;
        
        if (!businessHours[adjustedDay]) {
            // No business hours set for this day - consider all time as after hours
            return false;
        }
        
        const startTime = businessHours[adjustedDay].start;
        const endTime = businessHours[adjustedDay].end;
        
        // Handle overnight shifts (e.g., 22:00 to 06:00)
        if (startTime > endTime) {
            return timeOfDay >= startTime || timeOfDay <= endTime;
        } else {
            return timeOfDay >= startTime && timeOfDay <= endTime;
        }
    }

    // Analyze device logs to find working hours usage
    function analyzeWorkingHoursUsage(rawLogs, businessHours) {
        const workingHoursSessions = [];
        let currentSession = null;

        // Sort logs by timestamp to ensure chronological order
        const sortedLogs = rawLogs.sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));

        sortedLogs.forEach((log, index) => {
            // Fix date parsing - handle Y-m-d H:i:s format properly
            let logTime;
            if (log.timestamp) {
                // If timestamp is in Y-m-d H:i:s format, convert it properly
                if (typeof log.timestamp === 'string' && log.timestamp.includes(' ')) {
                    // Convert '2025-05-31 21:04:46' to ISO format
                    const [datePart, timePart] = log.timestamp.split(' ');
                    logTime = new Date(datePart + 'T' + timePart + '.000Z');
                } else {
                    logTime = new Date(log.timestamp);
                }
            } else {
                console.error('No timestamp found in log:', log);
                return;
            }
            
            const dayOfWeek = logTime.getDay(); // 0=Sunday, 1=Monday, etc.
            const timeOfDay = logTime.toTimeString().substring(0, 8);
            
            // Check if this time is during business hours
            const isWorkingHours = isBusinessHoursTime(dayOfWeek, timeOfDay, businessHours);
            
            if (isWorkingHours) {
                if (!currentSession) {
                    // Start new session
                    currentSession = {
                        start_time: logTime,
                        start_location: {
                            latitude: log.latitude,
                            longitude: log.longitude
                        },
                        logs: [],
                        last_activity_time: logTime,
                        last_activity_location: {
                            latitude: log.latitude,
                            longitude: log.longitude
                        }
                    };
                }
                
                currentSession.logs.push(log);
                currentSession.last_activity_time = logTime;
                currentSession.last_activity_location = {
                    latitude: log.latitude,
                    longitude: log.longitude
                };
            } else {
                // After hours - end current session if exists
                if (currentSession) {
                    const finalizedSession = finalizeWorkingSession(currentSession, workingHoursSessions.length + 1);
                    
                    // Only add session if it has meaningful activity
                    if (finalizedSession.working_hours > 0) {
                        workingHoursSessions.push(finalizedSession);
                    }
                    currentSession = null;
                }
            }
        });
        
        // Don't forget the last session
        if (currentSession) {
            const finalizedSession = finalizeWorkingSession(currentSession, workingHoursSessions.length + 1);
            
            // Only add session if it has meaningful activity
            if (finalizedSession.working_hours > 0) {
                workingHoursSessions.push(finalizedSession);
            }
        }
        
        return workingHoursSessions;
    }

    // Finalize a working session with calculated metrics
    function finalizeWorkingSession(session, sessionId) {
        let startTime = session.start_time;
        let endTime = session.last_activity_time;

        // Calculate working hours (only during business hours)
        const workingHours = Math.floor((endTime - startTime) / (1000 * 60 * 60)); // hours
        
        // Calculate distance
        let totalDistance = 0;
        let totalSpeed = 0;
        let speedCount = 0;
        
        for (let i = 1; i < session.logs.length; i++) {
            const prev = session.logs[i-1];
            const curr = session.logs[i];
            
            // Calculate distance
            const distance = calculateDistance(
                parseFloat(prev.latitude), parseFloat(prev.longitude),
                parseFloat(curr.latitude), parseFloat(curr.longitude)
            );
            totalDistance += distance;
            
            // Calculate speed
            const speed = parseFloat(curr.speed) || 0;
            if (speed > 0) {
                totalSpeed += speed;
                speedCount++;
            }
        }
        
        const avgSpeed = speedCount > 0 ? totalSpeed / speedCount : 0;
        const utilizationRate = workingHours > 0 ? (workingHours / 8) * 100 : 0; // Assuming 8-hour workday

        // Get vehicle_name from the first log (should be present after backend fix)
        const vehicleName = session.logs[0]?.vehicle_name || (window.reportDeviceName || null);
        
        return {
            id: sessionId,
            vehicle_id: session.logs[0]?.device_id,
            vehicle_name: vehicleName,
            start_time: startTime.toISOString(),
            end_time: endTime.toISOString(),
            working_hours: workingHours,
            distance_km: Math.round(totalDistance * 100) / 100,
            avg_speed: avgSpeed,
            utilization_rate: utilizationRate,
            day_of_week: startTime.toLocaleDateString('en-US', { weekday: 'long' }),
            date: startTime.toISOString().split('T')[0],
            start_location: session.start_location,
            end_location: session.last_activity_location
        };
    }

    // Calculate distance between two points
    function calculateDistance(lat1, lon1, lat2, lon2) {
        const R = 6371; // Earth's radius in km
        const dLat = (lat2 - lat1) * Math.PI / 180;
        const dLon = (lon2 - lon1) * Math.PI / 180;
        const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                  Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
                  Math.sin(dLon/2) * Math.sin(dLon/2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
        return R * c;
    }

    // Create ApexCharts sparkline
    function makeApexSparkline(divId, data, color, label, dates = []) {
        // Clear existing chart
        $("#" + divId).html('');
        
        var options = {
            chart: {
                type: 'line',
                height: 40,
                sparkline: { enabled: true }
            },
            series: [{
                name: label,
                data: data
            }],
            stroke: { curve: 'smooth', width: 2 },
            colors: [color],
            tooltip: {
                enabled: true,
                x: { show: true },
                y: {
                    title: {
                        formatter: function() { return label; }
                    },
                    formatter: function(value, { dataPointIndex }) {
                        let tooltipText = '';
                        
                        // Add date if available
                        if (dates && dates[dataPointIndex]) {
                            const date = new Date(dates[dataPointIndex]);
                            tooltipText += `<strong>${date.toLocaleDateString()}</strong><br>`;
                        }
                        
                        // Add value with proper formatting
                        if (label === '<?php echo e(__("Working Hours")); ?>') {
                            tooltipText += formatDuration(value * 60);
                        } else if (label === '<?php echo e(__("Utilization (%)")); ?>') {
                            tooltipText += parseFloat(value).toFixed(2) + '%';
                        } else if (label === '<?php echo e(__("Cost per Hour")); ?>') {
                            tooltipText += parseFloat(value).toFixed(2) + ' ' + currency;
                        } else {
                            tooltipText += parseFloat(value).toFixed(2);
                        }
                        
                        return tooltipText;
                    }
                }
            }
        };
        new ApexCharts(document.querySelector("#" + divId), options).render();
    }

    // Initialize sparklines
    function initializeSparklines(workingSessions, workingHours, utilizationRates, dates = []) {
        makeApexSparkline('apex-working-sessions', workingSessions, '#6366f1', '<?php echo e(__("Working Sessions")); ?>', dates);
        makeApexSparkline('apex-working-hours', workingHours, '#206bc4', '<?php echo e(__("Working Hours")); ?>', dates);
        makeApexSparkline('apex-utilization-rate', utilizationRates, '#22c55e', '<?php echo e(__("Utilization (%)")); ?>', dates);
    }

    // Update summary from processed data
    function updateSummaryFromData(data) {
        if (!data || data.length === 0) {
            $('#workingSessions').text('0');
            $('#workingHours').text('0 <?php echo e(__("h")); ?>');
            $('#utilizationRate').text('0%');
            $('#mostProductiveDay').text('-');
            $('#sessionsBreakdown').text('<?php echo e(__("Total working sessions")); ?>');
            $('#hoursBreakdown').text('<?php echo e(__("Total working hours")); ?>');
            $('#utilizationBreakdown').text('<?php echo e(__("Average utilization")); ?>');
            $('#productivityBreakdown').text('<?php echo e(__("Most productive day")); ?>');
            
            // Initialize empty sparklines
            initializeSparklines([], [], []);
            return;
        }
        
        const totalSessions = data.length;
        const totalWorkingHours = data.reduce((sum, item) => sum + item.working_hours, 0);
        const avgUtilizationRate = data.reduce((sum, item) => sum + item.utilization_rate, 0) / data.length;
        
        // Find most productive day
        const dayCounts = {};
        data.forEach(item => {
            const day = new Date(item.date).toLocaleDateString('en-US', { weekday: 'long' });
            dayCounts[day] = (dayCounts[day] || 0) + item.working_hours;
        });
        
        let mostProductiveDay = '-';
        let maxHours = 0;
        Object.keys(dayCounts).forEach(day => {
            if (dayCounts[day] > maxHours) {
                maxHours = dayCounts[day];
                mostProductiveDay = day;
            }
        });
        
        $('#workingSessions').text(totalSessions);
        $('#workingHours').text(formatDuration(totalWorkingHours * 60));
        $('#utilizationRate').text(avgUtilizationRate.toFixed(1) + '%');
        $('#mostProductiveDay').text(mostProductiveDay);
        
        $('#sessionsBreakdown').text('<?php echo e(__("Total working sessions")); ?>');
        $('#hoursBreakdown').text('<?php echo e(__("Total working hours")); ?>');
        $('#utilizationBreakdown').text('<?php echo e(__("Average utilization")); ?>');
        $('#productivityBreakdown').text('<?php echo e(__("Most productive day")); ?>');
        
        // Generate daily data for sparklines
        const dailyData = generateDailyData(data);
        initializeSparklines(dailyData.workingSessions, dailyData.workingHours, dailyData.utilizationRates, dailyData.dates);
    }

    // Update charts
    function updateCharts(sessions) {
        // Working hours trend chart
        const trendData = sessions.reduce((acc, session) => {
            const date = dateTimeFormatter(session.date);
            if (!acc[date]) acc[date] = { working: 0 };
            acc[date].working += session.working_hours;
            return acc;
        }, {});

        const dates = Object.keys(trendData).sort();
        const workingData = dates.map(date => trendData[date].working);

        renderWorkingHoursTrendChart(dates, workingData);

        // Hourly utilization chart (same logic as after-hours timeOfDayChart)
        const hourCounts = Array(24).fill(0);
        raw_logs.forEach(log => {
            const hour = log.timestamp.split(' ')[1].split(':')[0];
            hourCounts[hour]++;
        });
        renderHourlyUtilizationChart(hourCounts);

        // Vehicle comparison chart
        const vehicleData = sessions.reduce((acc, session) => {
            if (!acc[session.vehicle_name]) {
                acc[session.vehicle_name] = {
                    working_hours: 0,
                    utilization_rate: 0,
                    count: 0
                };
            }
            acc[session.vehicle_name].working_hours += session.working_hours;
            acc[session.vehicle_name].utilization_rate += session.utilization_rate;
            acc[session.vehicle_name].count += 1;
            return acc;
        }, {});

        // Calculate averages
        Object.keys(vehicleData).forEach(vehicle => {
            const count = vehicleData[vehicle].count;
            vehicleData[vehicle].utilization_rate /= count;
        });

        renderVehicleComparisonChart(vehicleData);
    }

    // Update table
    function updateTable(data) {
        if (data && data.length > 0) {
            tableWorkingHours.setData(data);
        } else {
            tableWorkingHours.setData([]);
        }
    }

    // Helper functions
    function formatDuration(minutes) {
        // Convert to integer minutes first
        const totalMinutes = Math.round(minutes);
        
        if (totalMinutes < 60) {
            return `${totalMinutes} <?php echo e(__("min")); ?>`;
        } else {
            const hours = Math.floor(totalMinutes / 60);
            const mins = totalMinutes % 60;
            if (mins === 0) {
                return `${hours} <?php echo e(__("h")); ?>`;
            } else {
                return `${hours} <?php echo e(__("h")); ?> ${mins} <?php echo e(__("m")); ?>`;
            }
        }
    }

    function toggleColumns() {
        showConfirmDialog({
            title: '<?php echo e(__("Info")); ?>',
            message: '<?php echo e(__("Column selector feature will be added")); ?>',
            confirmText: '<?php echo e(__("OK")); ?>',
            cancelText: '',
            type: 'info',
            onConfirm: function() {},
            onCancel: function() {}
        });
    }

    function printReport() {
        window.print();
    }

    // Go to vehicle settings page
    function goToVehicleSettings() {
        window.location.href = '/vehicles-list';
    }

    // State management functions
    function showLoadingState() {
        $('#loadingState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('.row.mb-3').addClass('d-none'); // Charts row
        $('#businessHoursCard').addClass('d-none');
        $('#vehicleComparisonChart').closest('.card').addClass('d-none');
        $('#working-hours-table').closest('.card').addClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
    }

    function hideLoadingState() {
        $('#loadingState').addClass('d-none');
    }

    function showNoDataState() {
        $('#noDataState').removeClass('d-none');
        $('#noVehicleState').addClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('.row.mb-3').addClass('d-none'); // Charts row
        $('#businessHoursCard').addClass('d-none');
        $('#vehicleComparisonChart').closest('.card').addClass('d-none');
        $('#working-hours-table').closest('.card').addClass('d-none');
    }

    function showNoVehicleState() {
        $('#noVehicleState').removeClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('.row.mb-3').addClass('d-none'); // Charts row
        $('#businessHoursCard').addClass('d-none');
        $('#vehicleComparisonChart').closest('.card').addClass('d-none');
        $('#working-hours-table').closest('.card').addClass('d-none');
    }

    function hideAllStates() {
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
        $('#summaryCards').removeClass('d-none');
        $('.row.mb-3').removeClass('d-none'); // Charts row
        $('#businessHoursCard').removeClass('d-none');
        $('#vehicleComparisonChart').closest('.card').removeClass('d-none');
        $('#working-hours-table').closest('.card').removeClass('d-none');
    }

    function renderWorkingHoursTrendChart(labels, data) {
        // Clear existing chart
        $("#workingHoursTrendChart").html('');
        
        // Format data to max 2 decimal places
        const formatData = (data) => data.map(val => parseFloat(val.toFixed(2)));
        
        var options = {
            chart: {
                type: 'area',
                height: 300,
                toolbar: { show: false }
            },
            series: [
                { name: '<?php echo e(__("Working Hours")); ?>', data: formatData(data) }
            ],
            xaxis: {
                categories: labels,
                title: { text: '<?php echo e(__("Date")); ?>' }
            },
            yaxis: {
                title: { text: '<?php echo e(__("Hours")); ?>' },
                labels: {
                    formatter: function(val) {
                        return parseFloat(val).toFixed(2);
                    }
                }
            },
            colors: ['#206bc4'],
            stroke: { curve: 'smooth', width: 2 },
            fill: {
                type: 'gradient',
                gradient: {
                    shadeIntensity: 1,
                    opacityFrom: 0.7,
                    opacityTo: 0.2,
                    stops: [0, 100]
                }
            },
            markers: { size: 4 },
            tooltip: {
                y: {
                    formatter: function(val) {
                        return parseFloat(val).toFixed(2) + ' <?php echo e(__("h")); ?>';
                    }
                }
            }
        };
        new ApexCharts(document.querySelector("#workingHoursTrendChart"), options).render();
    }

    function renderHourlyUtilizationChart(hourlyData) {
        // Clear existing chart
        $("#hourlyUtilizationChart").html('');
        
        var options = {
            chart: {
                type: 'bar',
                height: 300,
                toolbar: { show: false }
            },
            series: [
                { name: '<?php echo e(__("Usage")); ?>', data: hourlyData }
            ],
            xaxis: {
                categories: Array.from({length: 24}, (_, i) => i + ':00'),
                title: { text: '<?php echo e(__("Hour")); ?>' }
            },
            yaxis: {
                title: { text: '<?php echo e(__("Count")); ?>' },
                beginAtZero: true,
                labels: {
                    formatter: function(val) {
                        return parseFloat(val).toFixed(0);
                    }
                }
            },
            colors: ['#ff922b'],
            plotOptions: {
                bar: {
                    horizontal: false,
                    columnWidth: '55%',
                    endingShape: 'rounded'
                }
            },
            dataLabels: {
                enabled: true,
                formatter: function(val) {
                    return parseFloat(val).toFixed(0);
                }
            },
            tooltip: {
                y: {
                    formatter: function(val) {
                        return parseFloat(val).toFixed(0);
                    }
                }
            }
        };
        new ApexCharts(document.querySelector("#hourlyUtilizationChart"), options).render();
    }

    function renderVehicleComparisonChart(vehicleData) {
        // Clear existing chart
        $("#vehicleComparisonChart").html('');
        
        const vehicles = Object.keys(vehicleData);
        const utilizationRates = vehicles.map(v => parseFloat(vehicleData[v].utilization_rate.toFixed(2)));
        const workingHours = vehicles.map(v => parseFloat(vehicleData[v].working_hours.toFixed(2)));

        var options = {
            chart: {
                type: 'bar',
                height: 400,
                toolbar: { show: false }
            },
            series: [
                {
                    name: '<?php echo e(__("Utilization (%)")); ?>',
                    data: utilizationRates
                },
                {
                    name: '<?php echo e(__("Working Hours")); ?>',
                    data: workingHours
                }
            ],
            xaxis: {
                categories: vehicles,
                title: { text: '<?php echo e(__("Vehicles")); ?>' }
            },
            yaxis: [
                {
                    title: { text: '<?php echo e(__("Utilization (%)")); ?>' },
                    max: 100,
                    labels: {
                        formatter: function(val) {
                            return parseFloat(val).toFixed(2);
                        }
                    }
                },
                {
                    opposite: true,
                    title: { text: '<?php echo e(__("Working Hours")); ?>' },
                    max: 24,
                    labels: {
                        formatter: function(val) {
                            return parseFloat(val).toFixed(2);
                        }
                    }
                }
            ],
            colors: ['#22c55e', '#206bc4'],
            plotOptions: {
                bar: {
                    horizontal: false,
                    columnWidth: '55%',
                    endingShape: 'rounded'
                }
            },
            dataLabels: {
                enabled: true,
                formatter: function(val) {
                    return parseFloat(val).toFixed(2);
                }
            },
            tooltip: {
                y: {
                    formatter: function(val) {
                        return parseFloat(val).toFixed(2);
                    }
                }
            },
            legend: { position: 'top' }
        };
        new ApexCharts(document.querySelector("#vehicleComparisonChart"), options).render();
    }

</script>
<?php $__env->stopSection(); ?>
<?php echo $__env->make('layouts.app', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?><?php /**PATH C:\xampp\htdocs\aractakipv1\resources\views\reports\working-hours.blade.php ENDPATH**/ ?>