<?php $__env->startSection('title', __('After Hours Report')); ?>

<?php $__env->startSection('content'); ?>
<div class="page-wrapper" id="after-hours-report">
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted"><?php echo e(__('Reports')); ?></a> / <?php echo e(__('Security and Compliance
                        Reports')); ?>

                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-clock me-2"></i>
                        <?php echo e(__('After Hours Report')); ?>

                    </h2>
                    <div class="text-muted mt-1">
                        <?php echo e(__('Monitor vehicle usage outside of business hours')); ?>

                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Filters')); ?></h3>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('Vehicle')); ?></label>
                            <div class="input-group">
                                <select class="form-select" id="vehicleSelect">
                                    <option value=""><?php echo e(__('Select Vehicle')); ?></option>
                                </select>
                                <button type="button" class="btn btn-outline-secondary d-none" id="vehicleSettingsLink"
                                    onclick="goToVehicleSettings()" title="<?php echo e(__('Go to Vehicle Settings')); ?>">
                                    <i class="icon ti ti-settings"></i>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('Start Date')); ?></label>
                            <input type="date" class="form-control" id="startDate" value="<?php echo e(date('Y-m-d', strtotime('-1 week'))); ?>">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('End Date')); ?></label>
                            <input type="date" class="form-control" id="endDate" value="<?php echo e(date('Y-m-t')); ?>">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">&nbsp;</label>
                            <div>
                                <button type="button" class="btn btn-primary w-100" onclick="generateReport()">
                                    <i class="icon ti ti-search me-1"></i>
                                    <?php echo e(__('Search')); ?>

                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="row row-cards mb-3" id="summaryCards">
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader"><?php echo e(__('Total Usage')); ?></div>
                            </div>
                            <div class="h1 mb-0" id="totalSessions">-</div>
                            <div class="text-muted" id="sessionsTrend">
                                <?php echo e(__('Number of times vehicle was used after hours')); ?>

                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader"><?php echo e(__('Total Duration')); ?></div>
                            </div>
                            <div class="h1 mb-0" id="totalDuration">-</div>
                            <div class="text-muted" id="durationTrend">
                                <?php echo e(__('Total time spent using vehicle after hours')); ?>

                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader"><?php echo e(__('Total Distance')); ?></div>
                            </div>
                            <div class="h1 mb-0" id="totalDistance">-</div>
                            <div class="text-muted" id="distanceTrend"><?php echo e(__('Total distance traveled after hours')); ?>

                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="subheader"><?php echo e(__('Most Active')); ?></div>
                            </div>
                            <div class="h1 mb-0" id="mostActiveDay">-</div>
                            <div class="text-muted" id="mostActiveTime"><?php echo e(__('Day with most after-hours activity')); ?>

                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Charts Row -->
            <div class="row mb-3" id="chartsRow">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Daily Activity Trend')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="dailyTrendChart"></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Duration Distribution')); ?></h3>
                            <div class="card-subtitle text-muted"><?php echo e(__('How long were the after-hours usage periods?')); ?></div>
                        </div>
                        <div class="card-body">
                            <div id="durationChart"></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="row mb-3" id="chartsRow2">
                <div class="col-lg-8">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Time of Day Analysis')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="timeOfDayChart"></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-4">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Day of Week Activity')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="dayOfWeekChart"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Business Hours Settings -->
            <div class="card mb-3 d-none" id="businessHoursCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Business Hours Settings')); ?></h3>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-vcenter" id="businessHoursTable">
                            <thead>
                                <tr>
                                    <th><?php echo e(__('Day')); ?></th>
                                    <th><?php echo e(__('Start Time')); ?></th>
                                    <th><?php echo e(__('End Time')); ?></th>
                                    <th><?php echo e(__('Status')); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Map Visualization -->
            <div class="card mb-3 d-none" id="mapCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('After-Hours Activity Map')); ?></h3>
                    <div class="card-actions">
                        <div class="btn-list">
                            <button type="button" class="btn btn-sm btn-outline-primary" onclick="fitMapToBounds()">
                                <i class="icon ti ti-arrows-maximize me-1"></i>
                                <?php echo e(__('Fit to Bounds')); ?>

                            </button>
                            <button type="button" class="btn btn-sm btn-outline-info" onclick="toggleFullscreen()">
                                <i class="icon ti ti-maximize me-1"></i>
                                <?php echo e(__('Fullscreen')); ?>

                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div id="after-hours-map"></div>
                </div>
            </div>

            <!-- Detailed Usage Table -->
            <div class="card d-none" id="usageTableCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Detailed After-Hours Usage')); ?></h3>
                </div>
                <div class="card-body p-0">
                    <div id="usage-table"></div>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden"><?php echo e(__('Loading...')); ?></span>
                </div>
                <div class="mt-2"><?php echo e(__('Generating report...')); ?></div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-clock text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No after-hours usage found')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('No vehicle activity was detected outside of business hours for the selected period.')); ?>

                    </p>
                </div>
            </div>

            <!-- No Vehicle Selected State -->
            <div class="text-center py-4 d-none" id="noVehicleState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-car text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No Vehicle Selected')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('Please select a vehicle from the dropdown above to generate the after-hours report.')); ?>

                    </p>
                </div>
            </div>
        </div>
    </div>
</div>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('styles'); ?>
<style>
    #after-hours-report .icon {
        font-size: inherit !important;
    }

    #after-hours-map {
        height: 500px;
    }

    /* Prevent scrolling when map elements are focused */
    #after-hours-map:focus {
        outline: none;
    }

    /* Prevent page scrolling when map links are clicked */
    #after-hours-map a:focus,
    .map-link:focus {
        outline: none;
    }

    /* Prevent default link behavior */
    .map-link {
        text-decoration: none;
        cursor: pointer;
    }

    .after-hours-marker .map-icon-label {
        background-color: transparent !important;
        font-size: 15px !important;
        font-weight: 500 !important;
        color: #fff !important;
        padding: 0 !important;
        margin: 0 !important;
        border-radius: 0 !important;
        border: 0 !important;
    }

    .session-start-marker {
        background-color: #28a745 !important;
        border: 2px solid white !important;
        box-shadow: 0 0 4px rgba(0, 0, 0, 0.3) !important;
    }

    .session-end-marker {
        background-color: #dc3545 !important;
        border: 2px solid white !important;
        box-shadow: 0 0 4px rgba(0, 0, 0, 0.3) !important;
    }

    .temp-location-marker {
        background: transparent !important;
        border: none !important;
    }

    .temp-location-marker i {
        text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.5);
        animation: pulse 1s infinite;
    }

    @keyframes pulse {
        0% {
            transform: scale(1);
        }

        50% {
            transform: scale(1.2);
        }

        100% {
            transform: scale(1);
        }
    }

    .tabulator-cell[tabulator-field="duration_minutes"] {
        font-weight: 600;
    }

    .session-badge {
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 11px;
        font-weight: 600;
    }

    .session-long {
        background-color: #dc3545;
        color: white;
    }

    .session-medium {
        background-color: #fd7e14;
        color: white;
    }

    .session-short {
        background-color: #28a745;
        color: white;
    }

    /* Bounce animation for markers */
    .bounce-animation {
        animation: bounce 1s infinite;
    }

    @keyframes bounce {
        0%, 20%, 50%, 80%, 100% {
            transform: translateY(0);
        }
        40% {
            transform: translateY(-10px);
        }
        60% {
            transform: translateY(-5px);
        }
    }

    /* Session info popup styling */
    .session-info-popup .leaflet-popup-content-wrapper {
        background: rgba(255, 255, 255, 0.95);
        backdrop-filter: blur(10px);
        border-radius: 8px;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
        border: 1px solid rgba(0, 0, 0, 0.1);
    }

    .session-info-popup .leaflet-popup-content {
        margin: 8px 12px;
        font-size: 12px;
    }

    .session-popup-content h6 {
        color: #206bc4;
        margin-bottom: 8px;
        font-size: 13px;
        font-weight: 600;
    }

    .session-popup-content .row {
        margin: 0;
    }

    .session-popup-content .col-6 {
        padding: 0 4px;
    }

    .session-popup-content small {
        font-size: 10px;
    }

    .session-popup-content strong {
        font-size: 11px;
        color: #333;
    }

    .session-popup-content .border-top {
        border-top: 1px solid rgba(0, 0, 0, 0.1) !important;
    }

    .session-popup-content .border-top {
        border-top: 1px solid rgba(0, 0, 0, 0.1) !important;
    }

    /* Make session markers look clickable */
    .session-start-marker,
    .session-end-marker {
        cursor: pointer !important;
        transition: all 0.3s ease !important;
    }
</style>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('scripts'); ?>
<script>

    let reportData = null;
    let vehicles = [];
    let myMap;
    let tableAfterHours;
    
    // Chart variables
    let dailyTrendChart;
    let durationChart;
    let timeOfDayChart;
    let dayOfWeekChart;
    
    var speedUnit = '';
    var lengthUnit = '';
    var liquidUnit = '';
    var currency = '';

    $('#summaryCards').addClass('d-none');
    $('#chartsRow').addClass('d-none');
    $('#chartsRow2').addClass('d-none');
    $('#businessHoursCard').addClass('d-none');
    $('#usageTableCard').addClass('d-none');
    $('#mapCard').addClass('d-none');
    $('#loadingState').addClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#noVehicleState').removeClass('d-none');
    
    onAppReady(function() {        
        loadVehicles()
        .then(initializeMap)
        .then(initializeTable)
        .then(initializeCharts)
        .then(generateReport);
    });


    function initializeCharts() {
        try {
            // Daily Trend Chart
            const dailyTrendElement = document.getElementById('dailyTrendChart');
            if (dailyTrendElement) {
                dailyTrendChart = new ApexCharts(dailyTrendElement, {
                    chart: {
                        type: 'line',
                        height: 300,
                        toolbar: {
                            show: false
                        }
                    },
                    series: [{
                        name: '<?php echo e(__('Usage')); ?>',
                        data: []
                    }],
                    xaxis: {
                        categories: [],
                        title: {
                            text: '<?php echo e(__('Date')); ?>'
                        }
                    },
                    yaxis: {
                        title: {
                            text: '<?php echo e(__('Usage')); ?>'
                        },
                        min: 0
                    },
                    colors: ['#206bc4'],
                    stroke: {
                        curve: 'smooth',
                        width: 3
                    },
                    fill: {
                        type: 'gradient',
                        gradient: {
                            shadeIntensity: 1,
                            opacityFrom: 0.7,
                            opacityTo: 0.1,
                            stops: [0, 90, 100]
                        }
                    },
                    dataLabels: {
                        enabled: false
                    },
                    legend: {
                        show: false
                    },
                    tooltip: {
                        y: {
                            formatter: function(value) {
                                return value + ' <?php echo e(__('sessions')); ?>';
                            }
                        }
                    }
                });
                dailyTrendChart.render();
            } else {
                console.error('Daily trend chart element not found');
            }

            // Duration Distribution Chart
            const durationElement = document.getElementById('durationChart');
            if (durationElement) {
                durationChart = new ApexCharts(durationElement, {
                    chart: {
                        type: 'bar',
                        height: 300,
                        toolbar: {
                            show: false
                        }
                    },
                    series: [{
                        name: '<?php echo e(__('Usage')); ?>',
                        data: []
                    }],
                    xaxis: {
                        categories: [],
                        title: {
                            text: '<?php echo e(__('Duration (min)')); ?>'
                        }
                    },
                    yaxis: {
                        title: {
                            text: '<?php echo e(__('Count')); ?>'
                        },
                        min: 0
                    },
                    colors: ['#51cf66'],
                    plotOptions: {
                        bar: {
                            borderRadius: 4,
                            dataLabels: {
                                position: 'top'
                            }
                        }
                    },
                    dataLabels: {
                        enabled: true,
                        formatter: function(val) {
                            return val;
                        },
                        style: {
                            fontSize: '12px',
                            colors: ['#304758']
                        }
                    },
                    legend: {
                        show: false
                    },
                    tooltip: {
                        y: {
                            formatter: function(value) {
                                return value + ' <?php echo e(__('sessions')); ?>';
                            }
                        }
                    }
                });
                durationChart.render();
            } else {
                console.error('Duration chart element not found');
            }

            // Time of Day Chart
            const timeOfDayElement = document.getElementById('timeOfDayChart');
            if (timeOfDayElement) {
                timeOfDayChart = new ApexCharts(timeOfDayElement, {
                    chart: {
                        type: 'bar',
                        height: 250,
                        toolbar: {
                            show: false
                        }
                    },
                    series: [{
                        name: '<?php echo e(__('Usage')); ?>',
                        data: []
                    }],
                    xaxis: {
                        categories: [],
                        title: {
                            text: '<?php echo e(__('Hour')); ?>'
                        }
                    },
                    yaxis: {
                        title: {
                            text: '<?php echo e(__('Count')); ?>'
                        },
                        min: 0
                    },
                    colors: ['#ff922b'],
                    plotOptions: {
                        bar: {
                            borderRadius: 4,
                            dataLabels: {
                                position: 'top'
                            }
                        }
                    },
                    dataLabels: {
                        enabled: true,
                        formatter: function(val) {
                            return val;
                        },
                        style: {
                            fontSize: '11px',
                            colors: ['#304758']
                        }
                    },
                    legend: {
                        show: false
                    },
                    tooltip: {
                        y: {
                            formatter: function(value) {
                                return value + ' <?php echo e(__('sessions')); ?>';
                            }
                        }
                    }
                });
                timeOfDayChart.render();
            } else {
                console.error('Time of day chart element not found');
            }

            // Day of Week Chart
            const dayOfWeekElement = document.getElementById('dayOfWeekChart');
            if (dayOfWeekElement) {
                dayOfWeekChart = new ApexCharts(dayOfWeekElement, {
                    chart: {
                        type: 'donut',
                        height: 250,
                        toolbar: {
                            show: false
                        }
                    },
                    series: [],
                    labels: [],
                    colors: ['#206bc4', '#845ef7', '#ff922b', '#51cf66', '#ff6b6b', '#ffd43b', '#94d82d'],
                    plotOptions: {
                        pie: {
                            donut: {
                                size: '60%',
                                labels: {
                                    show: true,
                                    total: {
                                        show: true,
                                        label: '<?php echo e(__('Total')); ?>',
                                        fontSize: '16px',
                                        fontWeight: 600
                                    }
                                }
                            }
                        }
                    },
                    dataLabels: {
                        enabled: true,
                        formatter: function(val, opts) {
                            return opts.w.globals.seriesTotals[opts.seriesIndex];
                        }
                    },
                    legend: {
                        position: 'bottom'
                    },
                    tooltip: {
                        y: {
                            formatter: function(value) {
                                return value + ' <?php echo e(__('sessions')); ?>';
                            }
                        }
                    }
                });
                dayOfWeekChart.render();
            } else {
                console.error('Day of week chart element not found');
            }
        } catch (error) {
            console.error('Error initializing charts:', error);
        }
    }

    function updateCharts(sessions) {
        // Daily Trend
        const dateCounts = {};
        sessions.forEach(s => {
            const date = formatDate(s.date);
            dateCounts[date] = (dateCounts[date] || 0) + 1;
        });
        const dates = Object.keys(dateCounts).sort();
        
        if (dailyTrendChart) {
            dailyTrendChart.updateOptions({
                xaxis: {
                    categories: dates
                }
            });
            
            dailyTrendChart.updateSeries([{
                name: '<?php echo e(__('Usage')); ?>',
                data: dates.map(d => dateCounts[d])
            }]);
        } else {
            console.error('Daily trend chart is not initialized');
        }

        // Duration Distribution (grouped by meaningful time ranges)
        const durationRanges = {
            '0-15': 0,    // 0-15 minutes
            '15-30': 0,   // 15-30 minutes
            '30-60': 0,   // 30-60 minutes (1 hour)
            '60-120': 0,  // 1-2 hours
            '120+': 0     // 2+ hours
        };
        
        sessions.forEach(s => {
            const duration = s.duration_minutes;
            if (duration <= 15) {
                durationRanges['0-15']++;
            } else if (duration <= 30) {
                durationRanges['15-30']++;
            } else if (duration <= 60) {
                durationRanges['30-60']++;
            } else if (duration <= 120) {
                durationRanges['60-120']++;
            } else {
                durationRanges['120+']++;
            }
        });
        
        const durationLabels = [
            '<?php echo e(__("0-15 min")); ?>',
            '<?php echo e(__("15-30 min")); ?>', 
            '<?php echo e(__("30-60 min")); ?>',
            '<?php echo e(__("1-2 hours")); ?>',
            '<?php echo e(__("2+ hours")); ?>'
        ];
        
        if (durationChart) {
            durationChart.updateOptions({
                xaxis: {
                    categories: durationLabels
                }
            });
            
            durationChart.updateSeries([{
                name: '<?php echo e(__('Usage')); ?>',
                data: Object.values(durationRanges)
            }]);
        } else {
            console.error('Duration chart is not initialized');
        }

        // Time of Day (hourly)
        const hourCounts = Array(24).fill(0);
        sessions.forEach(s => {
            const hour = new Date(s.start_time).getHours();
            hourCounts[hour]++;
        });
        if (timeOfDayChart) {
            timeOfDayChart.updateOptions({
                xaxis: {
                    categories: Array.from({length:24},(_,i)=>i+':00')
                }
            });
            
            timeOfDayChart.updateSeries([{
                name: '<?php echo e(__('Usage')); ?>',
                data: hourCounts
            }]);
        } else {
            console.error('Time of day chart is not initialized');
        }

        // Day of Week
        const dowCounts = Array(7).fill(0);
        sessions.forEach(s => {
            const day = new Date(s.start_time).getDay();
            dowCounts[day]++;
        });
        if (dayOfWeekChart) {
            dayOfWeekChart.updateOptions({
                labels: [__('Sunday'),__('Monday'),__('Tuesday'),__('Wednesday'),__('Thursday'),__('Friday'),__('Saturday')]
            });
            
            dayOfWeekChart.updateSeries(dowCounts);
        } else {
            console.error('Day of week chart is not initialized');
        }
    }

    // Initialize map
    async function initializeMap() {
        myMap = new LeafletMap();
        myMap.initMap("after-hours-map", {
            // Default zoom level for Turkey
        });
    }

    // Refresh map when container becomes visible
    function refreshMap() {
        if (myMap && myMap.mapLet) {
            // Trigger a resize event to force the map to recalculate its dimensions
            myMap.mapLet.invalidateSize();
        } else {
            // If map doesn't exist, initialize it
            initializeMap();
        }
    }

    // Initialize table
    async function initializeTable() {
        tableAfterHours = createTabulator("#usage-table", {
            initialSort: [
                {column: "start_time", dir: "desc"}
            ],
            columns: [
                {title: "#", field: "id", width: 50, hozAlign: "center"},
                {
                    title: "<?php echo e(__('Date')); ?>", 
                    field: "date",
                    formatter: function(cell) {
                        return formatDate(cell.getValue());
                    }
                },
                {
                    title: "<?php echo e(__('Day')); ?>", 
                    field: "day_of_week",
                    sorter: "string",
                    formatter: function(cell) {
                        return __(cell.getValue());
                    }
                },
                {
                    title: "<?php echo e(__('Start Time')); ?>", 
                    field: "start_time",
                    formatter: function(cell) {
                        //return formatTime(cell.getValue());
                        const startDate = new Date(cell.getValue()).toISOString().split('T')[0];
                        const startTime = new Date(cell.getValue()).toTimeString().substring(0, 5);
                        return dateTimeFormatter(startDate + ' ' + startTime);
                    }
                },
                {
                    title: "<?php echo e(__('End Time')); ?>", 
                    field: "end_time",
                    formatter: function(cell) {
                        //return formatTime(cell.getValue());
                        // make cell.getValue() YYYY-MM-DD HH:MM:SS format
                        const endDate = new Date(cell.getValue()).toISOString().split('T')[0];
                        const endTime = new Date(cell.getValue()).toTimeString().substring(0, 5);
                        return dateTimeFormatter(endDate + ' ' + endTime);
                    }
                },
                {
                    title: "<?php echo e(__('Duration')); ?>", 
                    field: "duration_formatted", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const minutes = cell.getData().duration_minutes;
                        const durationText = cell.getValue();
                        
                        if (minutes >= 120) {
                            return `<span class="badge text-white bg-red">${durationText}</span>`;
                        } else if (minutes >= 60) {
                            return `<span class="badge text-white bg-orange">${durationText}</span>`;
                        } else {
                            return `<span class="badge text-white bg-green">${durationText}</span>`;
                        }
                    }
                },
                {
                    title: "<?php echo e(__('Distance')); ?>", 
                    field: "distance_km", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        return `${cell.getValue()} ${lengthUnit}`;
                    }
                },
                {
                    title: "<?php echo e(__('Start Location')); ?>", 
                    field: "start_location", 
                    formatter: function(cell) {
                        const data = cell.getData();
                        if (data.start_location && data.start_location.latitude && data.start_location.longitude) {
                            return `<a href="#" class="btn btn-sm btn-outline-primary map-link" onclick="showLocationOnMap(${data.start_location.latitude}, ${data.start_location.longitude}); return false;"><?php echo e(__('See on map')); ?></a>`;
                        }
                        return '-';
                    }
                },
                {
                    title: "<?php echo e(__('End Location')); ?>", 
                    field: "end_location", 
                    formatter: function(cell) {
                        const data = cell.getData();
                        if (data.end_location && data.end_location.latitude && data.end_location.longitude) {
                            return `<a href="#" class="btn btn-sm btn-outline-primary map-link" onclick="showLocationOnMap(${data.end_location.latitude}, ${data.end_location.longitude}); return false;"><?php echo e(__('See on map')); ?></a>`;
                        }
                        return '-';
                    }
                },
                {
                    title: "<?php echo e(__('Actions')); ?>", 
                    field: "actions", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const data = cell.getData();
                        let buttons = '';
                        
                        // Show session on map button
                        if (data.start_location && data.end_location) {
                            buttons += `<button class="btn btn-sm btn-primary me-1" onclick="showSessionOnMap(${data.id})" title="<?php echo e(__('Show session on map')); ?>">
                                <i class="ti ti-map-pin"></i>
                                <span class="d-none d-md-inline"><?php echo e(__("Session")); ?></span>
                            </button>`;
                        }
                        
                        // Vehicle history button
                        if (data.vehicle_id && data.start_time) {
                            const startDate = new Date(data.start_time).toISOString().split('T')[0];
                            const startTime = new Date(data.start_time).toTimeString().substring(0, 5);
                            const endTime = new Date(data.end_time).toTimeString().substring(0, 5);
                            //add 5 minutes to endTime
                            const endTimePlus5Minutes = new Date(data.end_time).getTime() + 5 * 60 * 1000;
                            const endTimePlus5MinutesString = new Date(endTimePlus5Minutes).toTimeString().substring(0, 5);
                            const urlEncoded = btoa(JSON.stringify({
                                did: data.vehicle_id,
                                date: startDate,
                                start_time: startTime,
                                end_time: endTimePlus5MinutesString
                            }));
                            buttons += `<a href="/vehicle-map/${urlEncoded}" class="btn btn-sm btn-info me-1" title="<?php echo e(__('Vehicle History')); ?>">
                                <i class="ti ti-history"></i>
                                <span class="d-none d-md-inline"><?php echo e(__("History")); ?></span>
                            </a>`;
                        }
                        
                        return buttons;
                    }
                }
            ]
        });
    }

    // Load vehicles for dropdown
    async function loadVehicles() {
        http.post({
            url: '/api/vehicles-list-with-alarm',
            data: {
                alarm_id: 307
            },
            loading: false
        }).then(function(response) {
            if (response.success && response.data.vehicles) {
                const select = $('#vehicleSelect');
                select.empty();
                select.append('<option value=""><?php echo e(__("Please select a vehicle")); ?></option>');
                
                // Separate vehicles with and without after-hours monitoring
                const vehiclesWithMonitoring = [];
                const vehiclesWithoutMonitoring = [];
                
                response.data.vehicles.forEach(vehicle => {
                    if (vehicle.has_alarm) {
                        vehiclesWithMonitoring.push(vehicle);
                    } else {
                        vehiclesWithoutMonitoring.push(vehicle);
                    }
                });
                
                // Add vehicles with monitoring first
                if (vehiclesWithMonitoring.length > 0) {
                    select.append('<optgroup label="<?php echo e(__("Vehicles with After-Hours Monitoring")); ?>">');
                    vehiclesWithMonitoring.forEach(vehicle => {
                        select.append(`<option value="${vehicle.did}">${vehicle.name} ✓</option>`);
                    });
                    select.append('</optgroup>');
                }
                
                // Add vehicles without monitoring
                if (vehiclesWithoutMonitoring.length > 0) {
                    select.append('<optgroup label="<?php echo e(__("Vehicles without After-Hours Monitoring")); ?>">');
                    vehiclesWithoutMonitoring.forEach(vehicle => {
                        select.append(`<option value="${vehicle.did}" disabled>${vehicle.name} ✗</option>`);
                    });
                    select.append('</optgroup>');
                }
                
                // Add settings link if there are vehicles without monitoring
                if (vehiclesWithoutMonitoring.length > 0) {
                    $('#vehicleSettingsLink').removeClass('d-none');
                } else {
                    $('#vehicleSettingsLink').addClass('d-none');
                }
                
                // Show warning if no vehicles have monitoring enabled
                if (vehiclesWithMonitoring.length === 0) {
                    showConfirmDialog({
                        title: '<?php echo e(__("Warning")); ?>',
                        message: '<?php echo e(__("No vehicles have after-hours monitoring enabled. Please enable after-hours monitoring for at least one vehicle to use this report.")); ?>',
                        confirmText: '<?php echo e(__("Go to Settings")); ?>',
                        cancelText: '<?php echo e(__("Cancel")); ?>',
                        type: 'warning',
                        onConfirm: function() {
                            window.location.href = '/vehicles-list';
                        },
                        onCancel: function() {}
                    });
                }
            }
        }).catch(function(error) {
            console.error('<?php echo e(__("Error loading vehicles")); ?>:', error);
        });
    }

    // Generate after-hours report
    async function generateReport() {
        const vehicleId = $('#vehicleSelect').val();
        const startDate = $('#startDate').val();
        const endDate = $('#endDate').val();

        if (!vehicleId) {
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("Please select a vehicle")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Check if selected vehicle has after-hours monitoring enabled
        const selectedOption = $('#vehicleSelect option:selected');
        if (selectedOption.is(':disabled') || selectedOption.text().includes('✗')) {
            const vehicleName = selectedOption.text().replace(' ✗', '');
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: `<?php echo e(__("After-hours monitoring is not enabled for")); ?> "${vehicleName}". <?php echo e(__("Please enable after-hours monitoring in vehicle settings to use this report.")); ?>`,
                confirmText: '<?php echo e(__("Go to Settings")); ?>',
                cancelText: '<?php echo e(__("Cancel")); ?>',
                type: 'danger',
                onConfirm: function() {
                    window.location.href = '/vehicles-list';
                },
                onCancel: function() {}
            });
            return;
        }

        if (!startDate || !endDate) {
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("Please select start and end dates")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                cancelText: '',
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Show loading state
        showLoadingState();

        const requestData = {
            device_id: vehicleId,
            start_date: startDate,
            end_date: endDate
        };

        http.post({
            url: '/api/reports/after-hours',
            data: {
                data: Base64Helper.encode(requestData)
            },
            loading: false
        })
        .then(data => {
            hideLoadingState();
            
            if (data.success) {
                // Set units after data is loaded
                speedUnit = __(sessionSettings.speedUnit || 'km/h');
                lengthUnit = __(sessionSettings.lengthUnit || 'km');
                liquidUnit = __(sessionSettings.liquidUnit || 'lt');
                currency = __(sessionSettings.currency || 'TRY');
                reportData = data.data;
                
                var message = __('#name# used #value# records').replace('#name#', __('After Hours Report')).replace('#value#', data.data.raw_logs ? data.data.raw_logs.length : 0);
                showReportDataProcessingInfo(message);
                
                //console.log("after-hours", reportData);
                processReportData(data.data);
            } else {
                showNoDataState();
                showConfirmDialog({
                    title: '<?php echo e(__("Error")); ?>',
                    message: data.message || '<?php echo e(__("Unknown error occurred")); ?>',
                    confirmText: '<?php echo e(__("OK")); ?>',
                    cancelText: '',
                    type: 'danger',
                    onConfirm: function() {},
                    onCancel: function() {}
                });
                console.error('Report generation failed:', data.message);
            }
        })
        .catch(err => {
            hideLoadingState();
            showNoDataState();
            var err =  err;
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: __(err.message),
                confirmText: '<?php echo e(__("OK")); ?>',
                cancelText: '',
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
            console.error('Error generating report:', err);
        });
    }

    // Process raw data and generate report
    function processReportData(data) {
        // Parse business hours from settings
        const businessHours = parseBusinessHours(data.settings);
        
        // Analyze after-hours usage with improved end time detection
        const afterHoursUsage = analyzeAfterHoursUsage(data.raw_logs, businessHours);
        
        if (afterHoursUsage.length === 0) {
            showNoDataState();
            return;
        }

        //console.log("processReportData", afterHoursUsage);
        // Calculate summary statistics
        const summary = calculateSummary(afterHoursUsage);
        
        // Display the processed data
        displayReport({
            device: data.device,
            settings: businessHours,
            report_period: data.report_period,
            summary: summary,
            detailed_usage: afterHoursUsage
        });
    }

    // Parse business hours from alarm settings
    function parseBusinessHours(settings) {
        const businessHours = {};
        
        for (let i = 1; i <= 7; i++) {
            const startTime = settings[`start_time_${i}`] || null;
            const endTime = settings[`end_time_${i}`] || null;
            
            if (startTime && endTime) {
                businessHours[i] = {
                    start: startTime,
                    end: endTime,
                    day_name: getDayName(i)
                };
            }
        }
        
        return businessHours;
    }

    // Get day name by number
    function getDayName(dayNumber) {
        const days = {
            1: '<?php echo e(__("Day_1")); ?>', // Monday
            2: '<?php echo e(__("Day_2")); ?>', // Tuesday
            3: '<?php echo e(__("Day_3")); ?>', // Wednesday
            4: '<?php echo e(__("Day_4")); ?>', // Thursday
            5: '<?php echo e(__("Day_5")); ?>', // Friday
            6: '<?php echo e(__("Day_6")); ?>', // Saturday
            7: '<?php echo e(__("Day_7")); ?>' // Sunday
        };
        
        return days[dayNumber] || `Day_${dayNumber}`;
    }

    // Analyze device logs to find after-hours usage with improved end time detection
    function analyzeAfterHoursUsage(rawLogs, businessHours) {
        const afterHoursSessions = [];
        let currentSession = null;

        // Sort logs by timestamp to ensure chronological order
        const sortedLogs = rawLogs.sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));

        sortedLogs.forEach((log, index) => {
            // Fix date parsing - handle Y-m-d H:i:s format properly
            let logTime;
            if (log.timestamp) {
                // If timestamp is in Y-m-d H:i:s format, convert it properly
                if (typeof log.timestamp === 'string' && log.timestamp.includes(' ')) {
                    // Convert '2025-05-31 21:04:46' to ISO format
                    const [datePart, timePart] = log.timestamp.split(' ');
                    logTime = new Date(datePart + 'T' + timePart + '.000Z');
                } else {
                    logTime = new Date(log.timestamp);
                }
            } else {
                console.error('No timestamp found in log:', log);
                return;
            }
            
            const dayOfWeek = logTime.getDay(); // 0=Sunday, 1=Monday, etc.
            const timeOfDay = logTime.toTimeString().substring(0, 8);
            
            // Check if this time is outside business hours
            const isAfterHours = isAfterHoursTime(dayOfWeek, timeOfDay, businessHours);
            
            if (isAfterHours) {
                if (!currentSession) {
                    // Start new session
                    currentSession = {
                        start_time: logTime,
                        start_location: {
                            latitude: log.latitude,
                            longitude: log.longitude
                        },
                        logs: [],
                        last_activity_time: logTime,
                        last_activity_location: {
                            latitude: log.latitude,
                            longitude: log.longitude
                        }
                    };
                }
                
                currentSession.logs.push(log);
                currentSession.last_activity_time = logTime;
                currentSession.last_activity_location = {
                    latitude: log.latitude,
                    longitude: log.longitude
                };
            } else {
                // Business hours - end current session if exists
                if (currentSession) {
                    // Find the actual end time by looking for stop points until next working time
                    const actualEndData = findActualSessionEnd(currentSession, sortedLogs, index, businessHours);
                    
                    const finalizedSession = finalizeSession({
                        ...currentSession,
                        end_time: actualEndData.end_time,
                        end_location: actualEndData.end_location
                    }, afterHoursSessions.length + 1);
                    
                    // Only add session if distance is greater than 0
                    if (finalizedSession.distance_km > 0) {
                        afterHoursSessions.push(finalizedSession);
                    }
                    currentSession = null;
                }
            }
        });
        
        // Don't forget the last session
        if (currentSession) {
            // Find the actual end time for the last session
            const actualEndData = findActualSessionEnd(currentSession, sortedLogs, sortedLogs.length, businessHours);
            
            const finalizedSession = finalizeSession({
                ...currentSession,
                end_time: actualEndData.end_time,
                end_location: actualEndData.end_location
            }, afterHoursSessions.length + 1);
            
            // Only add session if distance is greater than 0
            if (finalizedSession.distance_km > 0) {
                afterHoursSessions.push(finalizedSession);
            }
        }
        
        return afterHoursSessions;
    }

    // Find the next working datetime from a given time
    function findNextWorkingDateTime(fromTime, businessHours) {
        let currentTime = new Date(fromTime);
        let daysChecked = 0;
        const maxDaysToCheck = 14; // Prevent infinite loop
        
        while (daysChecked < maxDaysToCheck) {
            const dayOfWeek = currentTime.getDay();
            const adjustedDay = dayOfWeek === 0 ? 7 : dayOfWeek;
            const timeOfDay = currentTime.toTimeString().substring(0, 8);
            
            // Check if this day has business hours
            if (businessHours[adjustedDay]) {
                const startTime = businessHours[adjustedDay].start;
                const endTime = businessHours[adjustedDay].end;
                
                // Create datetime for business start time on this day
                const businessStartTime = new Date(currentTime);
                const [startHour, startMinute] = startTime.split(':').map(Number);
                businessStartTime.setHours(startHour, startMinute, 0, 0);
                
                // Check if we're currently in after-hours time
                const isAfterHours = isAfterHoursTime(dayOfWeek, timeOfDay, businessHours);
                
                if (isAfterHours) {
                    // We're in after-hours, so the next working time is the business start time of the next working day
                    // Move to next day first
                    currentTime.setDate(currentTime.getDate() + 1);
                    currentTime.setHours(0, 0, 0, 0);
                    daysChecked++;
                    continue;
                } else {
                    // We're in business hours, so the next working time is the business start time of the next working day
                    // Move to next day
                    currentTime.setDate(currentTime.getDate() + 1);
                    currentTime.setHours(0, 0, 0, 0);
                    daysChecked++;
                    continue;
                }
            } else {
                // No business hours set for this day - move to next day
                currentTime.setDate(currentTime.getDate() + 1);
                currentTime.setHours(0, 0, 0, 0);
                daysChecked++;
                continue;
            }
        }
        
        // If we reach here, we need to find the next working day
        // Start from the day after the original time
        let nextDay = new Date(fromTime);
        nextDay.setDate(nextDay.getDate() + 1);
        nextDay.setHours(0, 0, 0, 0);
        
        for (let i = 0; i < 7; i++) {
            const dayOfWeek = nextDay.getDay();
            const adjustedDay = dayOfWeek === 0 ? 7 : dayOfWeek;
            
            if (businessHours[adjustedDay]) {
                const startTime = businessHours[adjustedDay].start;
                const [startHour, startMinute] = startTime.split(':').map(Number);
                const nextWorkingTime = new Date(nextDay);
                nextWorkingTime.setHours(startHour, startMinute, 0, 0);
                
                return nextWorkingTime;
            }
            
            nextDay.setDate(nextDay.getDate() + 1);
        }
        
        // Fallback: if we can't find next working time, use current time + 24 hours
        const fallbackTime = new Date(fromTime.getTime() + 24 * 60 * 60 * 1000);
        return fallbackTime;
    }

    // Find actual session end by looking for stop points until next working time
    function findActualSessionEnd(session, allLogs, currentLogIndex, businessHours) {
        const nextWorkingTime = findNextWorkingDateTime(session.last_activity_time, businessHours);
        const stopThreshold = 5; // km/h - consider vehicle stopped if speed is below this
        const stopDurationThreshold = 10; // minutes - minimum time to consider it a stop
        const locationThreshold = 1; // km - consider same location if distance is less than this
        
        let lastStopTime = session.last_activity_time;
        let lastStopLocation = session.last_activity_location;
        let firstStopTime = null;
        let firstStopLocation = null;
        let currentStopStart = null;
        let currentStopLocation = null;
        let stopCount = 0;
        let sameLocationStops = [];
        
        // Look through logs after the session's last activity until next working time
        for (let i = currentLogIndex; i < allLogs.length; i++) {
            const log = allLogs[i];
            // Fix date parsing - handle Y-m-d H:i:s format properly
            let logTime;
            if (log.timestamp) {
                if (typeof log.timestamp === 'string' && log.timestamp.includes(' ')) {
                    const [datePart, timePart] = log.timestamp.split(' ');
                    logTime = new Date(datePart + 'T' + timePart + '.000Z');
                } else {
                    logTime = new Date(log.timestamp);
                }
            } else {
                continue;
            }
            
            if (logTime >= nextWorkingTime) {
                break;
            }
            const speed = parseFloat(log.speed) || 0;
            
            if (speed <= stopThreshold) {
                if (!currentStopStart) {
                    currentStopStart = logTime;
                    currentStopLocation = {
                        latitude: log.latitude,
                        longitude: log.longitude
                    };
                    stopCount++;
                } else {
                    const stopDuration = Math.floor((logTime - currentStopStart) / (1000 * 60));
                    if (stopDuration >= stopDurationThreshold) {
                        lastStopTime = logTime;
                        lastStopLocation = {
                            latitude: log.latitude,
                            longitude: log.longitude
                        };
                        const isSameLocation = sameLocationStops.some(stop => {
                            const distance = calculateDistance(
                                stop.latitude, stop.longitude,
                                log.latitude, log.longitude
                            );
                            return distance <= locationThreshold;
                        });
                        if (isSameLocation) {
                            sameLocationStops.push({
                                time: logTime,
                                latitude: log.latitude,
                                longitude: log.longitude,
                                duration: stopDuration
                            });
                        } else {
                            sameLocationStops = [{
                                time: logTime,
                                latitude: log.latitude,
                                longitude: log.longitude,
                                duration: stopDuration
                            }];
                            if (!firstStopTime) {
                                firstStopTime = logTime;
                                firstStopLocation = {
                                    latitude: log.latitude,
                                    longitude: log.longitude
                                };
                            }
                        }
                    }
                }
            } else {
                if (currentStopStart) {
                    const stopDuration = Math.floor((logTime - currentStopStart) / (1000 * 60));
                }
                currentStopStart = null;
                currentStopLocation = null;
            }
        }
        
        let finalEndTime = lastStopTime;
        let finalEndLocation = lastStopLocation;
        let endReason = "last_stop";
        
        if (sameLocationStops.length > 1) {
            const firstStop = sameLocationStops[0];
            finalEndTime = firstStop.time;
            finalEndLocation = {
                latitude: firstStop.latitude,
                longitude: firstStop.longitude
            };
            endReason = "first_stop_same_location";
        } else if (firstStopTime) {
            finalEndTime = firstStopTime;
            finalEndLocation = firstStopLocation;
            endReason = "first_stop";
        }
        
        const originalEndTime = session.last_activity_time;
        const actualEndTime = finalEndTime;
        const timeDifference = Math.floor((actualEndTime - originalEndTime) / (1000 * 60));
        
        return {
            end_time: finalEndTime,
            end_location: finalEndLocation,
            end_reason: endReason,
            same_location_stops: sameLocationStops.length
        };
    }

    // Check if given time is after hours
    function isAfterHoursTime(dayOfWeek, timeOfDay, businessHours) {
        // Adjust day of week (JavaScript uses 0=Sunday, we use 1=Monday)
        const adjustedDay = dayOfWeek === 0 ? 7 : dayOfWeek;
        
        if (!businessHours[adjustedDay]) {
            // No business hours set for this day - consider all time as after hours
            return true;
        }
        
        const startTime = businessHours[adjustedDay].start;
        const endTime = businessHours[adjustedDay].end;
        
        // Handle overnight shifts (e.g., 22:00 to 06:00)
        if (startTime > endTime) {
            return timeOfDay < endTime || timeOfDay > startTime;
        } else {
            return timeOfDay < startTime || timeOfDay > endTime;
        }
    }

    // Helper: Get true session logs and boundaries
    function getTrueSessionLogs(logs, afterHoursStart, nextWorkingStart) {
        const sessionLogs = logs.filter(log => {
            const logTime = new Date(log.timestamp);
            return logTime >= afterHoursStart && logTime < nextWorkingStart;
        });

        let trueStartIdx = sessionLogs.findIndex(log => parseFloat(log.speed) > 5);
        if (trueStartIdx === -1) trueStartIdx = 0;

        var endLog = sessionLogs.filter(log => log.speed > 0);
        let trueEndIdx = -1;
        for (let i = sessionLogs.length - 1; i >= 0; i--) {
            if (parseFloat(sessionLogs[i].speed) > 0) {
                trueEndIdx = i;
                break;
            }
        }
        if (trueEndIdx === -1) {
            trueEndIdx = sessionLogs.length - 1;
        }

        const logsForDistance = sessionLogs.slice(trueStartIdx, trueEndIdx + 1);

        return {
            startLog: sessionLogs[trueStartIdx],
            endLog: sessionLogs[trueEndIdx],
            logsForDistance
        };
    }

    // Finalize a session with calculated metrics
    function finalizeSession(session, sessionId) {
        let startTime = session.start_time;
        let endTime = session.end_time;

        // Cap endTime at the next working time
        const nextWorkingTime = findNextWorkingDateTime(startTime, reportData.settings);
        if (endTime > nextWorkingTime) {
            endTime = nextWorkingTime;
        }

        // Use only logs within after-hours period
        const afterHoursStart = new Date(startTime);
        const afterHoursEnd = new Date(nextWorkingTime);
        const { startLog, endLog, logsForDistance } = getTrueSessionLogs(session.logs, afterHoursStart, afterHoursEnd);

        // Set true start and end times
        if (startLog) startTime = new Date(startLog.timestamp);
        if (endLog) endTime = new Date(endLog.timestamp);

        // Calculate distance only for the true session logs
        let distance = calculateSessionDistance(logsForDistance);

        // If all logs are at the same location, set distance to 0
        let allSameLocation = true;
        if (logsForDistance && logsForDistance.length > 1) {
            const firstLat = parseFloat(logsForDistance[0].latitude);
            const firstLng = parseFloat(logsForDistance[0].longitude);
            for (let i = 1; i < logsForDistance.length; i++) {
                const lat = parseFloat(logsForDistance[i].latitude);
                const lng = parseFloat(logsForDistance[i].longitude);
                if (Math.abs(lat - firstLat) > 0.0001 || Math.abs(lng - firstLng) > 0.0001) {
                    allSameLocation = false;
                    break;
                }
            }
        }
        if (allSameLocation) {
            distance = 0;
        }

        const duration = Math.floor((endTime - startTime) / (1000 * 60)); // minutes

        return {
            id: sessionId,
            vehicle_id: reportData.device.did,
            start_time: startTime.toISOString(),
            end_time: endTime.toISOString(),
            duration_minutes: duration,
            duration_formatted: formatDuration(duration),
            distance_km: Math.round(distance * 100) / 100,
            start_location: session.start_location,
            end_location: session.end_location,
            day_of_week: startTime.toLocaleDateString('en-US', { weekday: 'long' }),
            date: startTime.toISOString().split('T')[0],
            end_reason: session.end_reason || 'last_stop',
            same_location_stops: session.same_location_stops || 0
        };
    }

    // Calculate distance for a session
    function calculateSessionDistance(logs) {
        let totalDistance = 0;
        for (let i = 1; i < logs.length; i++) {
            const prevLog = logs[i - 1];
            const currentLog = logs[i];
            const distance = calculateDistance(
                prevLog.latitude, prevLog.longitude,
                currentLog.latitude, currentLog.longitude
            );
            totalDistance += distance;
        }
        return totalDistance;
    }

    // Calculate distance between two GPS coordinates (Haversine formula)
    function calculateDistance(lat1, lon1, lat2, lon2) {
        const lat1Num = parseFloat(lat1);
        const lon1Num = parseFloat(lon1);
        const lat2Num = parseFloat(lat2);
        const lon2Num = parseFloat(lon2);
        const earthRadius = 6371;
        const latDelta = (lat2Num - lat1Num) * Math.PI / 180;
        const lonDelta = (lon2Num - lon1Num) * Math.PI / 180;
        const a = Math.sin(latDelta / 2) * Math.sin(latDelta / 2) +
                 Math.cos(lat1Num * Math.PI / 180) * Math.cos(lat2Num * Math.PI / 180) *
                 Math.sin(lonDelta / 2) * Math.sin(lonDelta / 2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        const distance = earthRadius * c;
        return distance;
    }

    // Format duration in human readable format
    function formatDuration(minutes) {
        const hours = Math.floor(minutes / 60);
        const mins = minutes % 60;
        
        // Use translations from tr.json
        const hourAbbr = __('hour') + ' ';
        const minuteAbbr = __('min') + ' ';
        
        if (hours > 0) {
            return `${hours}${hourAbbr} ${mins}${minuteAbbr}`;
        } else {
            return `${mins}${minuteAbbr}`;
        }
    }

    // Calculate summary statistics
    function calculateSummary(sessions) {
        const totalMinutes = sessions.reduce((sum, session) => sum + session.duration_minutes, 0);
        const totalDistance = sessions.reduce((sum, session) => sum + session.distance_km, 0);
        
        // Most active day
        const dayCounts = {};
        sessions.forEach(session => {
            dayCounts[session.day_of_week] = (dayCounts[session.day_of_week] || 0) + 1;
        });
        
        let mostActiveDay = null;
        if (Object.keys(dayCounts).length > 0) {
            const maxDay = Object.keys(dayCounts).reduce((a, b) => dayCounts[a] > dayCounts[b] ? a : b);
            mostActiveDay = {
                day: maxDay,
                sessions: dayCounts[maxDay]
            };
        }
        
        // Most active time period
        const hourCounts = {};
        sessions.forEach(session => {
            const startHour = new Date(session.start_time).getHours();
            hourCounts[startHour] = (hourCounts[startHour] || 0) + 1;
        });
        
        let mostActiveTime = null;
        if (Object.keys(hourCounts).length > 0) {
            const maxHour = Object.keys(hourCounts).reduce((a, b) => hourCounts[a] > hourCounts[b] ? a : b);
            mostActiveTime = {
                hour: parseInt(maxHour),
                time_range: `${maxHour.padStart(2, '0')}:00-${maxHour.padStart(2, '0')}:59`,
                sessions: hourCounts[maxHour]
            };
        }
        
        return {
            total_after_hours_sessions: sessions.length,
            total_after_hours_duration: {
                minutes: totalMinutes,
                formatted: formatDuration(totalMinutes)
            },
            total_after_hours_distance: Math.round(totalDistance * 100) / 100,
            most_active_day: mostActiveDay,
            most_active_time: mostActiveTime
        };
    }

    // Display report data
    function displayReport(data) {
        // Show all sections
        $('#summaryCards').removeClass('d-none');
        $('#chartsRow').removeClass('d-none');
        $('#chartsRow2').removeClass('d-none');
        $('#businessHoursCard').removeClass('d-none');
        $('#usageTableCard').removeClass('d-none');
        $('#mapCard').removeClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');

        // Update summary cards
        $('#totalSessions').text(data.summary.total_after_hours_sessions);
        $('#totalDuration').text(data.summary.total_after_hours_duration.formatted);
        $('#totalDistance').text(data.summary.total_after_hours_distance + ' ' + lengthUnit);
        
        if (data.summary.most_active_day) {
            $('#mostActiveDay').text(__(data.summary.most_active_day.day));
            $('#mostActiveTime').text(
                `${data.summary.most_active_day.sessions} <?php echo e(__('usage')); ?>`);
        }

        // Display business hours
        displayBusinessHours(data.settings);

        // Display detailed usage
        displayDetailedUsage(data.detailed_usage);

        // Update charts
        updateCharts(data.detailed_usage);
        
        // Refresh map after card becomes visible and update with sessions
        setTimeout(() => {
            refreshMap();
            updateMapWithSessions(data.detailed_usage);
        }, 100);
    }

    // Display business hours table
    function displayBusinessHours(settings) {
        const tbody = document.querySelector('#businessHoursTable tbody');
        tbody.innerHTML = '';

        const days = [__('Day_1'), __('Day_2'), __('Day_3'), __('Day_4'), __('Day_5'), __('Day_6'), __('Day_7')];
        
        days.forEach((day, index) => {
            const dayNumber = index + 1;
            const setting = settings[dayNumber];
            
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${day}</td>
                <td>${setting ? setting.start : '-'}</td>
                <td>${setting ? setting.end : '-'}</td>
                <td>
                    <span class="badge text-white ${setting ? 'bg-success' : 'bg-danger'}">
                        ${setting ? '<?php echo e(__("Seted")); ?>' : '<?php echo e(__("Not Set")); ?>'}
                    </span>
                </td>
            `;
            tbody.appendChild(row);
        });
    }

    // Display detailed usage table
    function displayDetailedUsage(usage) {
        if (usage && usage.length > 0) {
            tableAfterHours.setData(usage);
        } else {
            tableAfterHours.setData([]);
        }
    }

    // Show loading state
    function showLoadingState() {
        $('#loadingState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#chartsRow').addClass('d-none');
        $('#chartsRow2').addClass('d-none');
        $('#businessHoursCard').addClass('d-none');
        $('#usageTableCard').addClass('d-none');
        $('#mapCard').addClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
    }

    // Hide loading state
    function hideLoadingState() {
        $('#loadingState').addClass('d-none');
    }

    // Show no data state
    function showNoDataState() {
        $('#noDataState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#chartsRow').addClass('d-none');
        $('#chartsRow2').addClass('d-none');
        $('#businessHoursCard').addClass('d-none');
        $('#usageTableCard').addClass('d-none');
        $('#mapCard').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
    }

    // Show no vehicle selected state
    function showNoVehicleState() {
        $('#noVehicleState').removeClass('d-none');
    }

    // Hide no vehicle selected state
    function hideNoVehicleState() {
        $('#noVehicleState').addClass('d-none');
    }

    // Export report
    function exportReport() {
        if (!reportData) {
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("Please generate a report first")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                cancelText: '',
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Process data again for export
        const businessHours = parseBusinessHours(reportData.settings);
        const afterHoursUsage = analyzeAfterHoursUsage(reportData.raw_logs, businessHours);
        
        if (afterHoursUsage.length === 0) {
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("No data to export")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                cancelText: '',
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Create CSV content
        let csvContent = 'data:text/csv;charset=utf-8,';
        
        // Add headers
        csvContent += 'Date,Day,Start Time,End Time,Duration,Distance (km),Start Location,End Location\n';
        
        // Add data
        afterHoursUsage.forEach(session => {
            csvContent += `${formatDate(session.date)},${session.day_of_week},${formatTime(session.start_time)},${formatTime(session.end_time)},${session.duration_formatted},${session.distance_km},"${formatCoordinates(session.start_location)}","${formatCoordinates(session.end_location)}"\n`;
        });
        
        // Download file
        const encodedUri = encodeURI(csvContent);
        const link = document.createElement('a');
        link.setAttribute('href', encodedUri);
        link.setAttribute('download', `after-hours-report-${new Date().toISOString().split('T')[0]}.csv`);
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    // Print report
    function printReport() {
        window.print();
    }

    // Utility functions
    function formatDate(dateString) {
        return new Date(dateString).toLocaleDateString();
    }

    function formatTime(dateTimeString) {
        return new Date(dateTimeString).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    }

    function formatCoordinates(location) {
        if (!location || !location.latitude || !location.longitude) return '-';
        
        const lat = parseFloat(location.latitude);
        const lng = parseFloat(location.longitude);
        
        if (isNaN(lat) || isNaN(lng)) return '-';
        
        return `${lat.toFixed(6)}, ${lng.toFixed(6)}`;
    }

    // Map functions
    function updateMapWithSessions(sessions) {
        if (!myMap) {
            initializeMap();
        }
        
        // Clear existing markers
        myMap.removeMarkers();
        myMap.clearAllPolygons();
        myMap.removeCircles();
        
        if (!sessions || sessions.length === 0) {
            return;
        }
        
        // Add regions and places to the map
        addRegionsAndPlacesToMap();
        
        // Add session markers
        const markerModels = [];
        
        sessions.forEach((session, index) => {
            if (session.start_location && session.end_location) {
                const startLat = parseFloat(session.start_location.latitude);
                const startLng = parseFloat(session.start_location.longitude);
                const endLat = parseFloat(session.end_location.latitude);
                const endLng = parseFloat(session.end_location.longitude);
                
                if (!isNaN(startLat) && !isNaN(startLng) && !isNaN(endLat) && !isNaN(endLng)) {
                    // Start marker
                    markerModels.push({
                        id: `session-${session.id}-start`,
                        markerClassName: "after-hours-marker session-start-marker",
                        label: `<div class="badge bg-green">${formatTime(session.start_time)}</div>`,
                        popup: `
                            <strong><?php echo e(__("Session Start")); ?></strong><br>
                            <?php echo e(__("Date")); ?>: ${formatDate(session.date)}<br>
                            <?php echo e(__("Time")); ?>: ${formatTime(session.start_time)}<br>
                            <?php echo e(__("Duration")); ?>: ${session.duration_formatted}<br>
                            <?php echo e(__("Distance")); ?>: ${session.distance_km} km<br><br>
                            <button class="btn btn-sm btn-primary" onclick="animateSessionOnMap(${session.id})">
                                <i class="ti ti-player-play me-1"></i><?php echo e(__("Show Session")); ?>

                            </button>
                        `,
                        coord: [startLat, startLng],
                        icon: {
                            name: "ti ti-player-play",
                            color: "green",
                            direction: 0,
                            className: "session-start-marker",
                        },
                        // Add click handler data
                        sessionId: session.id,
                        isStartMarker: true
                    });
                    
                    // End marker
                    markerModels.push({
                        id: `session-${session.id}-end`,
                        markerClassName: "after-hours-marker session-end-marker",
                        label: `<div class="badge text-white bg-red">${formatTime(session.end_time)}</div>`,
                        popup: `
                            <strong><?php echo e(__("Session End")); ?></strong><br>
                            <?php echo e(__("Date")); ?>: ${formatDate(session.date)}<br>
                            <?php echo e(__("Time")); ?>: ${formatTime(session.end_time)}<br>
                            <?php echo e(__("Duration")); ?>: ${session.duration_formatted}<br>
                            <?php echo e(__("Distance")); ?>: ${session.distance_km} km<br><br>
                            <button class="btn btn-sm btn-primary" onclick="animateSessionOnMap(${session.id})">
                                <i class="ti ti-player-play me-1"></i><?php echo e(__("Show Session")); ?>

                            </button>
                        `,
                        coord: [endLat, endLng],
                        icon: {
                            name: "ti ti-player-stop",
                            color: "red",
                            direction: 0,
                            className: "session-end-marker",
                        },
                        // Add click handler data
                        sessionId: session.id,
                        isEndMarker: true
                    });
                }
            }
        });
        
        // Add markers to map
        if (markerModels.length > 0) {
            myMap.pinMarkers(markerModels);
            
            // Add route lines
            sessions.forEach(session => {
                if (session.start_location && session.end_location) {
                    const startLat = parseFloat(session.start_location.latitude);
                    const startLng = parseFloat(session.start_location.longitude);
                    const endLat = parseFloat(session.end_location.latitude);
                    const endLng = parseFloat(session.end_location.longitude);
                    
                    if (!isNaN(startLat) && !isNaN(startLng) && !isNaN(endLat) && !isNaN(endLng)) {
                        const route = L.polyline([
                            [startLat, startLng],
                            [endLat, endLng]
                        ], {
                            color: '#007bff',
                            weight: 3,
                            opacity: 0.7
                        }).addTo(myMap.mapLet);
                    }
                }
            });
            
            // Add click handlers to markers after they're added to the map
            setTimeout(() => {
                markerModels.forEach(markerModel => {
                    if (markerModel.sessionId) {
                        const markerElement = document.querySelector(`[data-marker-id="${markerModel.id}"]`);
                        if (markerElement) {
                            markerElement.style.cursor = 'pointer';
                            markerElement.addEventListener('click', function(e) {
                                // Prevent default popup behavior
                                e.stopPropagation();
                                
                                // Trigger the same animation as the table button
                                animateSessionOnMap(markerModel.sessionId);
                            });
                        }
                    }
                });
            }, 500);
            
            // Fit map to show all markers
            setTimeout(() => myMap.fitBounds(), 100);
        }
    }

    // Add regions and places to the map
    function addRegionsAndPlacesToMap() {
        // Load regions data
        http.post({
            url: '/regions-list',
            loading: false
        }).then(function(regionsResponse) {
            if (regionsResponse.success && regionsResponse.data) {
                const regions = regionsResponse.data;
                
                // Add regions to map
                regions.forEach(function(region) {
                    if (region.polygon_json) {
                        myMap.addPolygon({
                            id: region.id,
                            label: region.name,
                            color: "gray",
                            fillColor: "gray",
                            coords: region.polygon_json,
                        });
                    }
                });
            }
        }).catch(function(error) {
            console.error('Error loading regions:', error);
        });

        // Load places data
        http.post({
            url: '/places-list',
            loading: false
        }).then(function(placesResponse) {
            if (placesResponse.success && placesResponse.data) {
                const places = placesResponse.data;
                
                // Add places to map
                var placeMarkerColor = "gray";
                places.forEach(function(place) {
                    if (place.latitude && place.longitude) {
                        var marker = [
                            {
                                id: place.id,
                                markerClassName: "place-marker",
                                device_id: 0,
                                label: place.name,
                                status: place.status_id,
                                popup: place.name,
                                coord: [
                                    parseFloat(place.latitude),
                                    parseFloat(place.longitude),
                                ],
                                icon: {
                                    name: "fa-solid fa-location-dot",
                                    color: placeMarkerColor,
                                    direction: 0,
                                    className: "",
                                },
                            },
                        ];
                        myMap.pinMarkers(marker);
                        
                        // Add circle if radius is defined
                        if (place.radius && place.radius > 0) {
                            var circle = L.circle([place.latitude, place.longitude], {
                                radius: place.radius,
                                color: placeMarkerColor,
                                fillColor: placeMarkerColor,
                                fillOpacity: 0.2,
                            }).addTo(myMap.mapLet);
                        }
                    }
                });
            }
        }).catch(function(error) {
            console.error('Error loading places:', error);
        });
    }

    function showLocationOnMap(lat, lng) {
        if (!lat || !lng || isNaN(lat) || isNaN(lng)) {
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: '<?php echo e(__("Invalid coordinates")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                cancelText: '',
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }
        
        // Prevent any default scrolling behavior
        event.preventDefault();
        
        // Center map on the location with smooth animation
        myMap.mapLet.setView([parseFloat(lat), parseFloat(lng)], 16, {
            animate: true,
            duration: 0.5
        });
        
        // Add a temporary marker to highlight the location
        const tempMarker = L.marker([lat, lng], {
            icon: L.divIcon({
                className: 'temp-location-marker',
                html: '<i class="fa-solid fa-crosshairs" style="color: red; font-size: 20px;"></i>',
                iconSize: [20, 20],
                iconAnchor: [10, 10]
            })
        }).addTo(myMap.mapLet);
        
        // Remove the temporary marker after 3 seconds
        setTimeout(() => {
            myMap.mapLet.removeLayer(tempMarker);
        }, 3000);
        
        // Ensure map container is visible and focused
        $('#mapCard').removeClass('d-none');
        $('#after-hours-map').focus();
    }

    function showSessionOnMap(sessionId) {
        // Prevent any default scrolling behavior
        event.preventDefault();
        
        // Add loading animation to the clicked button
        const clickedButton = event.target.closest('button');
        const originalContent = clickedButton.innerHTML;
        clickedButton.innerHTML = '<i class="ti ti-loader ti-spin"></i> <?php echo e(__("Loading...")); ?>';
        clickedButton.disabled = true;
        
        // Call the animation function
        animateSessionOnMap(sessionId, clickedButton, originalContent);
    }

    // Main animation function that can be called from table button or map markers
    function animateSessionOnMap(sessionId, clickedButton = null, originalContent = null) {
        // Find the session in the table data
        const session = table.getData().find(s => s.id === sessionId);
        if (!session || !session.start_location || !session.end_location) {
            // Restore button if provided
            if (clickedButton && originalContent) {
                clickedButton.innerHTML = originalContent;
                clickedButton.disabled = false;
            }
            
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: '<?php echo e(__("Session data not found")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                cancelText: '',
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }
        
        const startLat = parseFloat(session.start_location.latitude);
        const startLng = parseFloat(session.start_location.longitude);
        const endLat = parseFloat(session.end_location.latitude);
        const endLng = parseFloat(session.end_location.longitude);
        
        if (isNaN(startLat) || isNaN(startLng) || isNaN(endLat) || isNaN(endLng)) {
            // Restore button if provided
            if (clickedButton && originalContent) {
                clickedButton.innerHTML = originalContent;
                clickedButton.disabled = false;
            }
            
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: '<?php echo e(__("Invalid session coordinates")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                cancelText: '',
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }
        
        // Ensure map container is visible with fade-in animation
        $('#mapCard').removeClass('d-none').hide().fadeIn(300);
        
        // Add a brief delay for visual feedback
        setTimeout(() => {
            // Center map to show both start and end points with smooth animation
            const bounds = L.latLngBounds([[startLat, startLng], [endLat, endLng]]);
            myMap.mapLet.fitBounds(bounds, { 
                padding: [30, 30],
                animate: true,
                duration: 1.2,
                easeLinearity: 0.25
            });
            
            // Add temporary markers with enhanced styling
            const startMarker = L.marker([startLat, startLng], {
                icon: L.divIcon({
                    className: 'temp-location-marker',
                    html: '<i class="fa-solid fa-play" style="color: green; font-size: 24px; text-shadow: 2px 2px 4px rgba(0,0,0,0.5);"></i>',
                    iconSize: [24, 24],
                    iconAnchor: [12, 12]
                })
            }).addTo(myMap.mapLet);
            
            const endMarker = L.marker([endLat, endLng], {
                icon: L.divIcon({
                    className: 'temp-location-marker',
                    html: '<i class="fa-solid fa-stop" style="color: red; font-size: 24px; text-shadow: 2px 2px 4px rgba(0,0,0,0.5);"></i>',
                    iconSize: [24, 24],
                    iconAnchor: [12, 12]
                })
            }).addTo(myMap.mapLet);
            
            // Add bounce animation to markers
            startMarker.getElement()?.classList.add('bounce-animation');
            endMarker.getElement()?.classList.add('bounce-animation');
            
            // Add animated route line
            const routeLine = L.polyline([
                [startLat, startLng],
                [endLat, endLng]
            ], {
                color: '#007bff',
                weight: 4,
                opacity: 0.8,
                dashArray: '10, 10'
            }).addTo(myMap.mapLet);
            
            // Animate the route line
            let dashOffset = 0;
            const animateRoute = () => {
                dashOffset = (dashOffset + 0.5) % 20;
                routeLine.setStyle({ dashOffset: dashOffset });
                requestAnimationFrame(animateRoute);
            };
            animateRoute();
            
            // Add session info popup
            const midLat = (startLat + endLat) / 2;
            const midLng = (startLng + endLng) / 2;
            
            const sessionInfo = L.popup({
                className: 'session-info-popup',
                closeButton: true,
                autoClose: false,
                closeOnClick: false,
                maxWidth: 320,
                minWidth: 280
            })
            .setLatLng([midLat, midLng])
            .setContent(`
                <div class="session-popup-content">
                    <h6><i class="ti ti-clock me-1"></i><?php echo e(__("After-Hours Session")); ?></h6>
                    <div class="row">
                        <div class="col-6">
                            <small class="text-muted"><?php echo e(__("Date")); ?>:</small><br>
                            <strong>${formatDate(session.date)}</strong>
                        </div>
                        <div class="col-6">
                            <small class="text-muted"><?php echo e(__("Duration")); ?>:</small><br>
                            <strong>${session.duration_formatted}</strong>
                        </div>
                    </div>
                    <div class="row mt-2">
                        <div class="col-6">
                            <small class="text-muted"><?php echo e(__("Distance")); ?>:</small><br>
                            <strong>${session.distance_km} km</strong>
                        </div>
                        <div class="col-6">
                            <small class="text-muted"><?php echo e(__("Time")); ?>:</small><br>
                            <strong>${formatTime(session.start_time)} - ${formatTime(session.end_time)}</strong>
                        </div>
                    </div>
                    <div class="mt-2 pt-2 border-top">
                        <small class="text-muted">
                            <i class="ti ti-map-pin me-1"></i><?php echo e(__("Green marker: Start location")); ?><br>
                            <i class="ti ti-map-pin me-1"></i><?php echo e(__("Red marker: End location")); ?>

                        </small>
                    </div>
                    <div class="mt-2 pt-2 border-top">
                        <small class="text-info">
                            <i class="ti ti-info-circle me-1"></i><?php echo e(__("Improved calculation")); ?><br>
                            <i class="ti ti-target me-1"></i>${getStopPointDescription(session)}
                        </small>
                    </div>
                </div>
            `)
            .openOn(myMap.mapLet);
            
            // Restore button after animation starts
            if (clickedButton && originalContent) {
                clickedButton.innerHTML = originalContent;
                clickedButton.disabled = false;
            }
            
            // Store references for cleanup
            const animationElements = {
                startMarker: startMarker,
                endMarker: endMarker,
                routeLine: routeLine,
                sessionInfo: sessionInfo
            };
            
            // Remove temporary elements after 10 seconds
            setTimeout(() => {
                if (myMap.mapLet && !myMap.mapLet._removed) {
                    if (animationElements.startMarker && myMap.mapLet.hasLayer(animationElements.startMarker)) {
                        myMap.mapLet.removeLayer(animationElements.startMarker);
                    }
                    if (animationElements.endMarker && myMap.mapLet.hasLayer(animationElements.endMarker)) {
                        myMap.mapLet.removeLayer(animationElements.endMarker);
                    }
                    if (animationElements.routeLine && myMap.mapLet.hasLayer(animationElements.routeLine)) {
                        myMap.mapLet.removeLayer(animationElements.routeLine);
                    }
                    if (animationElements.sessionInfo) {
                        myMap.mapLet.closePopup(animationElements.sessionInfo);
                    }
                }
            }, 10000);
            
            // Ensure map container is focused
            $('#after-hours-map').focus();
            
        }, 500); // Delay for visual feedback
    }

    function fitMapToBounds() {
        // Prevent any default scrolling behavior
        event.preventDefault();
        
        if (myMap) {
            myMap.fitBounds();
        }
        
        // Ensure map container is visible and focused
        $('#mapCard').removeClass('d-none');
        $('#after-hours-map').focus();
    }

    // Toggle fullscreen
    function toggleFullscreen() {
        $('#after-hours-map')[0].requestFullscreen();
    }

    // Go to vehicle settings page
    function goToVehicleSettings() {
        window.location.href = '/vehicles-list';
    }

    function getStopPointDescription(session) {
        if (session.end_reason === 'last_stop') {
            return __('Using last stop point');
        } else if (session.end_reason === 'first_stop') {
            return __('Using first stop point');
        } else if (session.end_reason === 'first_stop_same_location') {
            return __('Using first stop point at the same location');
        } else {
            return __('Unknown');
        }
    }
</script>
<?php $__env->stopSection(); ?>
<?php echo $__env->make('layouts.app', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?><?php /**PATH C:\xampp\htdocs\aractakipv1\resources\views/reports/after-hours.blade.php ENDPATH**/ ?>