<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Log;

class PermissionHelper
{
    /**
     * Get all available permissions grouped by module
     */
    public static function getAvailablePermissions()
    {
        return [
            'vehicles' => [
                'name' => 'Vehicles',
                'permissions' => [
                    'vehicles' => 'Vehicle Management',
                ]
            ],
            'map' => [
                'name' => 'Map',
                'permissions' => [
                    'map' => 'Vehicle Map',
                ]
            ],
            'regions' => [
                'name' => 'Regions',
                'permissions' => [
                    'regions' => 'Region Management',
                ]
            ],
            'places' => [
                'name' => 'Places',
                'permissions' => [
                    'places' => 'Place Management',
                ]
            ],
            'reports' => [
                'name' => 'Reports',
                'permissions' => [
                    'reports' => 'Reports Page',
                    'reports_daily' => 'Daily Reports',
                    'reports_parking' => 'Parking Reports',
                    'reports_route' => 'Route Reports',
                    'reports_visit' => 'Visit Reports',
                    'reports_violations' => 'Violation Reports',
                    'reports_after_hours' => 'After Hours Reports',
                    'reports_speed' => 'Speed Reports',
                    'reports_harsh_driving' => 'Harsh Driving Reports',
                    'reports_fuel' => 'Fuel Reports',
                    'reports_toll' => 'Toll Reports',
                    'reports_working_hours' => 'Working Hours Reports',
                    'reports_inspection' => 'Inspection Reports',
                    'reports_mileage' => 'Mileage Reports',
                    'reports_vehicle_health' => 'Vehicle Health Reports',
                    'reports_driver_scoring' => 'Driver Scoring Reports',
                ]
            ],
            'settings' => [
                'name' => 'Settings',
                'permissions' => [
                    'settings' => 'Settings Page',
                    'settings_general' => 'General Settings',
                    'settings_fuel' => 'Fuel Settings',
                    'settings_maintenance' => 'Maintenance Settings',
                    'settings_alerts' => 'Alert Settings',
                    'settings_system' => 'System Settings',
                ]
            ],
            'users' => [
                'name' => 'User Management',
                'permissions' => [
                    'users' => 'User Management',
                ]
            ],
        ];
    }

    /**
     * Get all permissions as flat array
     */
    public static function getAllPermissions()
    {
        $permissions = [];
        $availablePermissions = self::getAvailablePermissions();
        
        foreach ($availablePermissions as $module => $moduleData) {
            foreach ($moduleData['permissions'] as $permissionKey => $permissionName) {
                $permissions[$permissionKey] = $permissionName;
            }
        }
        
        return $permissions;
    }

    /**
     * Get permissions grouped by module for frontend
     */
    public static function getPermissionsForFrontend()
    {
        return self::getAvailablePermissions();
    }

    /**
     * Check if user has permission to access a specific report or feature
     * 
     * @param int $userType User type (1=super admin, 2=admin, 3=manager, etc.)
     * @param object|null $permissions User permissions object
     * @param string $permissionKey Permission key to check
     * @return bool
     */
    public static function hasPermission($userType, $permissions, $permissionKey, $permissionType = "access")        
    {
        // Super admin, admin, and manager have access to everything
        if ($userType <= 3) {
            return true;
        }

        //Log::info(["hasPermission", $permissionType]);

        // Check if permissions object exists and has the required permission
        if ($permissions && isset($permissions->$permissionKey)) {
            return $permissions->$permissionKey->$permissionType == 1;
        }

        return false;
    }
} 