<?php

namespace App\Helpers;
use Illuminate\Support\Facades\Log; // Log facade'ını burada import ediyoruz
use Illuminate\Http\Request;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use Exception;

class PublicHelper
{
    
    public static function DecodeJWT(Request $request)
    {
        try {
            $token = $request->cookie(config('jwt.public_key'));
            // Check if JWT_SECRET is set
            $jwtSecret = config('jwt.secret');
            //$jwtSecret = "1d81f2a1be51e82dd41b1500339ea59f474f1f3bea7921486b8f2a5545f3f126";
            if (empty($jwtSecret)) {
                Log::error('DecodeJWT > JWT_SECRET environment variable is not set');
                return null;
            }

            $decoded = JWT::decode($token, new Key($jwtSecret, 'HS256'));
            return (array) $decoded;
        } catch (Exception $e) {
            Log::error('DecodeJWT > JWT decode error: ' . $e->getMessage());
            return null;
        }
    }

    public static function DateTimeFormarter($dateTime)
    {
        if (empty($dateTime)) {
            return null;
        }
        Log::info(["dateTime" => $dateTime]);
        try {
            // Use SystemSettingsHelper for consistent formatting
            return \App\Helpers\SystemSettingsHelper::formatDateTime($dateTime);
        } catch (\Exception $e) {
            // Fallback to old method
            $locales_formats = config('app.locales_formats');
            $locale = app()->getLocale();
            $format = $locales_formats[$locale]['php_datetime'] ?? 'Y-m-d H:i:s';
            
            try {
                $date = new \DateTime($dateTime);
                return $date->format($format);
            } catch (\Exception $e) {
                return null;
            }
        }
    }
    /*
    * get session settings
    */
    public static function getSessionSettings(Request $request)
    {
        $defaultSettings = [
            'success' => true,
            'data' => [
                'locale' => 'en',
                'timezone' => 'Europe/Istanbul',
                'dateFormat' => 'Y-m-d',
                'timeFormat' => 'H:i:s',
                'currency' => 'TRY',
                'speedUnit' => 'km/h',
                'weightUnit' => 'kg',
                'liquidUnit' => 'lt',
                'lengthUnit' => 'mt',
            ]
        ];
        
        
        try{
            $token = $request->cookie(config('jwt.public_key'));

            if($token){
                $tokenModel = PublicHelper::DecodeJWT($request);
            }
            else{
                $tokenModel = null;
            }
            
            // If no token or invalid token, return default settings
            if (!$tokenModel) {
                return response()->json($defaultSettings);
            }

            //Log::info("getSessionSettings > tokenModel", [$tokenModel]);

            // Return all session settings from JWT token with defaults
            return response()->json([
                'success' => true,
                'data' => [
                    'locale' => $tokenModel['locale'] ?? 'en',
                    'timezone' => $tokenModel['timezone'] ?? 'Europe/Istanbul',
                    'dateFormat' => $tokenModel['dateFormat'] ?? 'Y-m-d',
                    'timeFormat' => $tokenModel['timeFormat'] ?? 'H:i:s',
                    'currency' => $tokenModel['currency'] ?? 'TRY',
                    'speedUnit' => $tokenModel['speedUnit'] ?? 'km/h',
                    'weightUnit' => $tokenModel['weightUnit'] ?? 'kg',
                    'liquidUnit' => $tokenModel['liquidUnit'] ?? 'lt',
                    'lengthUnit' => $tokenModel['lengthUnit'] ?? 'mt',
                ]
            ]);
        }
        catch(\Exception $e){
            Log::error('Error in getSessionSettings: ' . $e->getMessage());
            
            // Return default settings on error
            return response()->json($defaultSettings);
        }   
    }
}
