<?php

namespace App\Helpers;

use App\Models\GeneralSetting;

class SettingsHelper
{
    /**
     * Get fuel price for current merchant
     */
    public static function getFuelPrice($fuelType = 'diesel', $mid = null, $storeId = null)
    {
        if (!$mid) {
            $mid = self::getCurrentMerchantId();
        }
        $storeId = $storeId ?? config('app.store_id');
        return GeneralSetting::getFuelPrice($mid, $fuelType, $storeId);
    }

    /**
     * Set fuel price for current merchant
     */
    public static function setFuelPrice($fuelType, $price, $mid = null, $storeId = null)
    {
        if (!$mid) {
            $mid = self::getCurrentMerchantId();
        }
        $storeId = $storeId ?? config('app.store_id');
        return GeneralSetting::setFuelPrice($mid, $fuelType, $price, $storeId);
    }

    /**
     * Get any setting value
     */
    public static function get($key, $default = null, $mid = null, $storeId = null)
    {
        if (!$mid) {
            $mid = self::getCurrentMerchantId();
        }
        $storeId = $storeId ?? config('app.store_id');
        return GeneralSetting::getValue($mid, $key, $default, $storeId);
    }

    /**
     * Set any setting value
     */
    public static function set($key, $value, $type = 'string', $category = 'general', $description = null, $mid = null, $storeId = null)
    {
        if (!$mid) {
            $mid = self::getCurrentMerchantId();
        }
        $storeId = $storeId ?? config('app.store_id');
        return GeneralSetting::setValue($mid, $key, $value, $type, $category, $description, $storeId);
    }

    /**
     * Get settings by category
     */
    public static function getByCategory($category, $mid = null, $storeId = null)
    {
        if (!$mid) {
            $mid = self::getCurrentMerchantId();
        }
        $storeId = $storeId ?? self::getCurrentStoreId();
        return GeneralSetting::getByCategory($mid, $category, $storeId);
    }

    /**
     * Get current merchant ID from session or token
     */
    private static function getCurrentMerchantId()
    {
        // Try to get from session first
        if (session()->has('user_id')) {
            return session('user_id');
        }

        // Try to get from JWT token if available
        if (request()->hasCookie(config('jwt.public_key'))) {
            try {
                $tokenModel = PublicHelper::DecodeJWT(request());
                return $tokenModel['userId'] ?? 1;
            } catch (\Exception $e) {
                return 1; // Default merchant ID
            }
        }

        return 1; // Default merchant ID
    }

    /**
     * Get current store ID from JWT token
     */
    private static function getCurrentStoreId()
    {
        // Try to get from JWT token if available
        if (request()->hasCookie(config('jwt.public_key'))) {
            try {
                $tokenModel = PublicHelper::DecodeJWT(request());
                return $tokenModel['storeId'] ?? config('app.store_id');
            } catch (\Exception $e) {
                return config('app.store_id'); // Default store ID
            }
        }

        return config('app.store_id'); // Default store ID
    }

    /**
     * Get all fuel prices
     */
    public static function getAllFuelPrices($mid = null, $storeId = null)
    {
        if (!$mid) {
            $mid = self::getCurrentMerchantId();
        }
        $storeId = $storeId ?? self::getCurrentStoreId();
        return GeneralSetting::getAllFuelPrices($mid, $storeId);
    }

    /**
     * Get maintenance settings
     */
    public static function getMaintenanceSettings($mid = null, $storeId = null)
    {
        if (!$mid) {
            $mid = self::getCurrentMerchantId();
        }
        $storeId = $storeId ?? self::getCurrentStoreId();
        return GeneralSetting::getMaintenanceSettings($mid, $storeId);
    }

    /**
     * Get alert settings
     */
    public static function getAlertSettings($mid = null, $storeId = null)
    {
        if (!$mid) {
            $mid = self::getCurrentMerchantId();
        }
        $storeId = $storeId ?? self::getCurrentStoreId();
        return GeneralSetting::getAlertSettings($mid, $storeId);
    }

    /**
     * Get system settings
     */
    public static function getSystemSettings($mid = null, $storeId = null)
    {
        if (!$mid) {
            $mid = self::getCurrentMerchantId();
        }
        $storeId = $storeId ?? self::getCurrentStoreId();
        return GeneralSetting::getSystemSettings($mid, $storeId);
    }

    /**
     * Calculate fuel cost based on consumption and fuel type
     */
    public static function calculateFuelCost($consumption, $fuelType = 'diesel', $mid = null, $storeId = null)
    {
        $fuelPrice = self::getFuelPrice($fuelType, $mid, $storeId);
        return $consumption * $fuelPrice;
    }

    /**
     * Get speed limit setting
     */
    public static function getSpeedLimit($mid = null, $storeId = null)
    {
        return self::get('speed_limit_default', 90, $mid, $storeId);
    }

    /**
     * Check if alerts are enabled
     */
    public static function isAlertEnabled($alertType, $mid = null, $storeId = null)
    {
        $key = $alertType . '_alerts_enabled';
        return self::get($key, false, $mid, $storeId);
    }

    /**
     * Get maintenance reminder days
     */
    public static function getMaintenanceReminderDays($mid = null, $storeId = null)
    {
        return self::get('maintenance_reminder_days', 30, $mid, $storeId);
    }

    /**
     * Get oil change interval
     */
    public static function getOilChangeInterval($mid = null, $storeId = null)
    {
        return self::get('oil_change_km', 10000, $mid, $storeId);
    }

    /**
     * Get tire change interval
     */
    public static function getTireChangeInterval($mid = null, $storeId = null)
    {
        return self::get('tire_change_km', 50000, $mid, $storeId);
    }
} 