<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Models\Place;
use App\Models\PlaceDevice;
use App\Http\Resources\PlaceResource;
use App\Traits\ApiResponse;
use App\Helpers\Base64Helper;
use App\Helpers\PublicHelper;

class PlaceController extends Controller
{
    use ApiResponse;

    /**
     * Get user ID from JWT token in the request cookies
     */
    private function getUserIdFromToken(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        return $tokenModel['userId'] ?? null;
    }

    /**
     * List all places, optionally with device match info
     */
    public function list(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        if (is_array($data) && isset($data['did'])) {
            $did = $data['did'];

            // Place ve eşleşen cihazları left join ile çek
            $places = Place::where('vts_places.sts', 1)
                ->where('vts_places.mid', $mid)
                ->leftJoin('vts_place_devices as pdm', function($join) use ($did) {
                    $join->on('vts_places.id', '=', 'pdm.pid')
                        ->where('pdm.did', '=', $did);
                })
                ->orderBy('vts_places.ord')
                ->select('vts_places.*', 'pdm.id as match_id', 'pdm.did as match_did', 'pdm.bid as match_status_id')
                ->get();

            return response()->json([
                'success' => true,
                'data' => PlaceResource::collection($places),
                'message' => $places->isEmpty() ? 'No places found.' : 'Places with device match info listed.'
            ]);
        } else {
            $places = Place::where('mid', $mid)
                ->orderBy('ord')
                ->get();
        }

        return response()->json([
            'success' => true,
            'data' => PlaceResource::collection($places),
        ]);
    }

    /**
     * Add a new place
     */
    public function add(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        // Use Base64 decoded data if present, otherwise use the request data directly
        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        // Validate input data
        $validator = \Illuminate\Support\Facades\Validator::make($data, [
            'name' => 'required|string|max:255',
            'order' => 'nullable|integer',
            'device_count' => 'nullable|integer',
            'status_id' => 'nullable|integer',
            'latitude' => 'nullable|string|max:50',
            'longitude' => 'nullable|string|max:50',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Auto-increment order if not provided
            if (is_null($data['order'] ?? null)) {
                $maxOrd = Place::where('mid', $mid)->max('ord');
                $ord = is_null($maxOrd) ? 1 : $maxOrd + 1;
            } else {
                $ord = $data['order'];
            }

            $place = Place::create([
                'mid' => $mid,
                'nm' => $data['name'],
                'ord' => $ord,
                'pcnt' => $data['device_count'] ?? 0,
                'sts' => $data['status_id'] ?? 1,
                'lat' => $data['latitude'] ?? '',
                'lng' => $data['longitude'] ?? '',
                'radius' => $data['radius'] ?? 0,
                'dt' => now(),
                'created_by' => $mid,
                'created_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'data' => new PlaceResource($place),
                'message' => 'Place added successfully.'
            ]);

        } catch (\Exception $e) {
            Log::error('Error creating place: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error creating place.'
            ], 500);
        }
    }

    /**
     * Edit an existing place
     */
    public function edit(Request $request)
    {
        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $id = $data['id'] ?? null;
        $mid = $this->getUserIdFromToken($request);

        $place = Place::find($id);
        if (!$place) {
            return $this->error('Place not found', 404);
        }

        // Check ownership
        if ($place->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        // Update fields
        $place->nm = $data['name'] ?? $place->nm;
        $place->ord = $data['order'] ?? $place->ord;
        $place->pcnt = $data['device_count'] ?? $place->pcnt;
        $place->sts = $data['status_id'] ?? $place->sts;
        $place->lat = $data['latitude'] ?? $place->lat;
        $place->lng = $data['longitude'] ?? $place->lng;
        $place->dt = now();
        $place->updated_by = $mid;
        $place->updated_at = now();

        $place->save();

        return response()->json([
            'success' => true,
            'data' => new PlaceResource($place),
            'message' => 'Place updated successfully.'
        ]);
    }

    /**
     * Delete a place
     */
    public function delete(Request $request)
    {
        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $id = $data['id'] ?? null;
        $mid = $this->getUserIdFromToken($request);

        $place = Place::find($id);
        if (!$place) {
            return $this->error('Place not found', 404);
        }

        // Check ownership
        if ($place->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        $place->delete();

        // PlaceDevice vts_place_devices match delete 
        PlaceDevice::where('pid', $id)->delete();

        // Reorder remaining places
        $this->reorderPlacesAfterDelete($mid);

        return response()->json([
            'success' => true,
            'message' => 'Place deleted successfully.'
        ]);
    }

    /**
     * Update places order
     */
    public function updateOrder(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $orders = $data['orders'] ?? $data['order'] ?? [];

        foreach ($orders as $orderData) {
            $place = Place::find($orderData['id']);

            // Check ownership
            if ($place && $place->mid == $mid) {
                $place->update([
                    'ord' => $orderData['order'] ?? $orderData['ord'],
                    'dt' => now(),
                    'updated_by' => $mid,
                    'updated_at' => now()
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Place order updated successfully.'
        ]);
    }

    /**
     * Duplicate a place
     */
    public function duplicate(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        // Validation
        $validator = \Illuminate\Support\Facades\Validator::make($data, [
            'id' => 'required|integer|exists:vts_places,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $originalPlace = Place::find($data['id']);

            if (!$originalPlace) {
                return response()->json([
                    'success' => false,
                    'message' => 'Place not found'
                ], 404);
            }

            // Check ownership
            if ($originalPlace->mid != $mid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 403);
            }

            // Get next order number
            $maxOrd = Place::where('mid', $mid)->max('ord');
            $newOrd = is_null($maxOrd) ? 1 : $maxOrd + 1;

            // Create duplicated place with " (Copy)" suffix
            $duplicatedPlace = Place::create([
                'mid' => $mid,
                'nm' => $originalPlace->nm . ' (Copy)',
                'ord' => $newOrd,
                'pcnt' => $originalPlace->pcnt,
                'sts' => $originalPlace->sts,
                'lat' => $originalPlace->lat,
                'lng' => $originalPlace->lng,
                'radius' => $originalPlace->radius,
                'dt' => now(),
                'created_by' => $mid,
                'created_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'data' => new PlaceResource($duplicatedPlace),
                'message' => 'Place duplicated successfully.'
            ]);

        } catch (\Exception $e) {
            \Log::error('Error duplicating place: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error duplicating place.'
            ], 500);
        }
    }

    /**
     * Toggle place status (active/inactive)
     */
    public function toggleStatus(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $id = $data['id'];
        $status = $data['status_id'] ?? $data['sts'];

        $place = Place::find($id);
        if (!$place) {
            return $this->error('Place not found', 404);
        }

        // Check ownership
        if ($place->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        $place->update([
            'sts' => $status,
            'dt' => now(),
            'updated_by' => $mid,
            'updated_at' => now()
        ]);

        $statusText = $status == 1 ? 'activated' : 'deactivated';

        return response()->json([
            'success' => true,
            'data' => new PlaceResource($place),
            'message' => "Place {$statusText} successfully."
        ]);
    }

    /**
     * Save place coordinates (latitude and longitude)
     */
    public function saveCoordinates(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $placeId = $data['id'];
        $latitude = $data['latitude'] ?? null;
        $longitude = $data['longitude'] ?? null;
        $radius = $data['radius'] ?? 0;

        $place = Place::find($placeId);
        if (!$place) {
            return $this->error('Place not found', 404);
        }

        // Check ownership
        if ($place->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        $place->lat = $latitude;
        $place->lng = $longitude;
        $place->radius = $radius;
        $place->dt = now();
        $place->updated_by = $mid;
        $place->updated_at = now();
        $place->save();

        return response()->json([
            'success' => true,
            'data' => new PlaceResource($place),
            'message' => 'Coordinates saved successfully.'
        ]);
    }

    /**
     * Set device for place (add or update device-place match)
     */
    public function setDevice(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        // Base64 decode the incoming data if present, otherwise use the request data directly
        $data = $request->has('data') ? Base64Helper::decode($request->input('data')) : $request->all();

        //Log::info('PlaceController>setDevice: ', ['data' => $data]);

        $placeId = $data['place_id'] ?? $data['pid'] ?? null;
        $deviceId = $data['device_id'] ?? $data['did'] ?? null;
        $bid = $data['status'] ?? 0;
        $gsm = $data['gsm'] ?? '';
        $email = $data['email'] ?? '';

        if (!$placeId || !$deviceId) {
            return $this->error('Place ID and Device ID are required.', 422);
        }

        // Find or create the matching record for place and device
        $match = PlaceDevice::firstOrNew([
            'mid' => $mid,
            'pid' => $placeId,
            'did' => $deviceId,
        ]);

        $match->bid = $bid;
        $match->gsm = $gsm;
        $match->email = $email;
        $match->mid = $mid;

        // If created, set created_by and created_at
        if (!$match->exists) {
            $match->created_by = $mid;
            $match->created_at = now();
        } else {
            // If updating, set updated_by and updated_at
            $match->updated_by = $mid;
            $match->updated_at = now();
        }

        $match->save();

        return response()->json([
            'success' => true,
            'data' => [],
            'message' => 'Device match for place updated successfully.'
        ]);
    }

    /**
     * Reorder places after deletion
     */
    private function reorderPlacesAfterDelete($mid)
    {
        $places = Place::where('mid', $mid)
            ->orderBy('ord', 'asc')
            ->get();

        foreach ($places as $index => $place) {
            $place->update(['ord' => $index + 1]);
        }
    }
}