<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\Region;
use App\Models\RegionDeviceMatch;
use App\Http\Resources\RegionResource;
use App\Traits\ApiResponse;
use App\Helpers\Base64Helper;
use App\Helpers\PublicHelper;

class RegionController extends Controller
{
    use ApiResponse;

    private function getUserIdFromToken(Request $request)
    {
        $tokenModel = PublicHelper::DecodeJWT($request);
        return $tokenModel['userId'] ?? null;
    }

    /**
     * List all regions (existing method - kept as is)
     */
    public function list(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

         $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        //Log::info('RegionController>list: ', ['data' => $data]);

        if (is_array($data) && isset($data['did'])) {
            $did = $data['did'];

            // Region ve eşleşen cihazları left join ile çek
            $regions = Region::where('vts_regions.sts', 1)
                ->where('vts_regions.mid', $mid)
                ->leftJoin('vts_region_devices as rdm', function($join) use ($did) {
                    $join->on('vts_regions.id', '=', 'rdm.pid')
                        ->where('rdm.did', '=', $did);
                })
                ->orderBy('vts_regions.ord')
                ->select('vts_regions.*', 'rdm.zone_type as match_status_id', 'rdm.did as match_did', 'rdm.price as price')
                ->get();

            return response()->json([
                'success' => true,
                'data' => RegionResource::collection($regions),
                'message' => $regions->isEmpty() ? 'No regions found.' : 'Regions with device match info listed.'
            ]);
        }
        else{

            $regions = Region::where('sts', 1)
                ->where('mid', $mid)
                ->orderBy('ord')
                ->get();
        }
        
        return response()->json([
            'success' => true,
            'data' => RegionResource::collection($regions),
        ]);
    }

     /**
     * Duplicate region
     */
    public function duplicate(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        // Base64 decode edilmiş data
        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        Log::info('RegionController>duplicate: ', ['data' => $data]);

        // Validation
        $validator = \Illuminate\Support\Facades\Validator::make($data, [
            'id' => 'required|integer|exists:vts_regions,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $originalRegion = Region::find($data['id']);

            if (!$originalRegion) {
                return response()->json([
                    'success' => false,
                    'message' => 'Region not found'
                ], 404);
            }

            // Check ownership
            if ($originalRegion->mid != $mid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized'
                ], 403);
            }

            // Get next order number
            $maxOrd = Region::where('mid', $mid)->max('ord');
            $newOrd = is_null($maxOrd) ? 1 : $maxOrd + 1;

            // Create duplicated region with " (Kopya)" suffix
            $duplicatedRegion = Region::create([
                'mid' => $mid,
                'nm' => $originalRegion->nm . ' (Kopya)',
                'ord' => $newOrd,
                'pcnt' => $originalRegion->pcnt,
                'sts' => $originalRegion->sts,
                'active' => $originalRegion->active,
                'ply' => $originalRegion->ply, // Polygon JSON'ı da kopyala
                'poly' => $originalRegion->poly, // WKT polygon'ı da kopyala
                'created_at' => now(),
                'created_by' => $mid,
            ]);

            return response()->json([
                'success' => true,
                'data' => new RegionResource($duplicatedRegion),
                'message' => 'Region duplicated successfully.'
            ]);

        } catch (\Exception $e) {
            Log::error('Error duplicating region: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error duplicating region.'
            ], 500);
        }
    }

    /**
     * Add new region (updated existing method)
     */
    public function add(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        // Base64 decode edilmiş data varsa onu kullan, yoksa direkt request'i
        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        // Validation için Validator kullan (data array'i üzerinde)
        $validator = \Illuminate\Support\Facades\Validator::make($data, [
            'name' => 'required|string|max:255',
            'order' => 'nullable|integer',
            'count' => 'nullable|integer',
            'status' => 'nullable|integer',
            'active' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Otomatik sıralama
            if (is_null($data['order'] ?? null)) {
                $maxOrd = Region::where('mid', $mid)->max('ord');
                $ord = is_null($maxOrd) ? 1 : $maxOrd + 1;
            } else {
                $ord = $data['order'];
            }

            $region = Region::create([
                'mid' => $mid,
                'nm' => $data['name'],
                'ord' => $ord,
                'pcnt' => $data['count'] ?? 0,
                'sts' => $data['status'] ?? 1,
                'active' => $data['active'] ?? 1,
                'created_at' => now(),
                'created_by' => $mid,
            ]);

            return response()->json([
                'success' => true,
                'data' => new RegionResource($region),
                'message' => 'Region added successfully.'
            ]);

        } catch (\Exception $e) {
            Log::error('Error creating region: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error creating region.'
            ], 500);
        }
    }

    /**
     * Edit region (existing method - kept compatible)
     */
    public function edit(Request $request)
    {
        $decodedData = Base64Helper::decode($request->input('data'));
        $id = $decodedData['id'];

        $region = Region::find($id);
        if (!$region) {
            return $this->error('Region not found', 404);
        }

        //Log::info('RegionController>edit: ', ['region' => $region, 'data' => $decodedData]);

        // Update fields
        $region->nm = $decodedData['name'] ?? $decodedData['nm'] ?? $region->nm;
        $region->ord = $decodedData['order'] ?? $decodedData['ord'] ?? $region->ord;
        $region->active = $decodedData['active'] ?? $region->active;
        $region->updated_at = now();
        $region->updated_by = $this->getUserIdFromToken($request);

        $region->save();

        return response()->json([
            'success' => true,
            'message' => 'Region updated successfully.'
        ]);
    }

    /**
     * Delete region (updated existing method)
     */
    public function delete(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        // Base64 decode edilmiş data varsa onu kullan
        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        //Log::info('RegionController>delete: ', ['data' => $data]);

        $id = $data['id'] ?? $request->id;
        $region = Region::find($id);

        if (!$region) {
            return $this->error('Region not found', 404);
        }

        // Check ownership
        if ($region->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        $region->delete();

        // Reorder remaining regions
        $this->reorderRegionsAfterDelete($mid);

        return response()->json([
            'success' => true,
            'message' => 'Region deleted successfully.'
        ]);
    }

    /**
     * Toggle region status
     */
    public function toggleStatus(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $id = $data['id'];
        $status = $data['status'] ?? $data['active'];

        $region = Region::find($id);
        if (!$region) {
            return $this->error('Region not found', 404);
        }

        // Check ownership
        if ($region->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        $region->update([
            'active' => $status,
            'updated_at' => now(),
            'updated_by' => $mid,
        ]);

        $statusText = $status == 1 ? 'activated' : 'deactivated';

        return response()->json([
            'success' => true,
            'data' => new RegionResource($region),
            'message' => "Region {$statusText} successfully."
        ]);
    }

    /**
     * Update regions order (updated existing method)
     */
    public function updateOrder(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        // Base64 decode edilmiş data
        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $orders = $data['orders'] ?? $data['order'] ?? [];

        foreach ($orders as $orderData) {
            $region = Region::find($orderData['id']);

            // Check ownership
            if ($region && $region->mid == $mid) {
                $region->update([
                    'ord' => $orderData['order'] ?? $orderData['ord'],
                    'updated_at' => now(),
                    'updated_by' => $mid,
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Region order updated successfully.'
        ]);
    }

    /**
     * Save region polygon
     */
    public function savePolygon(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        $data = $request->has('data') ?
            Base64Helper::decode($request->input('data')) :
            $request->all();

        $regionId = $data['id'];
        $coordinates = $data['coordinates']; // Frontend'den gelen koordinatlar

        $region = Region::find($regionId);
        if (!$region) {
            return $this->error('Region not found', 404);
        }

        // Check ownership
        if ($region->mid != $mid) {
            return $this->error('Unauthorized', 403);
        }

        // Koordinatları MySQL POLYGON formatına çevir
        $polygonString = $this->coordinatesToPolygonString($coordinates);

        try {
            if ($polygonString) {
                // Geçerli polygon varsa POLYGON formatında kaydet
                DB::statement("
                    UPDATE vts_regions
                    SET poly = ST_PolygonFromText(?),
                        ply = ?,
                        updated_at = ?,
                        updated_by = ?
                    WHERE id = ?
                ", [
                    $polygonString,
                    json_encode($coordinates),
                    now(),
                    $mid,
                    $regionId
                ]);
            } else {
                // Geçersiz koordinatlar varsa poly alanını NULL yap
                DB::statement("
                    UPDATE vts_regions
                    SET poly = NULL,
                        ply = ?,
                        updated_at = ?,
                        updated_by = ?
                    WHERE id = ?
                ", [
                    json_encode($coordinates),
                    now(),
                    $mid,
                    $regionId
                ]);
            }

            // Region'u tekrar yükle
            $region->refresh();

            return response()->json([
                'success' => true,
                'data' => new RegionResource($region),
                'message' => 'Polygon saved successfully.'
            ]);

        } catch (\Exception $e) {
            \Log::error('RegionController>savePolygon Error: ', [
                'error' => $e->getMessage(),
                'regionId' => $regionId,
                'coordinates' => $coordinates,
                'polygonString' => $polygonString ?? 'NULL'
            ]);

            return $this->error('Failed to save polygon: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Convert coordinates array to MySQL POLYGON string format
     */
    private function coordinatesToPolygonString($coordinates)
    {
        try {
            // Koordinatların geçerli olduğunu kontrol et
            if (!is_array($coordinates) || count($coordinates) < 3) {
                return false;
            }

            // Her koordinatın [lat, lng] formatında olduğunu kontrol et
            foreach ($coordinates as $coord) {
                if (!is_array($coord) || count($coord) !== 2) {
                    return false;
                }
                if (!is_numeric($coord[0]) || !is_numeric($coord[1])) {
                    return false;
                }
            }

            // İlk ve son koordinatın aynı olup olmadığını kontrol et (polygon kapalı olmalı)
            $firstCoord = $coordinates[0];
            $lastCoord = $coordinates[count($coordinates) - 1];

            if ($firstCoord[0] != $lastCoord[0] || $firstCoord[1] != $lastCoord[1]) {
                // Polygon'u kapatmak için ilk koordinatı sona ekle
                $coordinates[] = $firstCoord;
            }

            // MySQL POLYGON formatına çevir: POLYGON((lng1 lat1, lng2 lat2, ...))
            // NOT: MySQL'de lng(x) lat(y) sırası kullanılır
            $polygonPoints = [];
            foreach ($coordinates as $coord) {
                $lat = $coord[0];
                $lng = $coord[1];
                $polygonPoints[] = "$lng $lat";
            }

            $polygonString = 'POLYGON((' . implode(', ', $polygonPoints) . '))';

            return $polygonString;

        } catch (\Exception $e) {
            \Log::error('coordinatesToPolygonString Error: ', [
                'error' => $e->getMessage(),
                'coordinates' => $coordinates
            ]);
            return false;
        }
    }

    /**
     * Reorder regions after deletion
     */
    private function reorderRegionsAfterDelete($mid)
    {
        $regions = Region::where('mid', $mid)
            ->orderBy('ord', 'asc')
            ->get();

        foreach ($regions as $index => $region) {
            $region->update(['ord' => $index + 1]);
        }
    }

    /**
     * Set device for region
     */
    public function setDevice(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }

        //Log::info('RegionController>setDevice: ', ['request' => $request]);
        
        // Base64 decode the incoming data if present, otherwise use the request data directly
        $data = $request->has('data') ? Base64Helper::decode($request->input('data')) : $request->all();

        //Log::info('RegionController>setDevice: ', ['data' => $data]);

        $regionId = $data['region_id'] ?? null;
        $status = $data['status'] ?? null;
        $deviceId = $data['device_id'] ?? $request->input('device_id'); // Optionally get device id from request

        if (!$regionId || $status === null) {
            return response()->json([
                'success' => false,
                'message' => 'Region ID and status are required.'
            ]);
        }

        // If device id is required, check for its presence
        if (!$deviceId) {
            return response()->json([
                'success' => false,
                'message' => 'Device ID is required.'
            ]);
        }

        // Find or create the matching record for region and device
        $match = RegionDeviceMatch::firstOrNew([
            'mid' => $mid,
            'pid' => $regionId,
            'did' => $deviceId,
        ]);

        //Log::info('RegionController>setDevice: ', ['match' => $match]);

        $match->zone_type = $status;
        $match->mid = $mid;

        if($status == 0){
            $match->bid = 0; // notification status
        }

        //if created, set the created_by and created_at
        if (!$match->exists) {
            $match->created_by = $mid;
            $match->created_at = now();
        }
        else {
            // If updating, set updated_by and updated_at
            $match->updated_by = $mid;
            $match->updated_at = now();
        }

        $match->save();

        return response()->json([
                'success' => true,
                'data' => [],
                'message' => 'Device match status updated successfully.'
            ]);
    }

    /**
     * Set device price for region
     */
    public function setDevicePrice(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return $this->error('Unauthorized', 401);
        }
        
        // Base64 decode the incoming data if present, otherwise use the request data directly
        $data = $request->has('data') ? Base64Helper::decode($request->input('data')) : $request->all();

        $regionId = $data['region_id'] ?? null;
        $price = $data['price'] ?? null;
        $deviceId = $data['device_id'] ?? $request->input('device_id');

        if (!$regionId) {
            return response()->json([
                'success' => false,
                'message' => 'Region ID is required.'
            ]);
        }

        // If device id is required, check for its presence
        if (!$deviceId) {
            return response()->json([
                'success' => false,
                'message' => 'Device ID is required.'
            ]);
        }

        // Validate price
        if ($price !== null && $price !== '') {
            if (!is_numeric($price) || $price < 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Price must be a positive number.'
                ]);
            }
        }

        // Find or create the matching record for region and device
        $match = RegionDeviceMatch::firstOrNew([
            'mid' => $mid,
            'pid' => $regionId,
            'did' => $deviceId,
        ]);

        $match->price = $price === '' ? null : $price;
        $match->mid = $mid;

        //if created, set the created_by and created_at
        if (!$match->exists) {
            $match->created_by = $mid;
            $match->created_at = now();
        }
        else {
            // If updating, set updated_by and updated_at
            $match->updated_by = $mid;
            $match->updated_at = now();
        }

        $match->save();

        return response()->json([
                'success' => true,
                'data' => [],
                'message' => 'Device match price updated successfully.'
            ]);
    }

    /**
     * Get all regions and all related vehicles with their prices
     */
    public function getAllRegionVehiclePrices(Request $request)
    {
        $mid = $this->getUserIdFromToken($request);
        if (!$mid) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        // Join regions, vts_region_devices, and devices
        $results = \DB::table('vts_regions as region')
            ->join('vts_region_devices as rdm', function($join) use ($mid) {
                $join->on('region.id', '=', 'rdm.pid');
            })
            ->join('vts_devices as device', function($join) use ($mid) {
                $join->on('rdm.did', '=', 'device.did')
                     ->where('device.mid', '=', $mid)
                     ->where('device.sts', '=', 3);
            })
            ->where('region.mid', $mid)
            ->where('region.sts', 1)
            ->select(
                'region.id as region_id',
                'region.nm as region_name',
                'region.ply as polygon_json',
                'device.did as vehicle_id',
                'device.nm as vehicle_name',
                'rdm.price as price'
            )
            ->orderBy('region.ord')
            ->orderBy('device.nm')
            ->get();

        // Group by region
        $grouped = [];
        foreach ($results as $row) {
            $grouped[$row->region_id]['region_id'] = $row->region_id;
            $grouped[$row->region_id]['region_name'] = $row->region_name;
            $grouped[$row->region_id]['polygon_json'] = $row->polygon_json;
            $grouped[$row->region_id]['vehicles'][] = [
                'vehicle_id' => $row->vehicle_id,
                'vehicle_name' => $row->vehicle_name,
                'price' => $row->price
            ];
        }

        return response()->json([
            'success' => true,
            'data' => array_values($grouped)
        ]);
    }
}
