<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

     /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'vts_users'; // Tablonun adını burada belirtiyoruz.

    /**
     * The column name for the email address (database uses 'ml')
     *
     * @return string
     */
    public function getEmailForPasswordReset()
    {
        return $this->ml;
    }

    /**
     * Get the name of the unique identifier for the user.
     * Override to use 'ml' column instead of 'email'
     *
     * @return string
     */
    public function getAuthIdentifierName()
    {
        return 'ml';
    }

    /**
     * Get the column name for the "username" (email in this case)
     * Used by authentication
     *
     * @return string
     */
    public function username()
    {
        return 'ml';
    }

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'nm',
        'ml',  // Email kolonu database'de 'ml' olarak
        'psw', // Password kolonu database'de 'psw' olarak
        'mid',
        'sid',
        'typ',
        'img_url',
        'gsm',
        'extra_data_json',
        'msnd',
        'ky',
        'created_at',
        'created_by',
        'updated_at',
        'updated_by',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'psw' => 'hashed',  // Database'de password kolonu 'psw'
            'created_at' => 'datetime',
            'updated_at' => 'datetime',
            'extra_data_json' => 'array',
        ];
    }
    
    /**
     * Get the password for authentication (database uses 'psw')
     */
    public function getAuthPassword()
    {
        return $this->psw;
    }

    /**
     * Accessor: Map 'email' attribute to 'ml' column
     */
    public function getEmailAttribute()
    {
        return $this->attributes['ml'] ?? null;
    }

    /**
     * Mutator: Map 'email' attribute to 'ml' column
     */
    public function setEmailAttribute($value)
    {
        $this->attributes['ml'] = $value;
    }

    /**
     * Get the direct permissions for this user
     */
    public function directPermissions()
    {
        return $this->hasMany(UserPermission::class);
    }

    /**
     * Check if user has direct permission for a specific action
     */
    public function hasDirectPermission($permissionName, $action = 'access')
    {
        $permission = $this->directPermissions()
            ->where('permission_name', $permissionName)
            ->first();

        if (!$permission) {
            return false;
        }

        switch ($action) {
            case 'access':
                return (bool) $permission->can_access;
            case 'view':
                return (bool) $permission->can_view;
            case 'edit':
                return (bool) $permission->can_edit;
            case 'delete':
                return (bool) $permission->can_delete;
            default:
                return false;
        }
    }

    /**
     * Check if user has any direct permission for a specific permission
     */
    public function hasAnyDirectPermission($permissionName)
    {
        return $this->directPermissions()
            ->where('permission_name', $permissionName)
            ->where(function ($query) {
                $query->where('can_access', true)
                      ->orWhere('can_view', true)
                      ->orWhere('can_edit', true)
                      ->orWhere('can_delete', true);
            })
            ->exists();
    }

    /**
     * Sync direct permissions for user
     */
    public function syncDirectPermissions($permissions)
    {
        // Delete existing permissions
        $this->directPermissions()->delete();
        
        //\Log::info('User>syncDirectPermissions: ', ['permissions' => $permissions]);
        // Create new permissions
        foreach ($permissions as $permissionName => $actions) {
            $this->directPermissions()->create([
                'permission_name' => $permissionName,
                'can_access' => $actions['access'] == 'true' ? 1 : 0,
                'can_view' => $actions['view'] == 'true' ? 1 : 0,
                'can_edit' => $actions['edit'] == 'true' ? 1 : 0,
                'can_delete' => $actions['delete'] == 'true' ? 1 : 0,
            ]);
        }
    }

    /**
     * Get all direct permissions as array
     */
    public function getDirectPermissionsArray()
    {
        return $this->directPermissions()->get()->mapWithKeys(function ($permission) {
            return [$permission->permission_name => [
                'access' => (bool) $permission->can_access,
                'view' => (bool) $permission->can_view,
                'edit' => (bool) $permission->can_edit,
                'delete' => (bool) $permission->can_delete,
            ]];
        })->toArray();
    }

    /**
     * Check if user has access to a module
     */
    public function hasModuleAccess($moduleName)
    {
        $availablePermissions = \App\Helpers\PermissionHelper::getAvailablePermissions();
        
        if (!isset($availablePermissions[$moduleName])) {
            return false;
        }
        
        $modulePermissions = $availablePermissions[$moduleName]['permissions'];
        
        foreach ($modulePermissions as $permissionKey => $permissionName) {
            if ($this->hasDirectPermission($permissionKey, 'access')) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Get module-level permissions for menu display
     */
    public function getModulePermissions()
    {
        $availablePermissions = \App\Helpers\PermissionHelper::getAvailablePermissions();
        $modulePermissions = [];
        
        foreach ($availablePermissions as $moduleKey => $moduleData) {
            $modulePermissions[$moduleKey] = [
                'access' => $this->hasModuleAccess($moduleKey) ? 1 : 0,
                'view' => 0,
                'edit' => 0,
                'delete' => 0,
            ];
        }
        
        return $modulePermissions;
    }
}
