<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Blade;
use Illuminate\Cookie\Middleware\EncryptCookies;
use App\View\Components\Button;
use App\View\Components\Card;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Set locale from multiple sources (priority order):
        // 1. Cookie (set by frontend from localStorage)
        // 2. JWT token
        // 3. Session
        // 4. Config default
        $locale = request()->cookie('app_locale');
        
        if (!$locale) {
            try {
                $locale = \App\Helpers\SystemSettingsHelper::getLocale();
            } catch (\Exception $e) {
                // Database might not be available during config cache
                $locale = null;
            }
        }
        
        if (!$locale) {
            $locale = Session::get('locale', config('app.locale'));
        }
        
        App::setLocale($locale);
        EncryptCookies::except([config('jwt.public_key'), 'app_locale']);

        // Register Blade components
        Blade::component('x-button', \App\View\Components\Button::class);
        Blade::component('x-card', \App\View\Components\Card::class);
        Blade::component('x-alert', \App\View\Components\Alert::class);
        Blade::component('x-form', \App\View\Components\Form::class);
        Blade::component('x-modal', \App\View\Components\Modal::class);

        // Register Blade directives for permissions
        $this->registerBladeDirectives();
    }

    /**
     * Register Blade directives for permission checking
     */
    private function registerBladeDirectives(): void
    {
        // @permission directive
        Blade::directive('permission', function ($expression) {
            return "<?php if(in_array($expression, \$cookiePermissions ?? [])): ?>";
        });

        Blade::directive('endpermission', function () {
            return "<?php endif; ?>";
        });

        // @anypermission directive
        Blade::directive('anypermission', function ($expression) {
            return "<?php if(array_intersect(explode('|', $expression), \$cookiePermissions ?? [])): ?>";
        });

        Blade::directive('endanypermission', function () {
            return "<?php endif; ?>";
        });

        // @canaccess directive for checking specific permissions
        Blade::directive('canaccess', function ($expression) {
            return "<?php if(\\App\\Helpers\\PermissionHelper::hasPermission(\$cookieUserType ?? 999, \$cookiePermissions ?? null, $expression)): ?>";
        });

        Blade::directive('endcanaccess', function () {
            return "<?php endif; ?>";
        });


    }
}
