<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class DemoDataSeeder extends Seeder
{
    /**
     * Seed demo data from SQL dump file.
     * 
     * This imports the complete database dump including:
     * - Demo users
     * - Vehicles
     * - Categories
     * - Regions and Places
     * - Device logs
     * - Settings
     */
    public function run(): void
    {
        $this->command->info('🚀 Importing demo data from SQL dump...');
        
        // Path to SQL file
        $sqlPath = database_path('sql/vtc_database_all.sql');
        
        if (!File::exists($sqlPath)) {
            $this->command->error('❌ SQL file not found at: ' . $sqlPath);
            $this->command->warn('Please ensure vtc_database_all.sql exists in database/sql/');
            return;
        }
        
        // Read SQL file
        $sql = File::get($sqlPath);
        
        // Remove SQL comments and control statements
        $sql = preg_replace('/^--.*$/m', '', $sql);
        $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);
        $sql = preg_replace('/^\s*\/\*!.*?\*\/\s*;?\s*$/m', '', $sql);
        
        // Split by semicolon and clean up
        $statements = explode(';', $sql);
        
        $imported = 0;
        $skipped = 0;
        $errors = 0;
        
        foreach ($statements as $statement) {
            $statement = trim($statement);
            
            // Skip empty statements and MySQL control statements
            if (empty($statement) || 
                stripos($statement, '/*!') === 0 ||
                stripos($statement, 'SET @') === 0 ||
                stripos($statement, 'SET @@') === 0 ||
                stripos($statement, 'SET NAMES') === 0 ||
                stripos($statement, 'SET SQL_MODE') === 0 ||
                stripos($statement, 'SET TIME_ZONE') === 0 ||
                stripos($statement, 'SET FOREIGN_KEY_CHECKS') === 0 ||
                stripos($statement, 'SET CHARACTER_SET') === 0 ||
                stripos($statement, 'SET SQL_NOTES') === 0) {
                $skipped++;
                continue;
            }
            
            // Skip CREATE TABLE statements (tables should exist from migration)
            if (stripos($statement, 'CREATE TABLE') === 0) {
                $skipped++;
                continue;
            }
            
            // Skip ALTER TABLE statements
            if (stripos($statement, 'ALTER TABLE') === 0) {
                $skipped++;
                continue;
            }
            
            // Execute INSERT statements
            if (stripos($statement, 'INSERT INTO') === 0) {
                try {
                    DB::statement($statement);
                    $imported++;
                    
                    // Show progress for every 10 statements
                    if ($imported % 10 == 0) {
                        $this->command->info("  ✓ Imported {$imported} records...");
                    }
                } catch (\Exception $e) {
                    $errors++;
                    // Only log the first part of statement to avoid clutter
                    $preview = substr($statement, 0, 100) . '...';
                    $this->command->warn("  ⚠ Skipped statement: {$preview}");
                    $this->command->warn("  Error: " . $e->getMessage());
                }
            }
        }
        
        $this->command->newLine();
        $this->command->info('📊 Import Summary:');
        $this->command->info("  ✓ Successfully imported: {$imported} records");
        $this->command->info("  ⊘ Skipped: {$skipped} statements");
        if ($errors > 0) {
            $this->command->warn("  ⚠ Errors: {$errors} statements");
        }
        
        $this->command->newLine();
        $this->command->info('✅ Demo data imported successfully!');
        $this->command->newLine();
        $this->command->warn('🔐 Demo Login Credentials:');
        $this->command->warn('   Email: admin@demo.com');
        $this->command->warn('   Password: 123456');
        $this->command->newLine();
        $this->command->info('📝 Note: Change admin password after first login!');
    }
}


