var menuModel = {
    actives: ["regions"],
    shows: [],
};
menuSets(menuModel);

let myMap = new LeafletMap(); // class örneği oluştur

myMap.initMap("map1", {
    //zoomLevelDefault: 12 // opsiyonel ayar, default zoom
    enablePolygonDrawing: false, // Enable drawing controls
    drawingOptions: {
        position: "topleft",
        enablePolygon: true,
        enableRectangle: true,
        enableDelete: true,
        polygonColor: "#d32f2f",
        polygonFillColor: "#ffcdd2",
    },
});

var regionsListData = [];

onAppReady(function () {
    setupLeafletDrawLocalization();

    http.post({
        url: "/regions-list",
    }).then((response) => {
        ////console.log(response);
        var data = response.data;

        regionsList(data);

        regionsListData = data;

        data.forEach(function (da) {
            myMap.addPolygon({
                id: da.id,
                label: da.name,
                coords: da.polygon_json,
            });
        });

        myMap.fitBoundsToPolygons();

        // Load and display region statistics
        updateRegionStats(data);
    });
});

// ========================
// Updated JavaScript for Tabulator
// ========================

function regionsList(data) {
    var table = createTabulator("#regionsList", {
        buttons: [
            {
                label: __("Add New Region"),
                icon: "fas fa-plus",
                icon_color: "success",
                func: addNewRegion,
            },
        ],
        data: data,
        movableRows: true,
        rowHeader: {
            headerSort: false,
            resizable: false,
            minWidth: 30,
            width: 30,
            rowHandle: true,
            formatter: "handle",
        },
        initialSort: [
            //{column: "name", dir: "asc"}
        ],
        columns: [
            {
                title: __("Name"),
                field: "name",
                editor: "input",
                minWidth: 200,
                validator: "required",
            },
            {
                title: __("Status"),
                field: "active",
                width: 120,
                headerSort: false,
                formatter: function (cell, formatterParams) {
                    var value = cell.getValue();
                    var rowData = cell.getRow().getData();

                    if (value == 1) {
                        return `
                <label class="form-check form-switch toggle-status-btn" 
                    data-bs-toggle="popover" data-bs-trigger="hover" data-bs-placement="top" 
                    data-bs-content="${__("Toggle Status")}" data-status="0">
                      <input class="form-check-input bg-lime" type="checkbox" checked="checked">
                      <span class="form-check-label">${__("Active")}</span>
                    </label>
                `;
                    } else {
                        return `
                <label class="form-check form-switch toggle-status-btn" 
                    data-bs-toggle="popover" data-bs-trigger="hover" data-bs-placement="top" 
                    data-bs-content="${__("Toggle Status")}" data-status="1">
                      <input class="form-check-input bg-red" type="checkbox">
                      <span class="form-check-label">${__("Inactive")}</span>
                    </label>
                `;
                    }
                },
                cellClick: function (e, cell) {
                    // Event propagation'ı durdur
                    e.stopPropagation();
                    e.preventDefault();

                    if (e.target.closest(".toggle-status-btn")) {
                        var newStatus = e.target
                            .closest(".toggle-status-btn")
                            .getAttribute("data-status");
                        var rowData = cell.getRow().getData();

                        // Çift tıklamayı önlemek için debounce
                        clearTimeout(cell._statusToggleTimeout);
                        cell._statusToggleTimeout = setTimeout(function () {
                            toggleRegionStatus(
                                cell.getRow(),
                                rowData,
                                newStatus
                            );
                        }, 100);
                    }
                },
            },
            {
                title: __("Action"),
                width: 150,
                headerSort: false,
                formatter: function (cell, formatterParams) {
                    return `
                    <div class="btn-group" role="group">
                        <button class="btn btn-sm btn-outline-warning px-2 py-1 edit-row-btn"
                            data-bs-toggle="popover" data-bs-trigger="hover" data-bs-placement="top" 
                            data-bs-content="${__("Edit")}">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-success px-2 py-1 duplicate-row-btn"
                            data-bs-toggle="popover" data-bs-trigger="hover" data-bs-placement="top" 
                            data-bs-content="${__("Duplicate")}">
                            <i class="fas fa-copy"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger px-2 py-1 delete-row-btn"
                            data-bs-toggle="popover" data-bs-trigger="hover" data-bs-placement="top" 
                            data-bs-content="${__("Delete")}">
                            <i class="fas fa-trash"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-primary px-2 py-1 map-row-btn"
                            data-bs-toggle="popover" data-bs-trigger="hover" data-bs-placement="top" 
                            data-bs-content="${__("Map")}">
                            <i class="fas fa-map"></i>
                        </button>
                    </div>
                    `;
                },
                cellClick: function (e, cell) {
                    var rowData = cell.getRow().getData();

                    if (e.target.closest(".edit-row-btn")) {
                        editRegionRow(cell.getRow(), rowData);
                    } else if (e.target.closest(".duplicate-row-btn")) {
                        duplicateRegionRow(rowData);
                    } else if (e.target.closest(".delete-row-btn")) {
                        deleteRegionRow(cell.getRow(), rowData);
                    } else if (e.target.closest(".map-row-btn")) {
                        mapRegionRow(cell.getRow(), rowData);
                    }
                },
            },
        ],
    });

    // Row selection events
    table.on("rowSelected", function (row) {
        var rowData = row.getData();
        //console.log("Region selected:", rowData.name);

        // Burada seçilen bölgeyi haritada gösterebilirsiniz
        if (typeof selectRegionOnMap === "function") {
            selectRegionOnMap(rowData);
        }
    });

    table.on("rowDeselected", function (row) {
        //console.log("Region deselected");

        // Haritadaki seçimi kaldır
        if (typeof deselectRegionOnMap === "function") {
            deselectRegionOnMap();
        }
    });

    // Row movement events
    table.on("rowMoved", function (row) {
        //console.log("Row moved:", row.getData().name);

        // Sıralama değiştiğinde backend'e gönder
        updateRowOrder(table.getData());

        // Initialize popovers after row movement
        popoverInit();
    });

    // Cell editing events
    table.on("cellEdited", function (cell) {
        var rowData = cell.getRow().getData();
        //console.log("Cell edited:", rowData);

        // Düzenleme sonrası backend'e gönder
        updateRegionData(rowData);

        // Initialize popovers after cell editing
        popoverInit();

        // Update statistics after editing
        updateRegionStats(table.getData());
    });

    // Row context menu (sağ tık menüsü)
    table.on("rowContext", function (e, row) {
        e.preventDefault();
        showRowContextMenu(e, row);
    });

    // Global table variable for external access
    window.regionsTable = table;

    // Initialize popovers for action buttons
    setTimeout(function () {
        popoverInit();
    }, 100);

    return table;
}

// Function to update region statistics
function updateRegionStats(regions) {
    const totalRegions = regions.length;
    const activeRegions = regions.filter(
        (region) => region.active === 1 || region.sts === 1
    ).length;
    const passiveRegions = totalRegions - activeRegions;

    // Update the display with animation
    animateCounter("#totalRegions", totalRegions);
    animateCounter("#activeRegions", activeRegions);
    animateCounter("#passiveRegions", passiveRegions);
}

// Function to animate counter numbers
function animateCounter(selector, finalValue) {
    const element = $(selector);
    const startValue = 0;
    const duration = 1000;
    const increment = finalValue / (duration / 16); // 60fps
    let currentValue = startValue;

    const timer = setInterval(function () {
        currentValue += increment;
        if (currentValue >= finalValue) {
            currentValue = finalValue;
            clearInterval(timer);
        }
        element.text(Math.floor(currentValue));
    }, 16);
}

// CRUD Functions

function addNewRegion() {
    var newRegion = {
        name: __("New Region"), // Backend için 'nm' field
        active: 1,
        order: null, // Backend otomatik set edecek
    };

    const postModel = Base64Helper.encode(newRegion);

    http.post({
        url: "/region-add",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                // Tabulator'a yeni satır ekle
                var row = window.regionsTable.addRow(response.data, true);

                showNotification("success", __("Region added successfully"));

                // Initialize popovers for new row
                popoverInit();

                // Update statistics
                updateRegionStats(window.regionsTable.getData());
            } else {
                showNotification("error", __("Error adding region"));
            }
        })
        .catch((error) => {
            console.error("Add error:", error);
            showNotification("error", __("Error adding region"));
        });
}

function editRegionRow(row, rowData) {
    // Name cell'ini düzenleme moduna al
    var nameCell = row.getCell("name");
    if (nameCell) {
        nameCell.edit();
    }

    // Satırı seç
    row.select();
}

function duplicateRegionRow(rowData) {
    var duplicatedRegion = {
        id: rowData.id, // Backend için 'nm' field
        name: rowData.name + " (" + __("Copy") + ")", // Backend için 'nm' field
        active: rowData.active,
        order: null, // Backend otomatik set edecek
    };

    const postModel = Base64Helper.encode(duplicatedRegion);

    http.post({
        url: "/region-duplicate",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                // Tabulator'a yeni satır ekle
                var row = window.regionsTable.addRow(response.data, true);
                showNotification(
                    "success",
                    __("Region duplicated successfully")
                );

                // Initialize popovers for new row
                popoverInit();

                // Update statistics
                updateRegionStats(window.regionsTable.getData());
            } else {
                showNotification("error", __("Error duplicating region"));
            }
        })
        .catch((error) => {
            console.error("Duplicate error:", error);
            showNotification("error", __("Error duplicating region"));
        });
}

function deleteRegionRow(row, rowData) {
    showConfirmDialog({
        title: __("Delete Region"),
        message:
            __("Are you sure you want to delete") + " : " + rowData.name + "",
        confirmText: __("Delete"),
        cancelText: __("Cancel"),
        type: "danger",
        onConfirm: function () {
            const postModel = Base64Helper.encode({ id: rowData.id });

            http.post({
                url: "/region-delete",
                data: {
                    data: postModel,
                },
            })
                .then((response) => {
                    if (response.success) {
                        row.delete();
                        showNotification(
                            "success",
                            __("Region deleted successfully")
                        );

                        // Haritadaki bölgeyi kaldır
                        myMap.removePolygon(rowData.id);

                        // Update statistics
                        updateRegionStats(window.regionsTable.getData());
                    } else {
                        showNotification("error", __("Error deleting region"));
                    }
                })
                .catch((error) => {
                    console.error("Delete error:", error);
                    showNotification("error", __("Error deleting region"));
                });
        },
    });
}

// Backend Communication Functions

function updateRegionData(rowData) {
    // Backend'in beklediği field mapping
    const mappedData = {
        id: rowData.id,
        nm: rowData.name, // 'name' -> 'nm'
        active: rowData.active,
        ord: rowData.order,
    };

    const postModel = Base64Helper.encode(mappedData);

    http.post({
        url: "/region-edit",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                showNotification("success", __("Region updated successfully"));
            } else {
                showNotification("error", __("Error updating region"));
            }
        })
        .catch((error) => {
            console.error("Update error:", error);
            showNotification("error", __("Error updating region"));
        });
}

function saveNewRegion(regionData) {
    http.post({
        url: "/region-add",
        data: regionData,
    })
        .then((response) => {
            if (response.success && response.data.id) {
                // Geçici ID'yi gerçek ID ile değiştir
                window.regionsTable.updateData([
                    {
                        id: regionData.id,
                        ...response.data,
                    },
                ]);

                showNotification("success", __("Region added successfully"));
            } else {
                showNotification("error", __("Error adding region"));
            }
        })
        .catch((error) => {
            console.error("Add error:", error);
            showNotification("error", __("Error adding region"));
        });
}

function deleteRegionFromBackend(regionId) {
    return http.post({
        url: "/region-delete",
        data: { id: regionId },
    });
}

function updateRowOrder(tableData) {
    var orderData = tableData.map(function (row, index) {
        return {
            id: row.id,
            order: index + 1,
        };
    });

    const postModel = Base64Helper.encode({ orders: orderData });

    http.post({
        url: "/regions-update-order",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                //console.log("Row order updated");
            }
        })
        .catch((error) => {
            console.error("Order update error:", error);
        });
}

// Utility Functions

function generateTempId() {
    return "temp_" + Date.now() + "_" + Math.random().toString(36).substr(2, 9);
}

function showRowContextMenu(e, row) {
    var rowData = row.getData();

    // Context menu HTML
    var contextMenu = $(
        '<div class="dropdown-menu show" style="position: fixed; z-index: 9999;">' +
            '<h6 class="dropdown-header">' +
            rowData.name +
            "</h6>" +
            '<button class="dropdown-item context-edit"><i class="fas fa-edit"></i> ' +
            __("Edit") +
            "</button>" +
            '<button class="dropdown-item context-duplicate"><i class="fas fa-copy"></i> ' +
            __("Duplicate") +
            "</button>" +
            '<div class="dropdown-divider"></div>' +
            '<button class="dropdown-item context-toggle-status"><i class="fas fa-toggle-' +
            (rowData.active ? "off" : "on") +
            '"></i> ' +
            (rowData.active ? __("Deactivate") : __("Activate")) +
            "</button>" +
            '<div class="dropdown-divider"></div>' +
            '<button class="dropdown-item text-danger context-delete"><i class="fas fa-trash"></i> ' +
            __("Delete") +
            "</button>" +
            "</div>"
    );

    // Position the menu
    contextMenu.css({
        left: e.pageX + "px",
        top: e.pageY + "px",
    });

    // Add to body
    $("body").append(contextMenu);

    // Event handlers
    contextMenu.find(".context-edit").click(function () {
        editRegionRow(row, rowData);
        contextMenu.remove();
    });

    contextMenu.find(".context-duplicate").click(function () {
        duplicateRegionRow(rowData);
        contextMenu.remove();
    });

    contextMenu.find(".context-toggle-status").click(function () {
        toggleRegionStatus(row, rowData);
        contextMenu.remove();
    });

    contextMenu.find(".context-delete").click(function () {
        deleteRegionRow(row, rowData);
        contextMenu.remove();
    });

    // Remove menu on outside click
    $(document).one("click", function () {
        contextMenu.remove();
    });
}

function toggleRegionStatus(row, rowData) {
    var newStatus = rowData.active == 1 ? 0 : 1;

    const postModel = Base64Helper.encode({
        id: rowData.id,
        status: newStatus,
    });

    http.post({
        url: "/region-toggle-status",
        data: {
            data: postModel,
        },
    })
        .then((response) => {
            if (response.success) {
                row.update({ active: newStatus });
                showNotification("success", __("Status updated successfully"));

                // Update statistics
                updateRegionStats(window.regionsTable.getData());
            } else {
                showNotification("error", __("Error updating status"));
            }
        })
        .catch((error) => {
            console.error("Toggle status error:", error);
            showNotification("error", __("Error updating status"));
        });
}

// Public API functions for external use
window.addNewRegion = addNewRegion;
window.duplicateRegionRow = duplicateRegionRow;
window.editRegionRow = editRegionRow;
window.deleteRegionRow = deleteRegionRow;
window.toggleRegionStatus = toggleRegionStatus;
window.refreshRegionsTable = function () {
    http.post({
        url: "/regions-list",
    }).then((response) => {
        window.regionsTable.setData(response.data);
        // Initialize popovers for refreshed data
        popoverInit();
        // Update statistics with new data
        updateRegionStats(response.data);
    });
};

// Function to refresh statistics
window.refreshRegionStats = function () {
    if (window.regionsTable) {
        updateRegionStats(window.regionsTable.getData());
    }
};

// Keyboard shortcuts
$(document).keydown(function (e) {
    // Ctrl+N: Add new region
    if (e.ctrlKey && e.keyCode === 78) {
        e.preventDefault();
        addNewRegion();
    }

    // Delete key: Delete selected row
    if (e.keyCode === 46) {
        var selectedRows = window.regionsTable.getSelectedRows();
        if (selectedRows.length > 0) {
            deleteRegionRow(selectedRows[0], selectedRows[0].getData());
        }
    }
});

// Enhanced mapRegionRow function with polygon editing capabilities
function mapRegionRow(row, rowData) {
    // 1. Clear all polygons except the selected one
    clearOtherPolygons(rowData.id);

    // 2. Fit bounds to the selected polygon
    myMap.fitBoundsToPolygons([rowData.id]);

    // 3. Enable drawing tools for editing
    enablePolygonEditingMode(rowData.id);

    // 4. Setup coordinate saving to hidden input
    setupCoordinateSaving(rowData.id);

    // 5. Update UI to show editing mode
    showPolygonEditingInterface(rowData);
}

// Clear all polygons except the specified one
function clearOtherPolygons(keepPolygonId) {
    var allPolygonIds = myMap.getPolygonIds();

    allPolygonIds.forEach(function (polygonId) {
        if (polygonId !== keepPolygonId) {
            myMap.removePolygon(polygonId);
        }
    });

    //console.log("Cleared all polygons except:", keepPolygonId);
}

// Enable polygon editing mode with drawing tools
function enablePolygonEditingMode(polygonId) {
    // Enable drawing controls if not already enabled
    if (!myMap.drawControl) {
        myMap.initDrawingControls({
            position: "topleft",
            enablePolygonDrawing: true, // Enable polygon drawing
            enablePolygon: true,
            enableRectangle: true,
            enableDelete: true,
            enableEdit: true,
            polygonColor: "#d32f2f",
            polygonFillColor: "#ffcdd2",
        });
    }

    // Set up custom event callbacks for this editing session
    myMap.setPolygonEventCallbacks({
        onCreated: function (layer, newPolygonId) {
            //console.log("New polygon created during editing:", newPolygonId);
            handleNewPolygonCreated(layer, newPolygonId, polygonId);
        },
        onEdited: function (layer, editedPolygonId) {
            //console.log("Polygon edited:", editedPolygonId);
            handlePolygonEdited(layer, editedPolygonId);
        },
        onDeleted: function (layer, deletedPolygonId) {
            //console.log("Polygon deleted:", deletedPolygonId);
            handlePolygonDeleted(deletedPolygonId);
        },
    });

    //console.log("Polygon editing mode enabled for:", polygonId);
}

// Setup coordinate saving to hidden input
function setupCoordinateSaving(polygonId) {
    // Create or find hidden input for coordinates
    var hiddenInput = document.getElementById(
        "polygon-coordinates-" + polygonId
    );
    if (!hiddenInput) {
        hiddenInput = document.createElement("input");
        hiddenInput.type = "hidden";
        hiddenInput.id = "polygon-coordinates-" + polygonId;
        hiddenInput.name = "polygon_coordinates[" + polygonId + "]";
        document.body.appendChild(hiddenInput);
    }

    // Save current polygon coordinates
    var currentPolygon = myMap.getPolygon(polygonId);
    if (currentPolygon) {
        var coordinates = myMap.getPolygonCoordinates(polygonId);
        hiddenInput.value = JSON.stringify(coordinates);
        //console.log("Current coordinates saved for polygon:", polygonId);
    }

    return hiddenInput;
}

// Handle new polygon created during editing
function handleNewPolygonCreated(layer, newPolygonId, originalPolygonId) {
    // Remove the original polygon since we're replacing it
    myMap.removePolygon(originalPolygonId);

    // Update the new polygon with the original ID
    myMap.removePolygon(newPolygonId);

    // Re-add with original ID
    var coordinates = layer.getLatLngs();
    myMap.addPolygon({
        id: originalPolygonId,
        coords: coordinates[0], // Get the outer ring
        color: "#d32f2f",
        fillColor: "#ffcdd2",
        label: "Region Polygon",
    });

    // Update hidden input with new coordinates
    updateHiddenInput(originalPolygonId, coordinates[0]);

    // Show save/cancel buttons
    showPolygonSaveOptions(originalPolygonId);
}

// Handle polygon edited
function handlePolygonEdited(layer, polygonId) {
    var coordinates = layer.getLatLngs();
    updateHiddenInput(polygonId, coordinates[0]);
    showPolygonSaveOptions(polygonId);
    //console.log("Polygon coordinates updated:", polygonId);
}

// Handle polygon deleted
function handlePolygonDeleted(polygonId) {
    // Clear hidden input
    var hiddenInput = document.getElementById(
        "polygon-coordinates-" + polygonId
    );
    if (hiddenInput) {
        hiddenInput.value = "";
    }

    showPolygonSaveOptions(polygonId);
    //console.log("Polygon deleted:", polygonId);
}

// Update hidden input with new coordinates
function updateHiddenInput(polygonId, coordinates) {
    var hiddenInput = document.getElementById(
        "polygon-coordinates-" + polygonId
    );
    if (!hiddenInput) {
        hiddenInput = setupCoordinateSaving(polygonId);
    }

    // Convert coordinates to simple lat/lng array format
    var coordArray = coordinates.map(function (coord) {
        return [coord.lat, coord.lng];
    });

    hiddenInput.value = JSON.stringify(coordArray);

    //console.log("Hidden input updated for polygon:", polygonId, coordArray);
}

// Show polygon editing interface
function showPolygonEditingInterface(rowData) {
    // Remove existing editing interface
    removePolygonEditingInterface();

    // Create editing interface HTML
    var editingInterface = $(`
        <div class="btn-group w-100">
            <button class="btn btn-primary btn-square mb-0" onclick="exitPolygonEditingMode()">
                <i class="fas fa-arrow-left me-3"></i> ${__("Back to List")}
            </button>
        </div>
        <div id="polygon-editing-interface" class="p-3">
        <input type="hidden" id="polygon-coordinates-${
            rowData.id
        }" name="polygon_coordinates[${rowData.id}]" value='${JSON.stringify(
        rowData.polygon_json
    )}'>
        
            <h3 class="mb-3">
                ${rowData.name}
            </h3>

            <div class="mb-3">
                <small class="text-muted">
                    ${__(
                        "Use the drawing tools to modify the polygon boundaries"
                    )}
                </small>
            </div>

            <div id="polygon-save-options-${rowData.id}" style="display: none;">
                <div class="alert alert-warning alert-sm mb-2">
                    <i class="fas fa-exclamation-triangle"></i>
                    ${__("Polygon has been modified")}
                </div>

                <div class="btn-group w-100 mb-2">
                    <button class="btn btn-success save-polygon-btn"
                        data-polygon-id="${rowData.id}">
                        <i class="fas fa-save"></i> ${__("Save")}
                    </button>
                    <button class="btn btn-secondary cancel-polygon-btn"
                        data-polygon-id="${rowData.id}">
                        <i class="fas fa-times"></i> ${__("Cancel")}
                    </button>
                </div>
            </div>

            <div class="mt-2">
                <small class="text-muted">
                    <strong>${__("Instructions")}:</strong><br>
                    <i class="fas fa-square"></i> ${__(
                        "Click the polygon tool to draw new boundaries"
                    )}<br>
                    <i class="fas fa-edit"></i> ${__(
                        "Click the edit tool to modify existing polygon"
                    )}<br>
                    <i class="fas fa-trash"></i> ${__(
                        "Click the delete tool to remove polygon"
                    )}
                </small>
            </div>
        </div>
    `);

    $("#regionForm").html(editingInterface);
    $("#regionForm").show();
    $("#regionsList").hide();

    // Add event handlers
    setupEditingInterfaceEvents(rowData.id);
}

// Setup event handlers for editing interface
function setupEditingInterfaceEvents(polygonId) {
    // Save button handler
    $(document).on("click", ".save-polygon-btn", function () {
        var polygonId = $(this).data("polygon-id");
        savePolygonChanges(polygonId);
    });

    // Cancel button handler
    $(document).on("click", ".cancel-polygon-btn", function () {
        var polygonId = $(this).data("polygon-id");
        cancelPolygonChanges(polygonId);
    });
}

// Show save/cancel options
function showPolygonSaveOptions(polygonId) {
    $(`#polygon-save-options-${polygonId}`).show();
}

// Save polygon changes to backend
function savePolygonChanges(polygonId) {
    var hiddenInput = document.getElementById(
        "polygon-coordinates-" + polygonId
    );

    try {
        var coordinates = [];
        if (hiddenInput && hiddenInput.value) {
            coordinates = JSON.parse(hiddenInput.value);
        }

        // Prepare data for backend
        var postData = {
            id: polygonId,
            coordinates: coordinates,
        };

        const postModel = Base64Helper.encode(postData);

        // Show loading state
        $(".save-polygon-btn")
            .prop("disabled", true)
            .html('<i class="fas fa-spinner fa-spin"></i> ' + __("Saving..."));

        http.post({
            url: "/region-save-polygon",
            data: {
                data: postModel,
            },
        })
            .then((response) => {
                if (response.success) {
                    showNotification(
                        "success",
                        __("Polygon updated successfully")
                    );

                    // Update table data
                    if (window.regionsTable) {
                        window.regionsTable.updateData([
                            {
                                id: polygonId,
                                polygon_json: coordinates,
                            },
                        ]);
                    }

                    // Hide save options
                    $(`#polygon-save-options-${polygonId}`).hide();
                } else {
                    showNotification("error", __("Error updating polygon"));
                }
            })
            .catch((error) => {
                console.error("Save polygon error:", error);
                showNotification("error", __("Error updating polygon"));
            })
            .finally(() => {
                // Reset button state
                $(".save-polygon-btn")
                    .prop("disabled", false)
                    .html('<i class="fas fa-save"></i> ' + __("Save"));
            });
    } catch (error) {
        console.error("Invalid coordinate data:", error);
        showNotification("error", __("Invalid coordinate data"));
    }
}

// Cancel polygon changes
function cancelPolygonChanges(polygonId) {
    // Reload original polygon data
    var polygon = regionsListData.find((region) => region.id === polygonId);

    if (polygon) {
        // Remove current polygon
        myMap.removePolygon(polygonId);

        // Re-add original polygon
        myMap.addPolygon({
            id: polygon.id,
            label: polygon.name,
            coords: polygon.polygon_json,
        });

        // Reset hidden input
        var hiddenInput = document.getElementById(
            "polygon-coordinates-" + polygonId
        );
        if (hiddenInput) {
            hiddenInput.value = JSON.stringify(polygon.polygon_json);
        }

        // Hide save options
        $(`#polygon-save-options-${polygonId}`).hide();

        showNotification("info", __("Changes cancelled"));
    }
}

// Exit polygon editing mode
function exitPolygonEditingMode() {
    // Remove editing interface
    removePolygonEditingInterface();

    $("#regionForm").hide();
    $("#regionsList").show();

    // Disable drawing controls
    if (myMap.drawControl) {
        myMap.mapLet.removeControl(myMap.drawControl);
        myMap.drawControl = null;
    }

    // Clear polygon event callbacks
    myMap.setPolygonEventCallbacks({
        onCreated: null,
        onEdited: null,
        onDeleted: null,
    });

    // Reload all polygons
    reloadAllPolygons();

    //console.log("Exited polygon editing mode");
}

// Remove polygon editing interface
function removePolygonEditingInterface() {
    $("#polygon-editing-interface").remove();

    // Remove event handlers
    $(document).off("click", ".save-polygon-btn");
    $(document).off("click", ".cancel-polygon-btn");
}

// Reload all polygons from backend
function reloadAllPolygons() {
    http.post({
        url: "/regions-list",
    }).then((response) => {
        var data = response.data;

        // Clear all current polygons
        myMap.clearAllPolygons();

        // Add all polygons back
        data.forEach(function (da) {
            myMap.addPolygon({
                id: da.id,
                label: da.name,
                coords: da.polygon_json,
            });
        });

        // Fit bounds to all polygons
        myMap.fitBoundsToPolygons();
    });
}

// Get polygon coordinates from hidden input
function getPolygonCoordinates(polygonId) {
    var hiddenInput = document.getElementById(
        "polygon-coordinates-" + polygonId
    );
    if (hiddenInput && hiddenInput.value) {
        try {
            return JSON.parse(hiddenInput.value);
        } catch (error) {
            console.error("Error parsing polygon coordinates:", error);
            return null;
        }
    }
    return null;
}

// Utility function to check if polygon has unsaved changes
function hasUnsavedPolygonChanges(polygonId) {
    return $(`#polygon-save-options-${polygonId}`).is(":visible");
}

// Export functions to global scope
window.mapRegionRow = mapRegionRow;
window.exitPolygonEditingMode = exitPolygonEditingMode;
window.savePolygonChanges = savePolygonChanges;
window.cancelPolygonChanges = cancelPolygonChanges;
window.getPolygonCoordinates = getPolygonCoordinates;
window.hasUnsavedPolygonChanges = hasUnsavedPolygonChanges;
