@extends('layouts.app')

@section('title', __('Driver Scoring Report'))

@section('content')
<div class="page-wrapper" id="driver-scoring-report">
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted">{{ __('Reports') }}</a> /
                        {{ __('Driver Performance Reports') }}
                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-award me-2"></i>
                        {{ __('Driver Scoring Report') }}
                    </h2>
                    <div class="text-muted mt-1">
                        {{ __('Comprehensive driver scoring based on safety, efficiency, and compliance metrics') }}
                    </div>
                </div>
                <div class="col-auto ms-auto d-print-none">
                    <div class="btn-list">
                        <button type="button" class="btn btn-primary" onclick="window.print()">
                            <i class="ti ti-printer me-1"></i>
                            {{ __('Print Report') }}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title">{{ __('Filters') }}</h3>
                    <div class="card-actions">
                        <button type="button" class="btn btn-sm btn-outline-primary" data-bs-toggle="collapse"
                            data-bs-target="#helpSection">
                            <i class="ti ti-help me-1"></i>{{ __('How to use this report') }}
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    <!-- Help Section -->
                    <div class="collapse mb-3" id="helpSection">
                        <div class="alert alert-info">
                            <h4 class="alert-heading"><i class="ti ti-info-circle me-2"></i>
                                {{ __('How to use the Driver Scoring Report') }}</h4>
                            <p class="mb-2"><strong>{{ __('Step 1:') }}</strong>
                                {{ __('Select a driver (optional) or leave as "All Drivers"') }}</p>
                            <p class="mb-2"><strong>{{ __('Step 2:') }}</strong>
                                {{ __('Set your date range for the analysis') }}</p>
                            <p class="mb-2"><strong>{{ __('Step 3:') }}</strong>
                                {{ __('Click "Generate Report" to see the results') }}</p>
                        </div>
                    </div>
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">{{ __('Driver') }}</label>
                            <select class="form-select" id="driverSelect">
                                <option value="">{{ __('All Drivers') }}</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">{{ __('Date Range') }}</label>
                            <div class="input-group">
                                <input type="date" class="form-control" id="startDate"
                                    value="{{ date('Y-m-d', strtotime('-1 week')) }}">
                                <span class="input-group-text">{{ __('to') }}</span>
                                <input type="date" class="form-control" id="endDate" value="{{ date('Y-m-d') }}">
                            </div>
                        </div>
                        <div class="col-md-2 d-flex align-items-end">
                            <button type="button" class="btn btn-primary w-100" onclick="generateReport()">
                                <i class="icon ti ti-search me-1"></i>
                                {{ __('Generate Report') }}
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Overview Cards -->
            <div class="row row-cards mb-3 d-none" id="driverOverviewCards">
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Average Driver Score') }}</div>
                            <div class="h1 mb-0" id="averageDriverScore">-</div>
                            <div class="text-muted small" id="scoreBreakdown">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Top Performers') }}</div>
                            <div class="h1 mb-0" id="topPerformers">-</div>
                            <div class="text-muted small" id="topPerformersBreakdown">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Needs Improvement') }}</div>
                            <div class="h1 mb-0" id="needsImprovement">-</div>
                            <div class="text-muted small" id="improvementBreakdown">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Violation Count') }}</div>
                            <div class="h1 mb-0" id="totalViolations">-</div>
                            <div class="text-muted small" id="violationsBreakdown">-</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Charts Row 1 -->
            <div class="row mb-3 d-none" id="chartsRow1">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Driver Score Distribution') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="driverScoreChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Score Components Breakdown') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="scoreBreakdownChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Charts Row 2 -->
            <div class="row mb-3 d-none" id="chartsRow2">
                <div class="col-lg-8">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Violation Trends Over Time') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="violationTrendsChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-4">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Hourly Performance') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="hourlyPerformanceChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Driver Comparison Chart -->
            <div class="row mb-3 d-none" id="driverComparisonRow">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Driver Performance Comparison') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="driverComparisonChart" style="height: 400px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Main Driver Scoring Table -->
            <div class="card mb-3 d-none" id="driverScoringTableCard">
                <div class="card-header">
                    <h3 class="card-title">
                        {{ __('Driver Scoring Details') }}
                        <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                            title="{{ __('Detailed scoring information for each driver. Click Details to see more.') }}"></i>
                    </h3>
                    <div class="card-actions">
                        <div class="text-muted small">
                            {{ __('Click the Details button to see detailed information about each driver.') }}
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div id="driver-scoring-table"></div>
                </div>
            </div>

            <!-- Violation Details Table -->
            <div class="card mb-3 d-none" id="violationDetailsTableCard">
                <div class="card-header">
                    <h3 class="card-title">
                        {{ __('Violation Details') }}
                        <i class="ti ti-info-circle text-muted ms-1" data-bs-toggle="tooltip"
                            title="{{ __('Detailed violation information for all drivers.') }}"></i>
                    </h3>
                </div>
                <div class="card-body p-0">
                    <div id="violation-details-table"></div>
                </div>
            </div>

            <!-- No Vehicle State -->
            <div class="text-center py-4" id="noVehicleState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-car text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title">{{ __('No Driver Selected') }}</p>
                    <p class="empty-subtitle text-muted">
                        {{ __('To generate a driver scoring report, please select a driver from the dropdown menu above or leave as All Drivers.') }}
                    </p>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">{{ __('Loading...') }}</span>
                </div>
                <div class="mt-2">{{ __('Generating driver scoring report...') }}</div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-award text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title">{{ __('No driver scoring data found') }}</p>
                    <p class="empty-subtitle text-muted">
                        {{ __('No driver scoring data was found for the selected period.') }}
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Driver Details Modal -->
<div class="modal fade" id="driverDetailsModal" tabindex="-1" aria-labelledby="driverDetailsModalLabel"
    aria-hidden="true">
    <div class="modal-dialog modal-xl modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="driverDetailsModalLabel">{{ __('Driver Scoring Details') }}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="driverDetailsContent">
                    <!-- Driver details will be populated here -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ __('Close') }}</button>
            </div>
        </div>
    </div>
</div>

<!-- Violation Details Modal -->
<div class="modal fade" id="violationDetailsModal" tabindex="-1" aria-labelledby="violationDetailsModalLabel"
    aria-hidden="true">
    <div class="modal-dialog modal-fullscreen modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header bg-gradient-primary text-white">
                <h5 class="modal-title" id="violationDetailsModalLabel">
                    <i class="ti ti-map-pin me-2"></i>
                    <span id="violationModalTitle">{{ __('Violation Details') }}</span>
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body p-0">
                <div class="row g-0 h-100">
                    <!-- Map Column -->
                    <div class="col-md-8 h-100">
                        <div id="violationMap" style="height: 100vh; width: 100%;"></div>
                    </div>
                    <!-- Details Column -->
                    <div class="col-md-4 h-100">
                        <div class="p-3 h-100 overflow-auto">
                            <!-- Summary Cards -->
                            <div class="row g-2 mb-3">
                                <div class="col-6">
                                    <div class="card bg-danger text-white">
                                        <div class="card-body p-2 text-center">
                                            <div class="h4 mb-0" id="totalViolationsCount">0</div>
                                            <small>{{ __('Total Violations') }}</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-6">
                                    <div class="card bg-warning text-white">
                                        <div class="card-body p-2 text-center">
                                            <div class="h4 mb-0" id="uniqueLocationsCount">0</div>
                                            <small>{{ __('Unique Locations') }}</small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Violation Type Filter -->
                            <div class="mb-3" id="violationTypeFilterDiv">
                                <label class="form-label fw-bold">{{ __('Filter by Type') }}</label>
                                <div class="btn-group w-100" role="group">
                                    <input type="radio" class="btn-check" name="violationTypeFilter" id="allTypes" value="all" checked>
                                    <label class="btn btn-outline-primary btn-sm" for="allTypes">{{ __('All') }}</label>
                                    
                                    <input type="radio" class="btn-check" name="violationTypeFilter" id="speedType" value="speed">
                                    <label class="btn btn-outline-danger btn-sm" for="speedType">{{ __('Speed') }}</label>
                                    
                                    <input type="radio" class="btn-check" name="violationTypeFilter" id="harshType" value="harsh">
                                    <label class="btn btn-outline-warning btn-sm" for="harshType">{{ __('Harsh') }}</label>
                                    
                                    <input type="radio" class="btn-check" name="violationTypeFilter" id="zoneType" value="zone">
                                    <label class="btn btn-outline-info btn-sm" for="zoneType">{{ __('Zone') }}</label>
                                </div>
                            </div>
                            
                            <!-- Violations List -->
                            <div class="mb-3">
                                <label class="form-label fw-bold">{{ __('Violations List') }}</label>
                                <div id="violationsList" class="list-group list-group-flush h-100" style="overflow-y: auto;">
                                    <!-- Violations will be populated here -->
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ __('Close') }}</button>
                <button type="button" class="btn btn-primary" onclick="exportViolationData()">
                    <i class="ti ti-download me-1"></i>{{ __('Export Data') }}
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('styles')
<style>
    .score-status-excellent {
        color: #28a745;
    }

    .score-status-good {
        color: #17a2b8;
    }

    .score-status-fair {
        color: #ffc107;
    }

    .score-status-poor {
        color: #fd7e14;
    }

    .score-status-critical {
        color: #dc3545;
    }

    .score-badge {
        font-size: 0.875rem;
        padding: 0.25rem 0.5rem;
        border-radius: 0.25rem;
        font-weight: 600;
    }

    .score-excellent {
        background-color: #d4edda;
        color: #155724;
    }

    .score-good {
        background-color: #d1ecf1;
        color: #0c5460;
    }

    .score-fair {
        background-color: #fff3cd;
        color: #856404;
    }

    .score-poor {
        background-color: #ffeaa7;
        color: #d63031;
    }

    .score-critical {
        background-color: #f8d7da;
        color: #721c24;
    }

    .severity-low {
        color: #28a745;
        font-weight: bold;
    }

    .severity-medium {
        color: #fd7e14;
        font-weight: bold;
    }

    .severity-high {
        color: #dc3545;
        font-weight: bold;
    }

    /* Fun Modal Styles */
    .bg-gradient-primary {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    }

    .cursor-pointer {
        cursor: pointer;
    }

    .violation-item {
        transition: all 0.3s ease;
        border-left: 4px solid transparent;
    }

    .violation-item:hover {
        background-color: #f8f9fa;
        border-left-color: #007bff;
        transform: translateX(5px);
    }

    .violation-item.speed {
        border-left-color: #dc3545;
    }

    .violation-item.harsh {
        border-left-color: #ffc107;
    }

    .violation-item.zone {
        border-left-color: #17a2b8;
    }

    .violation-badge {
        font-size: 0.75rem;
        padding: 0.2rem 0.4rem;
        border-radius: 0.25rem;
        font-weight: 600;
    }

    .violation-badge.speed {
        background-color: #f8d7da;
        color: #721c24;
    }

    .violation-badge.harsh {
        background-color: #fff3cd;
        color: #856404;
    }

    .violation-badge.zone {
        background-color: #d1ecf1;
        color: #0c5460;
    }

    .severity-indicator {
        width: 8px;
        height: 8px;
        border-radius: 50%;
        display: inline-block;
        margin-right: 8px;
    }

    .severity-low {
        background-color: #28a745;
    }

    .severity-medium {
        background-color: #ffc107;
    }

    .severity-high {
        background-color: #dc3545;
    }
</style>
@endsection

@section('scripts')
<script>
    var speedUnit = "";
    var lengthUnit = "";
    var currency = "";
    
    let reportData = [];
    let driverScoringTable;
    let violationDetailsTable;
    let driverList = [];
    
    // Initialize page state
    $('#driverOverviewCards').addClass('d-none');
    $('#driverScoringTableCard').addClass('d-none');
    $('#violationDetailsTableCard').addClass('d-none');
    $('#loadingState').addClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#chartsRow1').addClass('d-none');
    $('#chartsRow2').addClass('d-none');
    $('#driverComparisonRow').addClass('d-none');
    
    // Store original violation data for filtering
    let originalViolationData = [];
    
    // Global array for all violations
    let allViolations = [];

    onAppReady(function() {        
        loadDrivers().then(initTables).then(generateReport);
    });

    // Helper for duration calculation (minutes)
    function calculateDurationMinutes(start, end) {
        const startTime = new Date(start);
        const endTime = new Date(end);
        return Math.round((endTime - startTime) / 60000); // ms to minutes
    }

    // Process speed violations with duration and distance tracking
    function processSpeedViolations(deviceId, driverName, speedLimit, deviceLogs) {
        if (!Array.isArray(deviceLogs)) return;
        
        let currentViolation = null;
        let violationStartTime = null;
        let violationStartLocation = null;
        let totalDistance = 0;
        let lastLocation = null;
        
        // Sort logs by timestamp to ensure chronological order
        const sortedLogs = deviceLogs
            .filter(log => log.did === deviceId)
            .sort((a, b) => new Date(a.event_time || a.timestamp) - new Date(b.event_time || b.timestamp));
        
        sortedLogs.forEach((log, index) => {
            const logDetails = JSON.parse(log.log_details);
            const speed = logDetails.hiz;
            const currentTime = new Date(log.event_time || log.timestamp);
            const currentLocation = log.latitude && log.longitude ? { lat: parseFloat(log.latitude), lng: parseFloat(log.longitude) } : null;
            
            // Calculate distance if we have previous location
            if (lastLocation && currentLocation) {
                const distance = calculateDistance(lastLocation.lat, lastLocation.lng, currentLocation.lat, currentLocation.lng);
                totalDistance += distance;
            }
            
            if (speed > speedLimit) {
                // Speed violation detected
                if (!currentViolation) {
                    // Start new violation
                    violationStartTime = currentTime;
                    violationStartLocation = currentLocation;
                    currentViolation = {
                        device_id: deviceId,
                        driver_name: driverName,
                        violation_type: "speed",
                        timestamp: log.event_time || log.timestamp,
                        location: currentLocation ? `${currentLocation.lat},${currentLocation.lng}` : "",
                        details: {
                            speed: speed,
                            speed_limit: speedLimit,
                            start_time: violationStartTime,
                            start_location: violationStartLocation ? `${violationStartLocation.lat},${violationStartLocation.lng}` : "",
                            duration_minutes: 0,
                            distance_km: 0,
                            max_speed: speed,
                            avg_speed: speed,
                            speed_count: 1
                        },
                        severity: (speed - speedLimit > 20) ? "high" : (speed - speedLimit > 10) ? "medium" : "low"
                    };
                } else {
                    // Continue existing violation
                    currentViolation.details.max_speed = Math.max(currentViolation.details.max_speed, speed);
                    currentViolation.details.avg_speed = (currentViolation.details.avg_speed * currentViolation.details.speed_count + speed) / (currentViolation.details.speed_count + 1);
                    currentViolation.details.speed_count++;
                }
            } else {
                // Speed is now within limit - end violation if exists
                if (currentViolation) {
                    // Calculate final duration and distance
                    const durationMs = currentTime - violationStartTime;
                    const durationMinutes = Math.round((durationMs / (1000 * 60)) * 100) / 100; // Round to 2 decimal places
                    
                    currentViolation.details.duration_minutes = durationMinutes;
                    currentViolation.details.distance_km = Math.round(totalDistance * 100) / 100; // Round to 2 decimal places
                    currentViolation.details.end_time = currentTime;
                    currentViolation.details.end_location = currentLocation ? `${currentLocation.lat},${currentLocation.lng}` : "";
                    
                    // Add description for popup
                    currentViolation.details.description = `{{ __('Duration') }}: ${durationMinutes.toFixed(2)} min, {{ __('Distance') }}: ${currentViolation.details.distance_km.toFixed(2)} km, {{ __('Max Speed') }}: ${currentViolation.details.max_speed.toFixed(1)} km/h`;
                    
                    // Push to allViolations
                    allViolations.push(currentViolation);
                    
                    // Reset for next violation
                    currentViolation = null;
                    violationStartTime = null;
                    violationStartLocation = null;
                    totalDistance = 0;
                }
            }
            
            // Update last location for distance calculation
            if (currentLocation) {
                lastLocation = currentLocation;
            }
        });
        
        // Handle case where violation continues until end of logs
        if (currentViolation) {
            const finalTime = new Date(sortedLogs[sortedLogs.length - 1].event_time || sortedLogs[sortedLogs.length - 1].timestamp);
            const durationMs = finalTime - violationStartTime;
            const durationMinutes = Math.round((durationMs / (1000 * 60)) * 100) / 100;
            
            currentViolation.details.duration_minutes = durationMinutes;
            currentViolation.details.distance_km = Math.round(totalDistance * 100) / 100;
            currentViolation.details.end_time = finalTime;
            currentViolation.details.description = `{{ __('Duration') }}: ${durationMinutes.toFixed(2)} min, {{ __('Distance') }}: ${currentViolation.details.distance_km.toFixed(2)} km, {{ __('Max Speed') }}: ${currentViolation.details.max_speed.toFixed(1)} km/h`;
            
            allViolations.push(currentViolation);
        }
    }
    
    // Helper function to calculate distance between two points (Haversine formula)
    function calculateDistance(lat1, lon1, lat2, lon2) {
        const R = 6371; // Radius of the Earth in kilometers
        const dLat = (lat2 - lat1) * Math.PI / 180;
        const dLon = (lon2 - lon1) * Math.PI / 180;
        const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                  Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
                  Math.sin(dLon/2) * Math.sin(dLon/2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
        return R * c;
    }

    // Process harsh driving violations
    function processHarshDrivingViolations(deviceId, driverName, accelerationLimit, deviceLogs) {
        if (!Array.isArray(deviceLogs)) return;
        for (let i = 1; i < deviceLogs.length; i++) {
            const current = deviceLogs[i];
            const previous = deviceLogs[i - 1];
            if (current.did !== deviceId || previous.did !== deviceId) continue;
            const timeDiff = (new Date(current.event_time) - new Date(previous.event_time)) / 1000;
            if (timeDiff > 0) {
                const currentLogDetails = JSON.parse(current.log_details);
                const previousLogDetails = JSON.parse(previous.log_details);

                const speedDiff = parseFloat(currentLogDetails.hiz) - parseFloat(previousLogDetails.hiz);
                const acceleration = speedDiff / timeDiff;
                if (Math.abs(acceleration) > accelerationLimit) {
                    allViolations.push({
                        device_id: deviceId,
                        driver_name: driverName,
                        violation_type: "harsh",
                        timestamp: current.event_time,
                        location: current.latitude && current.longitude ? `${current.latitude},${current.longitude}` : "",
                        details: {
                            acceleration: acceleration,
                            event_type: acceleration > 0 ? 'acceleration' : 'deceleration',
                            speed: parseFloat(currentLogDetails.hiz),
                            previous_speed: parseFloat(previousLogDetails.hiz),
                            time_diff: timeDiff
                        },
                        severity: Math.abs(acceleration) > accelerationLimit * 1.5 ? 'high' : 'medium'
                    });
                }
            }
        }
    }

    // Process zone violations using entry/exit logic
    function processZoneViolations(zoneRawData) {
        let eventId = 1;
        let currentVisits = {};
        if (!Array.isArray(zoneRawData)) return;
        zoneRawData.forEach(record => {
            if (record.region_control !== '-') {
                const vehicleId = record.did;
                const visitKey = `region_${vehicleId}_${record.region_id}`;
                if (record.region_control === 'entry' && !currentVisits[visitKey]) {
                    // Entry event
                    const event = {
                        id: eventId++,
                        device_id: vehicleId,
                        driver_name: record.device_name,
                        violation_type: 'zone',
                        timestamp: record.event_time,
                        location: record.latitude && record.longitude ? `${record.latitude},${record.longitude}` : '',
                        details: {
                            region_id: record.region_id,
                            region_name: record.region_name,
                            event_type: 'entry',
                            region_control: record.region_status,
                            region_price: record.region_price,
                            log_id: record.log_id
                        },
                        severity: (record.region_status === 'no_entry') ? 'high' : 'low'
                    };
                    // Only push if region_status is 'no_entry' (violation)
                    if (record.region_status === 'no_entry') {
                        allViolations.push(event);
                    }
                    currentVisits[visitKey] = {
                        start_time: record.event_time,
                        event_id: event.id
                    };
                } else if (record.region_control === 'exit' && currentVisits[visitKey]) {
                    // Exit event
                    const visit = currentVisits[visitKey];
                    const duration = calculateDurationMinutes(visit.start_time, record.event_time);
                    const event = {
                        id: eventId++,
                        device_id: vehicleId,
                        driver_name: record.device_name,
                        violation_type: 'zone',
                        timestamp: record.event_time,
                        location: record.latitude && record.longitude ? `${record.latitude},${record.longitude}` : '',
                        details: {
                            region_id: record.region_id,
                            region_name: record.region_name,
                            event_type: 'exit',
                            region_control: record.region_status,
                            region_price: record.region_price,
                            duration_minutes: duration,
                            log_id: record.log_id
                        },
                        severity: (record.region_status === 'no_exit') ? 'high' : 'low'
                    };
                    // Only push if region_status is 'no_exit' (violation)
                    if (record.region_status === 'no_exit') {
                        allViolations.push(event);
                    }
                    delete currentVisits[visitKey];
                }
            }
        });
    }
    

    
    async function initTables() {
        // Initialize Driver Scoring Table
        driverScoringTable = createTabulator("#driver-scoring-table", {
            columns: [
                { title: "{{ __('Driver') }}", field: "driver_name", headerFilter: true },
                { title: "{{ __('Overall Score') }}", field: "overall_score", 
                    formatter: function(cell) {
                        const score = cell.getValue();
                        const status = getScoreStatus(score);
                        return `<span class="score-badge score-${status.class}">${score}%</span>`;
                    },
                    sorter: "number"
                },
                { title: "{{ __('Safety Score') }}", field: "safety_score", 
                    formatter: function(cell) { return cell.getValue() + '%'; },
                    sorter: "number"
                },
                { title: "{{ __('Efficiency Score') }}", field: "efficiency_score", 
                    formatter: function(cell) { return cell.getValue() + '%'; },
                    sorter: "number"
                },
                { title: "{{ __('Compliance Score') }}", field: "compliance_score", 
                    formatter: function(cell) { return cell.getValue() + '%'; },
                    sorter: "number"
                },
                { title: "{{ __('Violation Count') }}", field: "total_violations", 
                    formatter: function(cell) {
                        const violations = cell.getValue();
                        const data = cell.getRow().getData();
                        const driverName = data.driver_name || data.driver_id;
                        
                        if (violations > 0) {
                            return `<span class="badge text-white bg-danger cursor-pointer" 
                                       onclick="showViolationDetailsModal('${driverName}', 'all', ${violations})"
                                       style="cursor: pointer; transition: all 0.3s ease;"
                                       onmouseover="this.style.transform='scale(1.1)'"
                                       onmouseout="this.style.transform='scale(1)'"
                                       title="{{ __('Click to view details on map') }}">
                                       <i class="ti ti-eye me-1"></i>${violations}
                                   </span>`;
                        } else {
                            return violations;
                        }
                    },
                    sorter: "number"
                },
                { title: "{{ __('Actions') }}", formatter: function(cell) {
                    const data = cell.getRow().getData();
                    return `<button class="btn btn-sm btn-primary" onclick="viewDriverDetails('${data.driver_id}')">
                        <i class="ti ti-eye"></i> {{ __('Details') }}
                    </button>`;
                }}
            ]
        });

        // Initialize Violation Details Table
        violationDetailsTable = createTabulator("#violation-details-table", {
            paginationSize: 15,
            columns: [
                { title: "{{ __('Driver') }}", field: "driver_name", headerFilter: true },
                { title: "{{ __('Violation Type') }}", field: "violation_type", 
                    formatter: function(cell) {
                        const type = cell.getValue();
                        const typeInfo = getViolationTypeInfo(type);
                        return `<span class="badge text-white bg-${typeInfo.color}">${__(typeInfo.text)}</span>`;
                    }
                },
                { title: "{{ __('Violation Count') }}", field: "violation_count", 
                    formatter: function(cell) {
                        const count = cell.getValue();
                        const driverName = cell.getRow().getData().driver_name;
                        const violationType = cell.getRow().getData().violation_type;
                        return `<span class="badge text-white bg-danger cursor-pointer" 
                                   onclick="showViolationDetailsModal('${driverName}', '${violationType}', ${count})"
                                   style="cursor: pointer; transition: all 0.3s ease;"
                                   onmouseover="this.style.transform='scale(1.1)'"
                                   onmouseout="this.style.transform='scale(1)'">
                                   <i class="ti ti-eye me-1"></i>${count}
                               </span>`;
                    },
                    sorter: "number"
                },
                { title: "{{ __('Description') }}", field: "violation_description", 
                    formatter: function(cell) {
                        return `<span class="text-muted">${cell.getValue()}</span>`;
                    }
                },
                { title: "{{ __('Severity') }}", field: "severity", 
                    formatter: function(cell) {
                        const severity = cell.getValue();
                        const severityInfo = getSeverityInfo(severity);
                        return `<span class="severity-${severityInfo.class} badge text-white">${severityInfo.text}</span>`;
                    }
                }
            ]
        });
    }
    
    async function loadDrivers() {
        http.post({
            url: "/api/vehicles/list",
            loading: true
        }).then((response) => {
            if (response.success && response.data.vehicles) {
                const select = $('#driverSelect');
                select.empty();
                select.append('<option value="">{{ __("All Drivers") }}</option>');
                
                response.data.vehicles.forEach(vehicle => {
                    select.append(`<option value="${vehicle.did}">${vehicle.driver_name || vehicle.name}</option>`);
                });
                
                driverList = response.data.vehicles;
            }
        }).catch((error) => {
            console.error('{{ __("Error loading drivers") }}:', error);
        });
    }
    
    async function generateReport() {
        const driverId = $('#driverSelect').val();
        const startDate = $('#startDate').val();
        const endDate = $('#endDate').val();
        
        if (!startDate || !endDate) {
            showNotification("error", "{{ __('Please select date range') }}");
            return;
        }
        
        // Show loading state and hide no vehicle state
        $('#loadingState').removeClass('d-none');
        $('#noVehicleState').addClass('d-none');
        $('#driverOverviewCards').addClass('d-none');
        $('#driverScoringTableCard').addClass('d-none');
        $('#violationDetailsTableCard').addClass('d-none');
        $('#chartsRow1').addClass('d-none');
        $('#chartsRow2').addClass('d-none');
        $('#driverComparisonRow').addClass('d-none');
        $('#noDataState').addClass('d-none');

        $('#helpSection').removeClass('show');
        
        const requestData = {
            device_id: driverId,
            start_date: startDate,
            end_date: endDate
        };
        
        http.post({
            url: '/api/reports/driver-scoring',
            data: requestData,
            loading: false
        }).then((response) => {
            if (response.success && response.data) {
                reportData = response.data;

                //console.log(reportData);

                var message = __('#name# used #value# records').replace('#name#', __('Driver Scoring')).replace('#value#', reportData.zone_query_info.total_raw_records);
                showReportDataProcessingInfo(message);

                processRawData(response.data);
                
                $('#loadingState').addClass('d-none');
                $('#driverOverviewCards').removeClass('d-none');
                $('#driverScoringTableCard').removeClass('d-none');
                $('#violationDetailsTableCard').removeClass('d-none');
                $('#chartsRow1').removeClass('d-none');
                $('#chartsRow2').removeClass('d-none');
                $('#driverComparisonRow').removeClass('d-none');
            } else {
                showNotification("error", response.message || "{{ __('Error generating report') }}");
                $('#loadingState').addClass('d-none');
                $('#noVehicleState').addClass('d-none');
                $('#noDataState').removeClass('d-none');
            }
        }).catch((error) => {
            console.error("Report generation error:", error);
            showNotification("error", "{{ __('Error generating report') }}");
            $('#loadingState').addClass('d-none');
            $('#noVehicleState').addClass('d-none');
            $('#noDataState').removeClass('d-none');
        });
    }
    
    function processRawData(rawData) {
        // Set units
        speedUnit = sessionSettings.speedUnit || 'km/h';
        lengthUnit = sessionSettings.lengthUnit || 'km';
        currency = sessionSettings.currency || 'TRY';

        const deviceAlarmSettings = rawData.device_alarm_settings || [];
        const zoneRawData = rawData.zone_data || [];
        const zoneList = rawData.zone_list || [];
        const deviceLogs = rawData.device_logs || [];

        // 1. Clear the global violations array
        allViolations = [];

        // 2. Process speed and harsh driving violations for each device
        deviceAlarmSettings.forEach(device => {
            const deviceId = device.did;
            const driverName = device.name ?? device.did;
            // Find logs for this device
            const logsForDevice = zoneRawData.filter(log => log.did === deviceId);

            // Get speed limit from alarm settings (default to 80)
            let speedLimit = 80;
            const speedAlarm = deviceAlarmSettings.find(s => s.did === deviceId && s.aid === "304");
            if (speedAlarm && speedAlarm.alarm_settings) {
                const settings = JSON.parse(speedAlarm.alarm_settings);
                speedLimit = parseInt(settings.speed_limit) || 80;
            }

            // Get harsh threshold from alarm settings (default to 15)
            let harshThreshold = 15;
            const harshAlarm = deviceAlarmSettings.find(s => s.did === deviceId && s.aid === "305");
            if (harshAlarm && harshAlarm.alarm_settings) {
                const settings = JSON.parse(harshAlarm.alarm_settings);
                harshThreshold = parseInt(settings.p1) || 15;
            }

            // Call the process functions
            processSpeedViolations(deviceId, driverName, speedLimit, logsForDevice);
            processHarshDrivingViolations(deviceId, driverName, harshThreshold, logsForDevice);
        });

        // 3. Process zone violations (all at once)
        processZoneViolations(zoneRawData);

        // Process driver scoring data
        const driverScoringData = [];
        const violationDetails = [];
        const scoreDistribution = {
            excellent: 0,
            good: 0,
            fair: 0,
            poor: 0,
            critical: 0
        };

        // Get unique devices (since deviceAlarmSettings has multiple rows per device)
        const uniqueDevices = [];
        const deviceMap = new Map();
        
        deviceAlarmSettings.forEach(device => {
            if (!deviceMap.has(device.did)) {
                deviceMap.set(device.did, device);
                uniqueDevices.push(device);
            }
        });

        // Process each unique driver/vehicle
        uniqueDevices.forEach(device => {
            const driverScore = calculateDriverScore(device, zoneRawData, zoneList);
            driverScoringData.push(driverScore);
            
            // Update distribution
            const status = getScoreStatus(driverScore.overall_score);
            scoreDistribution[status.class]++;
            
            // Add violations to details
            if (driverScore.violations && driverScore.violations.length > 0) {
                driverScore.violations.forEach(violation => {
                    violationDetails.push({
                        driver_id: driverScore.driver_id,
                        driver_name: driverScore.driver_name,
                        violation_type: violation.type,
                        violation_count: violation.count,
                        violation_description: violation.description,
                        severity: getViolationSeverity(violation.type, violation.count),
                        date: new Date().toISOString().split('T')[0] // Placeholder date
                    });
                });
            }
        });

        // Update UI
        updateDriverOverview(driverScoringData);
        updateTables(driverScoringData, violationDetails);
        createCharts(driverScoringData, scoreDistribution);
    }
    
    function calculateDriverScore(device, zoneRawData, zoneList) {

        // Extract alarm settings - device_alarm_settings contains multiple rows per device
        // We need to find the specific alarm settings for this device
        const deviceAlarmSettings = reportData.device_alarm_settings || [];
        const deviceSpeedSettings = deviceAlarmSettings.find(s => s.did === device.did && s.aid === "304") || {};
        const deviceHarshSettings = deviceAlarmSettings.find(s => s.did === device.did && s.aid === "305") || {};
        const deviceWorkingHoursSettings = deviceAlarmSettings.find(s => s.did === device.did && s.aid === "307") || {};
        
        // Parse alarm settings JSON
        const speedSettings = deviceSpeedSettings.alarm_settings ? JSON.parse(deviceSpeedSettings.alarm_settings) : {};
        const harshSettings = deviceHarshSettings.alarm_settings ? JSON.parse(deviceHarshSettings.alarm_settings) : {};
        const workingHoursSettings = deviceWorkingHoursSettings.alarm_settings ? JSON.parse(deviceWorkingHoursSettings.alarm_settings) : {};
        
        // Get device settings for fuel consumption
        const deviceSettings = device.device_settings ? JSON.parse(device.device_settings) : {};
        
        // Calculate scores (implement your scoring logic here)
        const safetyScore = calculateSafetyScore(device);
        const efficiencyScore = calculateEfficiencyScore(device, deviceSettings);
        const complianceScore = calculateComplianceScore(device);
        
        // Calculate overall score (weighted average)
        const overallScore = Math.round((safetyScore * 0.4 + efficiencyScore * 0.3 + complianceScore * 0.3));
        
        // Count violations
        const speedViolations = allViolations.filter(v => v.violation_type === 'speed' && v.device_id === device.did).length;
        const harshEvents = allViolations.filter(v => v.violation_type === 'harsh' && v.device_id === device.did).length;
        const zoneViolations = allViolations.filter(v => v.violation_type === 'zone' && v.device_id === device.did).length;
        const workingHoursViolations = allViolations.filter(v => v.violation_type === 'working_hours' && v.device_id === device.did).length;
        
        const totalViolations = speedViolations + harshEvents + zoneViolations + workingHoursViolations;
        
        // Create violations array for details
        const violations = [];
        
        if (speedViolations > 0) {
            violations.push({
                type: 'speed',
                count: speedViolations,
                description: `${speedViolations} {{ __("speed violations") }} (${(speedSettings.speed_limit || 80).toFixed(1)} km/h limit)`
            });
        }
        
        if (harshEvents > 0) {
            violations.push({
                type: 'harsh',
                count: harshEvents,
                description: `${harshEvents} {{ __("harsh driving events") }} (${harshSettings.p1 || 15}/${harshSettings.p2 || 15} limit)`
            });
        }
        
        if (zoneViolations > 0) {
            violations.push({
                type: 'zone',
                count: zoneViolations,
                description: `${zoneViolations} {{ __("zone violations") }}`
            });
        }
        
        if (workingHoursViolations > 0) {
            violations.push({
                type: 'working_hours',
                count: workingHoursViolations,
                description: `${workingHoursViolations} {{ __("working hours violations") }}`
            });
        }
        
        return {
            driver_id: device.did,
            driver_name: device.name,
            overall_score: overallScore,
            safety_score: safetyScore,
            efficiency_score: efficiencyScore,
            compliance_score: complianceScore,
            total_violations: totalViolations,
            last_trip: device.last_trip || '-',
            violations: violations,
            // Additional data for charts
            speed_violations: speedViolations,
            harsh_events: harshEvents,
            zone_violations: zoneViolations,
            working_hours_violations: workingHoursViolations,
            fuel_consumption: deviceSettings.lt ? parseFloat(deviceSettings.lt).toFixed(1) : '10.0'
        };
    }
    
    function calculateSafetyScore(device) {
        const speedViolationCount = allViolations.filter(v => v.violation_type === 'speed' && v.device_id === device.did).length;
        const harshViolationCount = allViolations.filter(v => v.violation_type === 'harsh' && v.device_id === device.did).length;
        let safetyScore = 100;
        // Deduct points for speed violations (5 points per violation, max 40 points)
        const speedDeduction = Math.min(speedViolationCount * 5, 40);
        safetyScore -= speedDeduction;
        // Deduct points for harsh driving (3 points per event, max 30 points)
        const harshDeduction = Math.min(harshViolationCount * 3, 30);
        safetyScore -= harshDeduction;
        return Math.max(safetyScore, 0);
    }
    
    function calculateEfficiencyScore(device, deviceSettings) {
        let efficiencyScore = 100; // Start with perfect score
        
        // Fuel consumption calculation
        const fuelConsumption = deviceSettings.lt ? parseFloat(deviceSettings.lt) : 10.0; // Default 10.0L/100km
        
        // Score based on fuel efficiency (lower consumption = higher score)
        // Excellent: < 8L/100km, Good: 8-12L/100km, Fair: 12-16L/100km, Poor: > 16L/100km
        if (fuelConsumption < 8) {
            efficiencyScore = 95; // Excellent
        } else if (fuelConsumption <= 12) {
            efficiencyScore = 85; // Good
        } else if (fuelConsumption <= 16) {
            efficiencyScore = 70; // Fair
        } else {
            efficiencyScore = 50; // Poor
        }
        
        return efficiencyScore;
    }
    
    function calculateComplianceScore(device) {
        const zoneViolationCount = allViolations.filter(v => v.violation_type === 'zone' && v.device_id === device.did).length;
        let complianceScore = 100;
        // Deduct points for zone violations (4 points per violation, max 35 points)
        const zoneDeduction = Math.min(zoneViolationCount * 4, 35);
        complianceScore -= zoneDeduction;
        // Working hours violations can be added similarly if available
        return Math.max(complianceScore, 0);
    }
    
    function getScoreStatus(score) {
        if (score >= 90) return { class: 'excellent', label: '{{ __("Excellent") }}', color: '#28a745' };
        if (score >= 80) return { class: 'good', label: '{{ __("Good") }}', color: '#17a2b8' };
        if (score >= 70) return { class: 'fair', label: '{{ __("Fair") }}', color: '#ffc107' };
        if (score >= 60) return { class: 'poor', label: '{{ __("Poor") }}', color: '#fd7e14' };
        return { class: 'critical', label: '{{ __("Critical") }}', color: '#dc3545' };
    }
        
    function getViolationSeverity(type, count) {
        // Determine severity based on violation type and count
        switch (type) {
            case 'speed':
                if (count >= 5) return 'high';
                if (count >= 3) return 'medium';
                return 'low';
            case 'harsh':
                if (count >= 3) return 'high';
                if (count >= 2) return 'medium';
                return 'low';
            case 'zone':
                if (count >= 4) return 'high';
                if (count >= 2) return 'medium';
                return 'low';
            case 'working_hours':
                if (count >= 2) return 'high';
                return 'low';
            default:
                return 'low';
        }
    }
    
    function updateDriverOverview(driverData) {
        if (driverData.length === 0) return;
        
        const avgScore = Math.round(driverData.reduce((sum, driver) => sum + driver.overall_score, 0) / driverData.length);
        const topPerformers = driverData.filter(d => d.overall_score >= 90).length;
        const needsImprovement = driverData.filter(d => d.overall_score < 70).length;
        const totalViolations = driverData.reduce((sum, driver) => sum + driver.total_violations, 0);
        
        $('#averageDriverScore').text(avgScore + '%');
        $('#topPerformers').text(topPerformers);
        $('#needsImprovement').text(needsImprovement);
        $('#totalViolations').text(totalViolations);
        
        $('#scoreBreakdown').text(`${driverData.length} {{ __('drivers analyzed') }}`);
        $('#topPerformersBreakdown').text(`${Math.round(topPerformers/driverData.length*100)}% {{ __('of drivers') }}`);
        $('#improvementBreakdown').text(`${Math.round(needsImprovement/driverData.length*100)}% {{ __('of drivers') }}`);
        $('#violationsBreakdown').text(`${Math.round(totalViolations/driverData.length)} {{ __('per driver') }}`);
    }
    
    function updateTables(driverData, violationDetails) {
        driverScoringTable.setData(driverData);
        violationDetailsTable.setData(violationDetails);
        
        // Store original violation data for filtering
        originalViolationData = [...violationDetails];
    }
    
    function createCharts(driverData, scoreDistribution) {
        createDriverScoreDistributionChart(scoreDistribution);
        createScoreBreakdownChart(driverData);
        createViolationTrendsChart(); // now uses real data
        createHourlyPerformanceChart(); // now uses real data
        createDriverComparisonChart(driverData);
    }
    
    function createDriverScoreDistributionChart(distribution) {
        if (typeof ApexCharts === 'undefined') {
            console.error('ApexCharts is not loaded');
            return;
        }
        
        const options = {
            series: Object.values(distribution || {}),
            chart: {
                type: 'donut',
                height: 300
            },
            labels: ['{{ __("Excellent") }}', '{{ __("Good") }}', '{{ __("Fair") }}', '{{ __("Poor") }}', '{{ __("Critical") }}'],
            colors: ['#28a745', '#17a2b8', '#ffc107', '#fd7e14', '#dc3545'],
            legend: {
                position: 'bottom'
            },
            plotOptions: {
                pie: {
                    donut: {
                        size: '70%'
                    }
                }
            }
        };
        
        if (window.driverScoreChart && typeof window.driverScoreChart.destroy === 'function') {
            window.driverScoreChart.destroy();
        }
        
        const chartElement = document.querySelector("#driverScoreChart");
        if (chartElement) {
            window.driverScoreChart = new ApexCharts(chartElement, options);
            window.driverScoreChart.render();
        }
    }
    
    function createScoreBreakdownChart(drivers) {
        if (typeof ApexCharts === 'undefined') {
            console.error('ApexCharts is not loaded');
            return;
        }
        
        const chartElement = document.querySelector('#scoreBreakdownChart');
        if (!chartElement) return;
        
        if (!drivers || drivers.length === 0) {
            chartElement.innerHTML = '<div class="text-center text-muted py-4">{{ __("No driver data available") }}</div>';
            return;
        }
        
        const categories = drivers.map(d => d.driver_name);
        const safetyScores = drivers.map(d => d.safety_score);
        const efficiencyScores = drivers.map(d => d.efficiency_score);
        const complianceScores = drivers.map(d => d.compliance_score);
        const overallScores = drivers.map(d => d.overall_score);
        
        const options = {
            chart: {
                type: 'bar',
                height: 300,
                stacked: false,
                toolbar: { show: false }
            },
            series: [
                { name: '{{ __("Safety Score") }}', data: safetyScores },
                { name: '{{ __("Efficiency Score") }}', data: efficiencyScores },
                { name: '{{ __("Compliance Score") }}', data: complianceScores },
                { name: '{{ __("Overall Score") }}', data: overallScores }
            ],
            xaxis: {
                categories: categories,
                labels: { rotate: -45 }
            },
            yaxis: {
                title: { text: '{{ __("Score (%)") }}' },
                min: 0,
                max: 100
            },
            colors: ['#28a745', '#17a2b8', '#ffc107', '#dc3545'],
            legend: { position: 'top' },
            plotOptions: {
                bar: {
                    horizontal: false,
                    dataLabels: { position: 'top' },
                    columnWidth: '50%'
                }
            },
            dataLabels: {
                enabled: true,
                formatter: function(val) {
                    return val + '%';
                }
            }
        };
        
        if (window.scoreBreakdownChart && typeof window.scoreBreakdownChart.destroy === 'function') {
            window.scoreBreakdownChart.destroy();
        }
        
        window.scoreBreakdownChart = new ApexCharts(chartElement, options);
        window.scoreBreakdownChart.render();
    }
    
    // === Helper functions for chart data aggregation ===
    function getDayOfWeek(dateStr) {
        const date = new Date(dateStr);
        return (date.getDay() + 6) % 7; // 0=Monday, 6=Sunday
    }

    function aggregateViolationsByDay(violations) {
        const speed = [0,0,0,0,0,0,0];
        const harsh = [0,0,0,0,0,0,0];
        const zone = [0,0,0,0,0,0,0];
        violations.forEach(v => {
            if (!v.date) return;
            const dayIdx = getDayOfWeek(v.date);
            if (v.violation_type === 'speed') speed[dayIdx] += v.violation_count;
            if (v.violation_type === 'harsh') harsh[dayIdx] += v.violation_count;
            if (v.violation_type === 'zone') zone[dayIdx] += v.violation_count;
        });
        return { speed, harsh, zone };
    }

    function getHourOfDay(dateStr) {
        const date = new Date(dateStr);
        return date.getHours();
    }

    function aggregateViolationsByHour(violations) {
        const hourly = Array(24).fill(0);
        
        allViolations.forEach(v => {
            // Use event_time if available, otherwise fallback to date
            const timeStr = v.timestamp;
            if (!timeStr) return;
            const hour = timeStr.split(' ')[1].split(':')[0];
            hourly[hour] += v.violation_count || 1;
        });
        return hourly;
    }

    function aggregateViolationsByDate(violations) {
        const dateMap = {};
        violations.forEach(v => {
            if (!v.timestamp) return;
            const dateStr = v.timestamp.split(' ')[0]; // "YYYY-MM-DD"
            if (!dateMap[dateStr]) {
                dateMap[dateStr] = { speed: 0, harsh: 0, zone: 0 };
            }
            if (v.violation_type === 'speed') dateMap[dateStr].speed += 1;
            if (v.violation_type === 'harsh') dateMap[dateStr].harsh += 1;
            if (v.violation_type === 'zone') dateMap[dateStr].zone += 1;
        });
        // Sort dates
        const dates = Object.keys(dateMap).sort();
        const speed = dates.map(d => dateMap[d].speed);
        const harsh = dates.map(d => dateMap[d].harsh);
        const zone = dates.map(d => dateMap[d].zone);
        return { dates, speed, harsh, zone };
    }

    // === Update createViolationTrendsChart to use real data ===
    function createViolationTrendsChart() {
        if (typeof ApexCharts === 'undefined') {
            console.error('ApexCharts is not loaded');
            return;
        }
        const { dates, speed, harsh, zone } = aggregateViolationsByDate(allViolations);

        // Format dates for display
        const formattedDates = dates.map(date => dateFormatter(date));

        const options = {
            series: [
                { name: '{{ __("Speed Violations") }}', data: speed, yAxisIndex: 0 },
                { name: '{{ __("Harsh Driving") }}', data: harsh, yAxisIndex: 1 },
                { name: '{{ __("Zone Violations") }}', data: zone, yAxisIndex: 2 }
            ],
            chart: {
                type: 'line',
                height: 300
            },
            xaxis: {
                categories: formattedDates,
                title: { text: '{{ __("Date") }}' }
            },
            yaxis: [
                {
                    title: { text: '{{ __("Harsh/Zone Violations") }}' }
                },
                {
                    opposite: true,
                    title: { text: '{{ __("Speed Violations") }}' }
                }
            ],
            colors: ['#dc3545', '#fd7e14', '#ffc107']
        };
        if (window.violationTrendsChart && typeof window.violationTrendsChart.destroy === 'function') {
            window.violationTrendsChart.destroy();
        }
        const chartElement = document.querySelector("#violationTrendsChart");
        if (chartElement) {
            window.violationTrendsChart = new ApexCharts(chartElement, options);
            window.violationTrendsChart.render();
        }
    }

    // === Update createHourlyPerformanceChart to use real data ===
    function createHourlyPerformanceChart() {
        if (typeof ApexCharts === 'undefined') {
            console.error('ApexCharts is not loaded');
            return;
        }
        const hours = Array.from({length: 24}, (_, i) => i + ':00');
        const hourlyData = aggregateViolationsByHour(originalViolationData);
        const options = {
            series: [{ name: '{{ __("Violations") }}', data: hourlyData }],
            chart: {
                type: 'bar',
                height: 300
            },
            xaxis: {
                categories: hours
            },
            colors: ['#fd7e14']
        };
        if (window.hourlyPerformanceChart && typeof window.hourlyPerformanceChart.destroy === 'function') {
            window.hourlyPerformanceChart.destroy();
        }
        const chartElement = document.querySelector("#hourlyPerformanceChart");
        if (chartElement) {
            window.hourlyPerformanceChart = new ApexCharts(chartElement, options);
            window.hourlyPerformanceChart.render();
        }
    }
    
    function createDriverComparisonChart(drivers) {
        if (typeof ApexCharts === 'undefined') {
            console.error('ApexCharts is not loaded');
            return;
        }
        
        const categories = drivers.map(d => d.driver_name);
        const scores = drivers.map(d => d.overall_score);
        
        const options = {
            chart: {
                type: 'bar',
                height: 400,
                toolbar: { show: false }
            },
            series: [{ name: '{{ __("Driver Score") }}', data: scores }],
            xaxis: {
                categories: categories
            },
            plotOptions: {
                bar: { horizontal: true }
            },
            colors: ['#28a745'],
            dataLabels: {
                enabled: true,
                formatter: function(val) {
                    return val + '%';
                }
            }
        };
        
        if (window.driverComparisonChart && typeof window.driverComparisonChart.destroy === 'function') {
            window.driverComparisonChart.destroy();
        }
        
        const chartElement = document.querySelector("#driverComparisonChart");
        if (chartElement) {
            window.driverComparisonChart = new ApexCharts(chartElement, options);
            window.driverComparisonChart.render();
        }
    }
    
    function getScoreStatus(score) {
        if (score >= 90) return { class: 'excellent', text: '{{ __("Excellent") }}' };
        if (score >= 80) return { class: 'good', text: '{{ __("Good") }}' };
        if (score >= 70) return { class: 'fair', text: '{{ __("Fair") }}' };
        if (score >= 60) return { class: 'poor', text: '{{ __("Poor") }}' };
        return { class: 'critical', text: '{{ __("Critical") }}' };
    }
    
    function getViolationTypeInfo(type) {
        const types = {
            'speed': { color: 'danger', text: '{{ __("Speed Violation") }}' },
            'harsh': { color: 'warning', text: '{{ __("Harsh Driving") }}' },
            'zone': { color: 'info', text: '{{ __("Zone Violation") }}' },
            'working_hours': { color: 'primary', text: '{{ __("Working Hours") }}' }
        };
        return types[type] || { color: 'secondary', text: type };
    }
    
    function getSeverityInfo(severity) {
        const severities = {
            'low': { class: 'low', text: '{{ __("Low") }}' },
            'medium': { class: 'medium', text: '{{ __("Medium") }}' },
            'high': { class: 'high', text: '{{ __("High") }}' }
        };
        return severities[severity] || { class: 'low', text: severity };
    }
    
    function viewDriverDetails(driverId) {
        // Get driver from the processed driver scoring data, not raw reportData
        const driver = driverScoringTable.getData().find(d => d.driver_id === driverId);
        if (!driver) return;
        
        // Filter violation details table to show only violations for this driver
        const driverViolations = originalViolationData.filter(violation => violation.driver_id === driverId);
        
        // Create modal content with driver information and violations
        const modalContent = `
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="ti ti-user me-2"></i>
                    {{ __('Driver Details') }} - ${driver.driver_name}
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <!-- Driver Information Section -->
                <div class="row mb-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-body">
                                <h6 class="card-title">{{ __('Driver Information') }}</h6>
                                <p class="mb-1"><strong>{{ __('Name') }}:</strong> ${driver.driver_name}</p>
                                <p class="mb-1"><strong>{{ __('Driver ID') }}:</strong> ${driver.driver_id}</p>
                                <p class="mb-0"><strong>{{ __('Violation Count') }}:</strong> 
                                    <span class="badge text-white bg-${driver.total_violations > 0 ? 'danger' : 'success'}">${driver.total_violations}</span>
                                </p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-body">
                                <h6 class="card-title">{{ __('Score Breakdown') }}</h6>
                                <p class="mb-1">
                                    <strong>{{ __('Overall Score') }}:</strong> 
                                    <span class="score-badge score-${getScoreStatus(driver.overall_score).class}">${driver.overall_score}%</span>
                                </p>
                                <p class="mb-1"><strong>{{ __('Safety Score') }}:</strong> ${driver.safety_score}%</p>
                                <p class="mb-1"><strong>{{ __('Efficiency Score') }}:</strong> ${driver.efficiency_score}%</p>
                                <p class="mb-0"><strong>{{ __('Compliance Score') }}:</strong> ${driver.compliance_score}%</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Violations Section -->
                <div class="card">
                    <div class="card-header">
                        <h6 class="card-title mb-0">
                            <i class="ti ti-alert-triangle me-2"></i>
                            {{ __('Violation Details') }}
                            <span class="badge text-white bg-secondary ms-2">${driverViolations.length}</span>
                        </h6>
                    </div>
                    <div class="card-body p-0">
                        ${driverViolations.length > 0 ? `
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>{{ __('Type') }}</th>
                                            <th>{{ __('Count') }}</th>
                                            <th>{{ __('Description') }}</th>
                                            <th>{{ __('Severity') }}</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        ${driverViolations.map(violation => `
                                            <tr>
                                                <td>
                                                    <span class="badge text-white bg-${getViolationTypeInfo(violation.violation_type).color}">
                                                        ${getViolationTypeInfo(violation.violation_type).text}
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="badge text-white bg-danger">${violation.violation_count}</span>
                                                </td>
                                                <td>
                                                    <span class="text-muted">${violation.violation_description}</span>
                                                </td>
                                                <td>
                                                    <span class="severity-${getSeverityInfo(violation.severity).class} badge text-white">
                                                        ${getSeverityInfo(violation.severity).text}
                                                    </span>
                                                </td>
                                            </tr>
                                        `).join('')}
                                    </tbody>
                                </table>
                            </div>
                        ` : `
                            <div class="text-center text-muted py-4">
                                <i class="ti ti-check-circle text-success" style="font-size: 2rem;"></i>
                                <p class="mt-2 mb-0">{{ __('No violations found for this driver') }}</p>
                            </div>
                        `}
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="ti ti-x me-1"></i>
                    {{ __('Close') }}
                </button>
            </div>
        `;
        
        // Update modal content and show
        $('#driverDetailsModal .modal-content').html(modalContent);
        $('#driverDetailsModal').modal('show');
    }

    // === Violation Details Modal Functions ===
    let violationMap = null;
    let currentViolations = [];
    let currentFilter = 'all';

    function showViolationDetailsModal(driverName, violationType, count) {
        // Filter violations based on driver and type
        let filteredViolations = allViolations.filter(v => 
            v.driver_name === driverName && 
            (violationType === 'all' || v.violation_type === violationType)
        );

        // If no specific type, get all violations for this driver
        if (violationType === 'all') {
            filteredViolations = allViolations.filter(v => v.driver_name === driverName);
        }

        currentViolations = filteredViolations;
        currentFilter = 'all';

        // Update modal title
        $('#violationModalTitle').html(`
            <span class="text-white">${driverName}</span>
            <span class="badge bg-light text-dark ms-2">${count} {{ __('violations') }}</span>
        `);

        // Update summary counts
        $('#totalViolationsCount').text(filteredViolations.length);
        
        // Count unique locations
        const uniqueLocations = new Set();
        filteredViolations.forEach(v => {
            if (v.location) {
                uniqueLocations.add(v.location);
            }
        });
        $('#uniqueLocationsCount').text(uniqueLocations.size);

        // Initialize map
        initializeViolationMap(filteredViolations);

        // Populate violations list
        populateViolationsList(filteredViolations);

        // Show/hide filter div based on violation type
        if (violationType === 'all') {
            $('#violationTypeFilterDiv').show();
        } else {
            $('#violationTypeFilterDiv').hide();
        }

        // Show modal
        $('#violationDetailsModal').modal('show');
    }

    function initializeViolationMap(violations) {
        
        // Initialize map using project's LeafletMap class
        if (!violationMap) {
            violationMap = new LeafletMap();
            violationMap.initMap("violationMap", {
            });
        }   
        else {
            violationMap.removeMarkers();
        }
        // Convert violations to marker format for project's LeafletMap
        const markerModels = violations
            .filter(v => v.location)
            .map((violation, index) => {
                const [lat, lng] = violation.location.split(',').map(Number);
                if (isNaN(lat) || isNaN(lng)) return null;

                // Color coding based on violation type
                const colors = {
                    speed: '#dc3545',    // Red
                    harsh: '#ffc107',    // Yellow
                    zone: '#17a2b8'      // Blue
                };

                // Icon size based on severity
                const sizes = {
                    low: 16,
                    medium: 20,
                    high: 24
                };

                return {
                    id: `violation_${index}`,
                    coord: [lat, lng],
                    //label: `${violation.driver_name} - ${getViolationTypeInfo(violation.violation_type).text}`,
                    popup: createViolationPopup(violation),
                    icon: {
                        name: "fa-solid fa-circle",
                        color: colors[violation.violation_type] || '#6c757d',
                        width: sizes[violation.severity] || 20,
                        className: `violation-marker-${violation.violation_type}`
                    }
                };
            })
            .filter(marker => marker !== null);

        // Add markers to map
        if (markerModels.length > 0) {
            violationMap.pinMarkers(markerModels);
            
            // Use shorter delays for better responsiveness
            setTimeout(() => violationMap.refreshMap(), 200);
            setTimeout(() => violationMap.fitBounds(), 300);
        }
    }

    function createViolationPopup(violation) {
        const typeInfo = getViolationTypeInfo(violation.violation_type);
        const severityInfo = getSeverityInfo(violation.severity);
        const timestamp = new Date(violation.timestamp).toLocaleString();
        let details = '';
        if (violation.details) {
            if (violation.violation_type === 'speed') {
                details = `{{ __('Speed') }}: ${violation.details.speed.toFixed(1)} km/h<br>{{ __('Limit') }}: ${parseFloat(violation.details.speed_limit, 1).toFixed(1)} km/h`;
                if (violation.details.description) {
                    details += `<br><br><strong>{{ __('Violation Details') }}:</strong><br>${violation.details.description}`;
                }
                if (violation.details.start_location && violation.details.end_location) {
                    details += `<br><br><strong>{{ __('Route') }}:</strong><br>{{ __('Start') }}: ${violation.details.start_location}<br>{{ __('End') }}: ${violation.details.end_location}`;
                }
            } else if (violation.violation_type === 'harsh') {
                details = `{{ __('Acceleration') }}: ${violation.details.acceleration.toFixed(1)} km/h²<br>{{ __('Event Type') }}: ${violation.details.event_type}`;
            } else if (violation.violation_type === 'zone') {
                details = `{{ __('Zone') }} : ${violation.details.region_name || '-'}`;
                details += `<br>{{ __('Violation Type') }} : ${__(violation.details.event_type) || '-'}`;
                details += `<br>{{ __('Price') }} : ${violation.details.region_price || '-'}`;
            }
        }

        return `
            <div class="p-2" style="min-width: 200px;">
                <h6 class="mb-2">
                    <span class="badge bg-${typeInfo.color} text-white">${typeInfo.text}</span>
                </h6>
                <p class="mb-1"><strong>{{ __('Driver') }}:</strong> ${violation.driver_name}</p>
                <p class="mb-1"><strong>{{ __('Time') }}:</strong> ${timestamp}</p>
                <p class="mb-1"><strong>{{ __('Severity') }}:</strong> 
                    <span class="severity-${severityInfo.class} badge text-white">${severityInfo.text}</span>
                </p>
                ${details ? `<p class="mb-0"><strong>{{ __('Details') }}:</strong><br>${details}</p>` : ''}
            </div>
        `;
    }

    function populateViolationsList(violations) {
        const listContainer = $('#violationsList');
        listContainer.empty();

        if (violations.length === 0) {
            listContainer.html(`
                <div class="text-center text-muted py-4">
                    <i class="ti ti-check-circle text-success" style="font-size: 2rem;"></i>
                    <p class="mt-2 mb-0">{{ __('No violations found') }}</p>
                </div>
            `);
            return;
        }

        violations.forEach((violation, index) => {
            const typeInfo = getViolationTypeInfo(violation.violation_type);
            const severityInfo = getSeverityInfo(violation.severity);
            const timestamp = new Date(violation.timestamp).toLocaleString();

            const listItem = $(`
                <div class="list-group-item violation-item ${violation.violation_type}" 
                     onclick="highlightViolationOnMap(${index})" 
                     style="cursor: pointer;">
                    <div class="d-flex justify-content-between align-items-start">
                        <div class="flex-grow-1">
                            <div class="d-flex align-items-center mb-1">
                                <span class="severity-indicator severity-${severityInfo.class}"></span>
                                <span class="violation-badge badge bg-${typeInfo.color} text-white ${violation.violation_type}">
                                    ${typeInfo.text}
                                </span>
                                <small class="text-muted ms-auto">${timestamp}</small>
                            </div>
                            <div class="small text-muted">
                                <strong>{{ __('Driver') }}:</strong> ${violation.driver_name}
                            </div>
                            ${violation.location ? `
                                <div class="small text-muted">
                                    <strong>{{ __('Location') }}:</strong> ${violation.location}
                                </div>
                            ` : ''}
                        </div>
                    </div>
                </div>
            `);

            listContainer.append(listItem);
        });

        if (violationMap) {
            setTimeout(() => violationMap.refreshMap(), 200);
            setTimeout(() => violationMap.fitBounds(), 300);
        }
    }

    function highlightViolationOnMap(index) {
        if (!violationMap || !currentViolations[index]) return;

        const violation = currentViolations[index];
        if (!violation.location) return;

        const [lat, lng] = violation.location.split(',').map(Number);
        
        // Focus on the marker using project's method
        violationMap.focusMarker(`violation_${index}`);
        
        // Highlight in list
        highlightViolationInList(index);
    }

    function highlightViolationInList(index) {
        // Remove previous highlights
        $('#violationsList .list-group-item').removeClass('active');
        
        // Add highlight to current item
        const item = $('#violationsList .list-group-item').eq(index);
        item.addClass('active');
        
        // Scroll to item
        item[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
    }

    function exportViolationData() {
        if (currentViolations.length === 0) {
            showNotification("warning", "{{ __('No data to export') }}");
            return;
        }

        // Convert violations to CSV format
        const headers = ['{{ __("Driver") }}', '{{ __("Type") }}', '{{ __("Severity") }}', '{{ __("Timestamp") }}', '{{ __("Location") }}', '{{ __("Details") }}'];
        const csvData = currentViolations.map(v => [
            v.driver_name,
            getViolationTypeInfo(v.violation_type).text,
            getSeverityInfo(v.severity).text,
            new Date(v.timestamp).toLocaleString(),
            v.location || '',
            JSON.stringify(v.details || {})
        ]);

        // Create CSV content
        const csvContent = [headers, ...csvData]
            .map(row => row.map(cell => `"${cell}"`).join(','))
            .join('\n');

        // Create and download file
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        link.setAttribute('href', url);
        link.setAttribute('download', `violations_${new Date().toISOString().split('T')[0]}.csv`);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);

        showNotification("success", "{{ __('Data exported successfully') }}");
    }

    function initializePageState() {
        // Show no vehicle state by default
        $('#noVehicleState').removeClass('d-none');
        
        // Hide all other states
        $('#loadingState').addClass('d-none');
        $('#driverOverviewCards').addClass('d-none');
        $('#driverScoringTableCard').addClass('d-none');
        $('#violationDetailsTableCard').addClass('d-none');
        $('#chartsRow1').addClass('d-none');
        $('#chartsRow2').addClass('d-none');
        $('#driverComparisonRow').addClass('d-none');
        $('#noDataState').addClass('d-none');
    }

    // Add event listeners for filter buttons and modal
    $(document).ready(function() {
        // Initialize page state - show no vehicle state by default
        initializePageState();
        
        // Initialize help section to be open by default
        $('#helpSection').addClass('show');
        // Handle modal shown event for map initialization
        $('#violationDetailsModal').on('shown.bs.modal', function() {
            if (violationMap) {
                setTimeout(() => {
                    violationMap.refreshMap();
                    violationMap.fitBounds();
                }, 300);
            }
        });
        
        // Handle modal hidden event to clean up
        $('#violationDetailsModal').on('hidden.bs.modal', function() {
            // Optional: Clean up map when modal is closed
            if (violationMap) {
                violationMap.removeMarkers();
            }
            // Reset filter div to show by default for next modal open
            $('#violationTypeFilterDiv').show();
        });
        $('input[name="violationTypeFilter"]').on('change', function() {
            const filterValue = $(this).val();
            currentFilter = filterValue;
            
            let filteredViolations = currentViolations;
            if (filterValue !== 'all') {
                filteredViolations = currentViolations.filter(v => v.violation_type === filterValue);
            }
            
            // Update map
            if (violationMap) {
                // Clear existing markers
                violationMap.markersCluster.clearLayers();
                violationMap.markerBounds = [];
                
                // Re-add filtered markers
                const markerModels = filteredViolations
                    .filter(v => v.location)
                    .map((violation, index) => {
                        const [lat, lng] = violation.location.split(',').map(Number);
                        if (isNaN(lat) || isNaN(lng)) return null;

                        const colors = {
                            speed: '#dc3545',
                            harsh: '#ffc107',
                            zone: '#17a2b8'
                        };

                        const sizes = {
                            low: 16,
                            medium: 20,
                            high: 24
                        };

                        return {
                            id: `violation_${index}`,
                            coord: [lat, lng],
                            //label: `${violation.driver_name} - ${getViolationTypeInfo(violation.violation_type).text}`,
                            popup: createViolationPopup(violation),
                            icon: {
                                name: "fa-solid fa-circle",
                                color: colors[violation.violation_type] || '#6c757d',
                                width: sizes[violation.severity] || 20,
                                className: `violation-marker-${violation.violation_type}`
                            }
                        };
                    })
                    .filter(marker => marker !== null);

                if (markerModels.length > 0) {
                    violationMap.pinMarkers(markerModels);
                    setTimeout(() => violationMap.refreshMap(), 200);
                    setTimeout(() => violationMap.fitBounds(), 300);
                }
            }
            
            // Update list
            populateViolationsList(filteredViolations);
            
            // Update counts
            $('#totalViolationsCount').text(filteredViolations.length);
            const uniqueLocations = new Set();
            filteredViolations.forEach(v => {
                if (v.location) {
                    uniqueLocations.add(v.location);
                }
            });
            $('#uniqueLocationsCount').text(uniqueLocations.size);
        });
    });

</script>
@endsection