@extends('layouts.app')

@section('title', __('Toll Analysis'))

@section('content')
<div class="page-wrapper" id="toll-analysis-report">
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted">{{ __('Reports') }}</a> / {{ __('Cost and Efficiency Reports') }}
                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-road me-2"></i>
                        {{ __('Toll Analysis') }}
                    </h2>
                    <div class="text-muted mt-1">
                        {{ __('Toll cost and event analysis based on region entry/exit and configured prices') }}
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title">{{ __('Filters') }}</h3>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">{{ __('Vehicle') }}</label>
                            <select class="form-select" id="vehicleSelect">
                                <option value="">{{ __('All Vehicles') }}</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">{{ __('Region') }}</label>
                            <select class="form-select" id="regionSelect">
                                <option value="">{{ __('All Regions') }}</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">{{ __('Start Date') }}</label>
                            <input type="date" class="form-control" id="startDate" value="{{ date('Y-m-d', strtotime('-1 week')) }}">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">{{ __('End Date') }}</label>
                            <input type="date" class="form-control" id="endDate" value="{{ date('Y-m-d') }}">
                        </div>
                        <div class="col-md-2 d-flex align-items-end">
                            <button type="button" class="btn btn-primary w-100" onclick="generateReport()">
                                <i class="icon ti ti-search me-1"></i>
                                {{ __('Search') }}
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="row row-cards mb-3" id="summaryCards">
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Total Toll Cost') }}</div>
                            <div class="h1 mb-0" id="totalTollCost">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Total Toll Events') }}</div>
                            <div class="h1 mb-0" id="totalTollEvents">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Unique Vehicles') }}</div>
                            <div class="h1 mb-0" id="uniqueVehicles">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Unique Regions') }}</div>
                            <div class="h1 mb-0" id="uniqueRegions">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Days Count') }}</div>
                            <div class="h1 mb-0" id="daysCount">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader">{{ __('Average Cost/Event') }}</div>
                            <div class="h1 mb-0" id="averageCost">-</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Toll Events Table -->
            <div class="card mb-3" id="tollEventsTableCard">
                <div class="card-header">
                    <h3 class="card-title">{{ __('Toll Events Details') }}</h3>
                    <div class="card-actions">
                        <button type="button" class="btn btn-outline-primary btn-sm" onclick="exportToCSV()">
                            <i class="ti ti-download me-1"></i>
                            {{ __('Export CSV') }}
                        </button>
                    </div>
                </div>
                <div class="card-body p-0">
                    <div id="toll-events-table"></div>
                </div>
            </div>

            <!-- Charts Section -->
            <div class="row row-cards mb-3 d-none" id="chartsSection">
                <!-- Cost by Region Chart -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Cost by Region') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="costByRegionChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                <!-- Cost by Vehicle Chart -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Cost by Vehicle') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="costByVehicleChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                <!-- Toll Events Over Time Chart -->
                <div class="col-lg-12">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">{{ __('Toll Events Over Time') }}</h3>
                        </div>
                        <div class="card-body">
                            <div id="tollEventsOverTimeChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Map Visualization -->
            <div class="card mb-3 d-none" id="mapCard">
                <div class="card-header">
                    <h3 class="card-title">{{ __('Toll Events Map') }}</h3>
                </div>
                <div class="card-body p-0">
                    <div id="toll-map" style="height: 500px;"></div>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">{{ __('Loading...') }}</span>
                </div>
                <div class="mt-2">{{ __('Generating toll analysis report...') }}</div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-road text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title">{{ __('No toll data found') }}</p>
                    <p class="empty-subtitle text-muted">
                        {{ __('No toll data was found for the selected period.') }}
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('styles')
<style>
    #toll-map .map-icon-label {
        padding: 0px !important;
        border-width: 0px !important;
        font-size: 15px !important;
    }
    
    .toll-cost-high {
        color: #dc3545;
        font-weight: bold;
    }
    
    .toll-cost-medium {
        color: #fd7e14;
        font-weight: bold;
    }
    
    .toll-cost-low {
        color: #28a745;
        font-weight: bold;
    }
</style>
@endsection

@section('scripts')
<script>
// --- Unit Variables ---
var speedUnit = "";
var lengthUnit = "";
var currency = "";

let tollData = [];
let vehicleList = [];
let regionList = [];
let tollTable;
let myMap;
let processedData = [];

$('#summaryCards').addClass('d-none');
$('#tollEventsTableCard').addClass('d-none');
$('#mapCard').addClass('d-none');
$('#loadingState').removeClass('d-none');
$('#noDataState').addClass('d-none'); 

onAppReady(function() {        

    loadVehicles().then(loadRegions).then(initializeTollTable).then(generateReport);
});

async function loadVehicles() {
    http.post({ url: '/api/vehicles/list', loading: true })
        .then(function(response) {
            if (response.success && response.data.vehicles) {
                const select = $('#vehicleSelect');
                select.empty();
                select.append('<option value="">{{ __('All Vehicles') }}</option>');
                vehicleList = response.data.vehicles;
                response.data.vehicles.forEach(vehicle => {
                    select.append(`<option value="${vehicle.did}">${vehicle.name}</option>`);
                });
            }
        });
}

async function loadRegions() {
    http.post({ url: '/api/regions/list', loading: true })
        .then(function(response) {
            if (response.success && response.data) {
                const select = $('#regionSelect');
                select.empty();
                select.append('<option value="">{{ __('All Regions') }}</option>');
                regionList = response.data;
                response.data.forEach(region => {
                    select.append(`<option value="${region.id}">${region.name}</option>`);
                });
            }
        });
}

async function generateReport() {
    const vehicleId = $('#vehicleSelect').val();
    const regionId = $('#regionSelect').val();
    const startDate = $('#startDate').val();
    const endDate = $('#endDate').val();
    
    if (!startDate || !endDate) {
        showConfirmDialog({
            title: '{{ __("Warning") }}',
            message: '{{ __("Please select start and end dates") }}',
            confirmText: '{{ __("OK") }}',
            cancelText: '',
            type: 'warning',
            onConfirm: function() {},
            onCancel: function() {}
        });
        return;
    }
    
    showLoadingState();
    const requestData = { start_date: startDate, end_date: endDate };
    if (vehicleId) requestData.device_id = vehicleId;
    
    http.post({ url: '/api/reports/toll-analysis', data: requestData, loading: false })
        .then(data => {
            // --- Set units after data is loaded ---
            speedUnit = __(sessionSettings.speedUnit || 'km/h');
            lengthUnit = __(sessionSettings.lengthUnit || 'km');
            currency = __(sessionSettings.currency || 'TRY');
            // ---
            hideLoadingState();
            if (data.success) {
                tollData = data.data.device_logs;

                var message = __('#name# used #value# records').replace('#name#', __('Toll Analysis')).replace('#value#', tollData.length);
                showReportDataProcessingInfo(message);

                processTollData(regionId);
            } else {
                showNoDataState();
                showConfirmDialog({
                    title: '{{ __("Error") }}',
                    message: data.message || '{{ __("Unknown error occurred") }}',
                    confirmText: '{{ __("OK") }}',
                    cancelText: '',
                    type: 'danger',
                    onConfirm: function() {},
                    onCancel: function() {}
                });
            }
        })
        .catch(err => {
            hideLoadingState();
            showNoDataState();
            showConfirmDialog({
                title: '{{ __("Error") }}',
                message: err.message,
                confirmText: '{{ __("OK") }}',
                cancelText: '',
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
        });
}

async function processTollData(regionId) {
    if (!tollData || tollData.length === 0) {
        showNoDataState();
        return;
    }
    // Filter by region if needed
    let filtered = tollData;
    if (regionId) {
        filtered = filtered.filter(e => e.region_id == regionId);
    }
    // Pair entry/exit sessions
    const sessions = pairEntryExitSessions(filtered);
    if (sessions.length === 0) {
        showNoDataState();
        return;
    }
    // Only use entry events for summary
    let totalCost = 0, totalEvents = 0;
    const uniqueVehicles = new Set();
    const uniqueRegions = new Set();
    sessions.forEach(s => {
        const price = parseFloat(s.price || 0);
        totalCost += price;
        totalEvents++;
        uniqueVehicles.add(s.entry.did);
        uniqueRegions.add(s.entry.region_id); 
    });
    // Calculate days count
    const startDate = $('#startDate').val();
    const endDate = $('#endDate').val();
    let daysCount = '-';
    if (startDate && endDate) {
        const start = new Date(startDate);
        const end = new Date(endDate);
        daysCount = (Math.ceil((end - start) / (1000 * 60 * 60 * 24)) + 1) + ' ' + __('Days');
    }
    // Calculate average cost
    const averageCost = totalEvents > 0 ? (totalCost / totalEvents) : 0;
    // Update summary cards
    $('#totalTollCost').text(totalCost.toFixed(2) + ' ' + currency);
    $('#totalTollEvents').text(totalEvents);
    $('#uniqueVehicles').text(uniqueVehicles.size);
    $('#uniqueRegions').text(uniqueRegions.size);
    $('#daysCount').text(daysCount);
    $('#averageCost').text(averageCost.toFixed(2) + ' ' + currency);
    // Store processed data for export
    processedData = sessions;
    // Update table
    updateTollTable(sessions);
    // Update charts to use sessions
    updateCharts(sessions);
    // Update map with proper refresh
    updateMap(filtered);
    // Show all sections
    $('#summaryCards').removeClass('d-none');
    $('#tollEventsTableCard').removeClass('d-none');
    $('#chartsSection').removeClass('d-none');
    $('#mapCard').removeClass('d-none');
    $('#noDataState').addClass('d-none');
}

function initializeTollTable() {
    tollTable = createTabulator("#toll-events-table", {
        initialSort: [ {column: "entry_time", dir: "desc"} ],
        columns: [
            {title: "#", field: "id", hozAlign: "center"},
            {title: "{{ __('Region') }}", field: "region_name"},
            {title: "{{ __('Vehicle') }}", field: "vehicle_name"},
            {title: "{{ __('Entry Date/Time') }}", field: "entry_time"},
            {title: "{{ __('Exit Date/Time') }}", field: "exit_time"},
            {title: "{{ __('Toll Cost') }}", field: "cost", hozAlign: "center", formatter: function(cell) {
                const cost = cell.getValue();
                let cssClass = 'toll-cost-low';
                if (cost > 200) cssClass = 'toll-cost-high';
                else if (cost > 100) cssClass = 'toll-cost-medium';
                return `<span class="${cssClass}">${(parseFloat(cost) || 0).toFixed(2)} ${currency}</span>`;
            }},
            {title: "{{ __('Show on Map') }}", field: "entry_log_id", width: 120, hozAlign: "center", formatter: function(cell) {
                const logId = cell.getValue();
                return `<button class='btn btn-outline-primary btn-sm' onclick='focusMapOnEntry(${logId})'>{{ __('Show') }}</button>`;
            }}
        ]
    });
}

function updateTollTable(sessions) {
    const tableData = sessions.map((s, idx) => {
        return {
            id: idx + 1,
            region_name: s.region,
            vehicle_name: s.vehicle,
            entry_time: dateTimeFormatter(s.entry.event_time),
            exit_time: dateTimeFormatter(s.exit.event_time),
            cost: parseFloat(s.price || 0),
            entry_log_id: s.entry.log_id
        };
    });
    tollTable.setData(tableData);
}

function updateCharts(sessions) {
    if (!sessions || sessions.length === 0) return;

    // Cost by Region
    const regionCost = {};
    sessions.forEach(s => {
        if (!regionCost[s.region]) regionCost[s.region] = 0;
        regionCost[s.region] += parseFloat(s.price || 0);
    });
    const regionNames = Object.keys(regionCost);
    const regionCosts = Object.values(regionCost).map(v => parseFloat(v.toFixed(2)));
    const regionOptions = {
        series: [{ name: `{{ __('Cost') }} (${currency})`, data: regionCosts }],
        chart: { type: 'bar', height: 300 },
        xaxis: { categories: regionNames },
        colors: ['#007bff'],
        dataLabels: { enabled: false },
        tooltip: { y: { formatter: val => val.toFixed(2) + ' ' + currency } },
        legend: { show: false }
    };
    if (window.costByRegionChart && typeof window.costByRegionChart.destroy === 'function') window.costByRegionChart.destroy();
    const regionChartEl = document.querySelector("#costByRegionChart");
    if (regionChartEl) {
        window.costByRegionChart = new ApexCharts(regionChartEl, regionOptions);
        window.costByRegionChart.render();
    }
    
    // Cost by Vehicle
    const vehicleCost = {};
    sessions.forEach(s => {
        if (!vehicleCost[s.vehicle]) vehicleCost[s.vehicle] = 0;
        vehicleCost[s.vehicle] += parseFloat(s.price || 0);
    });
    const vehicleNames = Object.keys(vehicleCost);
    const vehicleCosts = Object.values(vehicleCost).map(v => parseFloat(v.toFixed(2)));
    const vehicleOptions = {
        series: [{ name: `{{ __('Cost') }} (${currency})`, data: vehicleCosts }],
        chart: { type: 'bar', height: 300 },
        xaxis: { categories: vehicleNames },
        colors: ['#28a745'],
        dataLabels: { enabled: false },
        tooltip: { y: { formatter: val => val.toFixed(2) + ' ' + currency } },
        legend: { show: false }
    };
    if (window.costByVehicleChart && typeof window.costByVehicleChart.destroy === 'function') window.costByVehicleChart.destroy();
    const vehicleChartEl = document.querySelector("#costByVehicleChart");
    if (vehicleChartEl) {
        window.costByVehicleChart = new ApexCharts(vehicleChartEl, vehicleOptions);
        window.costByVehicleChart.render();
    }
    
    // Toll Events Over Time
    const dateEvents = {};
    sessions.forEach(e => {
        const date = (e.entry.event_time || e.entry.timestamp).split(' ')[0];
        if (!dateEvents[date]) dateEvents[date] = 0;
        dateEvents[date]++;
    });
    const dates = Object.keys(dateEvents).sort();
    const eventCounts = dates.map(d => dateEvents[d]);
    const eventsOptions = {
        series: [{ name: '{{ __('Toll Events') }}', data: eventCounts }],
        chart: { type: 'line', height: 300 },
        xaxis: { categories: dates },
        colors: ['#dc3545'],
        dataLabels: { enabled: false },
        tooltip: { y: { formatter: val => val } },
        legend: { show: false }
    };
    if (window.tollEventsOverTimeChart && typeof window.tollEventsOverTimeChart.destroy === 'function') window.tollEventsOverTimeChart.destroy();
    const eventsChartEl = document.querySelector("#tollEventsOverTimeChart");
    if (eventsChartEl) {
        window.tollEventsOverTimeChart = new ApexCharts(eventsChartEl, eventsOptions);
        window.tollEventsOverTimeChart.render();
    }
}

function pairEntryExitSessions(events) {
    // Group by vehicle and region
    const grouped = {};
    events.forEach(e => {
        const key = `${e.did}_${e.region_id}`;
        if (!grouped[key]) grouped[key] = [];
        grouped[key].push(e);
    });
    const sessions = [];
    Object.values(grouped).forEach(eventList => {
        // Sort by event_time
        eventList.sort((a, b) => new Date(a.event_time) - new Date(b.event_time));
        let entry = null;
        eventList.forEach(e => {
            if (e.region_control === 'entry') {
                entry = e;
            } else if (e.region_control === 'exit' && entry) {
                // Pair found
                sessions.push({
                    entry,
                    exit: e,
                    vehicle: entry.device_name,
                    region: entry.region_name,
                    price: entry.region_price,
                });
                entry = null; // Reset for next session
            }
        });
    });
    return sessions;
}

function updateMap(data) {
    // Initialize map if not exists
    if (!myMap) {
        myMap = new LeafletMap();
        myMap.initMap("toll-map", {});
    }
    // Clear existing map elements
    myMap.removeMarkers();
    myMap.clearAllPolygons();
    myMap.removeCircles();
    myMap.removePolyline();
    if (!data || data.length === 0) {
        // Still show regions even if no events
        addRegionsToMap();
        return;
    }
    // Add regions to map first
    addRegionsToMap();
    // Pair entry/exit sessions
    const sessions = pairEntryExitSessions(data);
    // Add markers and polylines for each session
    sessions.forEach((session, idx) => {
        // Entry marker
        const entryLat = parseFloat(session.entry.latitude);
        const entryLng = parseFloat(session.entry.longitude);
        if (!isNaN(entryLat) && !isNaN(entryLng)) {
            myMap.pinMarkers([{
                id: `entry-${session.entry.log_id}`,
                label: `<div class='badge' style='background-color: #28a745; color: white;'>${session.vehicle} - {{ __('Entry') }}</div>`,
                device_id: session.entry.did,
                popup: `
                    <strong>{{ __('Entry') }} - ${session.vehicle}</strong><br>
                    {{ __('Region') }}: ${session.region}<br>
                    {{ __('Date/Time') }}: ${dateTimeFormatter(session.entry.event_time)}<br>
                    {{ __('Toll Cost') }}: ${session.price} ${currency}<br>
                `,
                coord: [entryLat, entryLng],
                icon: {
                    name: "fa-solid fa-sign-in-alt",
                    color: "#28a745",
                    className: "toll-marker",
                }
            }]);
        }
        // Exit marker
        const exitLat = parseFloat(session.exit.latitude);
        const exitLng = parseFloat(session.exit.longitude);
        if (!isNaN(exitLat) && !isNaN(exitLng)) {
            myMap.pinMarkers([{
                id: `exit-${session.exit.log_id}`,
                label: `<div class='badge' style='background-color: #dc3545; color: white;'>${session.vehicle} - {{ __('Exit') }}</div>`,
                device_id: session.exit.did,
                popup: `
                    <strong>{{ __('Exit') }} - ${session.vehicle}</strong><br>
                    {{ __('Region') }}: ${session.region}<br>
                    {{ __('Date/Time') }}: ${dateTimeFormatter(session.exit.event_time)}<br>
                    {{ __('Toll Cost') }}: ${session.price} ${currency}<br>
                `,
                coord: [exitLat, exitLng],
                icon: {
                    name: "fa-solid fa-sign-out-alt",
                    color: "#dc3545",
                    className: "toll-marker",
                }
            }]);
        }
        var groupColors = generateDistinctColors(sessions.length);
        // Polyline between entry and exit
        if (!isNaN(entryLat) && !isNaN(entryLng) && !isNaN(exitLat) && !isNaN(exitLng)) {
            myMap.addPolyline({
                coords: [[entryLat, entryLng], [exitLat, exitLng]],
                color: groupColors[idx],
                weight: 5,
                opacity: 0.7,
                smoothFactor: 1,
            });
        }
    });
    // Fit map to all markers and polylines
    setTimeout(() => {
        if (myMap.mapLet) {
            myMap.mapLet.invalidateSize();
            myMap.fitBounds();
        }
    }, 300);
}

function addRegionsToMap() {
    // Add all regions to the map for context
    regionList.forEach(region => {
        if (region.polygon_json) {
            try {
                const polygonData = typeof region.polygon_json === 'string' ? 
                    JSON.parse(region.polygon_json) : region.polygon_json;
                
                myMap.addPolygon({
                    id: region.id,
                    label: region.name,
                    color: "#6c757d", // Gray border
                    fillColor: "#f8f9fa", // Light gray fill
                    fillOpacity: 0.3,
                    coords: polygonData,
                });
            } catch (e) {
                console.warn('Error parsing polygon data for region:', region.name, e);
            }
        }
    });
}

function exportToCSV() {
    if (!processedData || processedData.length === 0) {
        showConfirmDialog({
            title: '{{ __("Warning") }}',
            message: '{{ __("No data to export") }}',
            confirmText: '{{ __("OK") }}',
            cancelText: '',
            type: 'warning',
            onConfirm: function() {},
            onCancel: function() {}
        });
        return;
    }
    
    // Create CSV content
    const headers = [
        '{{ __("Date/Time") }}',
        '{{ __("Vehicle") }}',
        '{{ __("Region") }}',
        '{{ __("Event Type") }}',
        `{{ __("Toll Cost") }} (${currency})`,
        '{{ __("Latitude") }}',
        '{{ __("Longitude") }}',
        '{{ __("Region Status") }}'
    ];
    
    const csvContent = [
        headers.join(','),
        ...processedData.map((item, idx) => {
            let eventType = 'entry/exit';
            if (item.region_control === 'entry') eventType = 'entry';
            else if (item.region_control === 'exit') eventType = 'exit';
            
            return [
                dateTimeFormatter(item.event_time || item.timestamp),
                item.device_name,
                item.region_name,
                eventType,
                (item.toll_cost || 0).toFixed(2),
                item.latitude,
                item.longitude,
                item.region_status
            ].join(',');
        })
    ].join('\n');
    
    // Create and download file
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `toll-analysis-${new Date().toISOString().split('T')[0]}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

function showLoadingState() {
    $('#loadingState').removeClass('d-none');
    $('#summaryCards').addClass('d-none');
    $('#tollEventsTableCard').addClass('d-none');
    $('#chartsSection').addClass('d-none');
    $('#mapCard').addClass('d-none');
    $('#noDataState').addClass('d-none');
}

function hideLoadingState() {
    $('#loadingState').addClass('d-none');
}

function showNoDataState() {
    $('#noDataState').removeClass('d-none');
    $('#summaryCards').addClass('d-none');
    $('#tollEventsTableCard').addClass('d-none');
    $('#chartsSection').addClass('d-none');
    $('#mapCard').addClass('d-none');
}

// Place this at the end of your <script> section, outside of any $(function() { ... })
function focusMapOnEntry(logId) {
    // Scroll the map into view
    document.getElementById('toll-map').scrollIntoView({ behavior: 'smooth', block: 'center' });

    if (!myMap || !myMap.mapLet) return;
    // Find marker by id
    const markerId = `entry-${logId}`;
    let found = false;
    if (myMap.markersCluster) {
        myMap.markersCluster.getLayers().forEach(marker => {
            if (marker.options && marker.options.id === markerId) {
                myMap.mapLet.setView(marker.getLatLng(), 16);
                marker.openPopup();
                found = true;
            }
        });
    }
    if (!found && myMap.markers) {
        myMap.markers.forEach(marker => {
            if (marker.options && marker.options.id === markerId) {
                myMap.mapLet.setView(marker.getLatLng(), 16);
                marker.openPopup();
            }
        });
    }
}
</script>
@endsection