@extends('layouts.app')

@section('title', __('General Settings'))

@section('content')
<div class="page-wrapper">
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        {{ __('Settings') }}
                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-settings me-2"></i>
                        {{ __('General Settings') }}
                    </h2>
                    <div class="text-muted mt-1">
                        {{ __('Manage system settings, fuel prices, maintenance intervals, and alert configurations') }}
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="page-body">
        <div class="container-xl">


            <!-- Settings Tabs -->
            <div class="card">
                <div class="card-header">
                    <ul class="nav nav-tabs card-header-tabs" data-bs-toggle="tabs">    
                        @canaccess('settings_fuel', 'view')
                        <li class="nav-item">
                            <a href="#fuel-tab" class="nav-link active" data-bs-toggle="tab">
                                <i class="ti ti-gas-station me-1"></i>
                                {{ __('Fuel Settings') }}
                            </a>
                        </li>
                        @endcanaccess
                        @canaccess('settings_maintenance', 'view')
                        <li class="nav-item">
                            <a href="#maintenance-tab" class="nav-link" data-bs-toggle="tab">
                                <i class="ti ti-tools me-1"></i>
                                {{ __('Maintenance') }}
                            </a>
                        </li>
                        @endcanaccess
                        @canaccess('settings_alerts', 'view')
                        <li class="nav-item">
                            <a href="#alerts-tab" class="nav-link" data-bs-toggle="tab">
                                <i class="ti ti-bell me-1"></i>
                                {{ __('Alerts') }}
                            </a>
                        </li>
                        @endcanaccess
                        <li class="nav-item">
                            <a href="#system-tab" class="nav-link" data-bs-toggle="tab">
                                <i class="ti ti-device-desktop me-1"></i>
                                {{ __('System') }}
                            </a>
                        </li>
                    </ul>
                </div>
                <div class="card-body">
                    <div class="tab-content">
                        @canaccess('settings_fuel', 'view')
                        <!-- Fuel Settings Tab -->
                        <div class="tab-pane active show" id="fuel-tab">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h3 class="card-title">{{ __('Fuel Prices') }}</h3>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Diesel Price') }}</label>
                                                <input type="number" class="form-control" id="fuel_price_diesel" step="0.01" min="0">
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Gasoline Price') }}</label>
                                                <input type="number" class="form-control" id="fuel_price_gasoline" step="0.01" min="0">
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('LPG Price') }}</label>
                                                <input type="number" class="form-control" id="fuel_price_lpg" step="0.01" min="0">
                                            </div>
                                            <button type="button" class="btn btn-primary" onclick="saveFuelSettings()">
                                                <i class="icon ti ti-device-floppy me-1"></i>
                                                {{ __('Save Fuel Settings') }}
                                            </button>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h3 class="card-title">{{ __('Fuel Cost Calculator') }}</h3>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Fuel Type') }}</label>
                                                <select class="form-select" id="calc_fuel_type">
                                                    <option value="diesel">{{ __('Diesel') }}</option>
                                                    <option value="gasoline">{{ __('Gasoline') }}</option>
                                                    <option value="lpg">{{ __('LPG') }}</option>
                                                </select>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Consumption (Liters)') }}</label>
                                                <input type="number" class="form-control" id="calc_consumption" step="0.1" min="0">
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Total Cost') }}</label>
                                                <div class="input-group">
                                                    <input type="text" class="form-control" id="calc_total_cost" readonly>
                                                    <span class="input-group-text" id="calc_currency">TRY</span>
                                                </div>
                                            </div>
                                            <button type="button" class="btn btn-outline-primary" onclick="calculateFuelCost()">
                                                <i class="icon ti ti-calculator me-1"></i>
                                                {{ __('Calculate') }}
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        @endcanaccess
                        @canaccess('settings_maintenance', 'view')
                        <!-- Maintenance Settings Tab -->
                        <div class="tab-pane" id="maintenance-tab">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h3 class="card-title">{{ __('Maintenance Intervals') }}</h3>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Maintenance Reminder (Days)') }}</label>
                                                <input type="number" class="form-control" id="maintenance_reminder_days" min="1" max="365">
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Oil Change Interval (km)') }}</label>
                                                <input type="number" class="form-control" id="oil_change_km" min="1000" step="1000">
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Tire Change Interval (km)') }}</label>
                                                <input type="number" class="form-control" id="tire_change_km" min="1000" step="1000">
                                            </div>
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                                    <input class="form-check-input" type="checkbox" id="maintenance_enabled">
                                                    <label class="form-check-label" for="maintenance_enabled">
                                                        {{ __('Enable Maintenance Reminders') }}
                                                    </label>
                                                </div>
                                            </div>
                                            <button type="button" class="btn btn-primary" onclick="saveMaintenanceSettings()">
                                                <i class="icon ti ti-device-floppy me-1"></i>
                                                {{ __('Save Maintenance Settings') }}
                                            </button>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h3 class="card-title">{{ __('Maintenance Summary') }}</h3>
                                        </div>
                                        <div class="card-body">
                                            <div class="row">
                                                <div class="col-6">
                                                    <div class="text-center">
                                                        <div class="h1 mb-0" id="maintenance_summary_reminder">-</div>
                                                        <div class="text-muted">{{ __('Days') }}</div>
                                                    </div>
                                                </div>
                                                <div class="col-6">
                                                    <div class="text-center">
                                                        <div class="h1 mb-0" id="maintenance_summary_oil">-</div>
                                                        <div class="text-muted">{{ __('km') }}</div>
                                                    </div>
                                                </div>
                                            </div>
                                            <hr>
                                            <div class="row">
                                                <div class="col-6">
                                                    <div class="text-center">
                                                        <div class="h1 mb-0" id="maintenance_summary_tire">-</div>
                                                        <div class="text-muted">{{ __('km') }}</div>
                                                    </div>
                                                </div>
                                                <div class="col-6">
                                                    <div class="text-center">
                                                        <div class="h1 mb-0" id="maintenance_summary_status">-</div>
                                                        <div class="text-muted">{{ __('Status') }}</div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        @endcanaccess
                        @canaccess('settings_alerts', 'view')
                        <!-- Alerts Settings Tab -->
                        <div class="tab-pane" id="alerts-tab">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h3 class="card-title">{{ __('Speed & Safety Alerts') }}</h3>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Default Speed Limit (km/h)') }}</label>
                                                <input type="number" class="form-control" id="speed_limit_default" min="20" max="200">
                                            </div>
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                                    <input class="form-check-input" type="checkbox" id="geofence_alerts_enabled">
                                                    <label class="form-check-label" for="geofence_alerts_enabled">
                                                        {{ __('Enable Geofence Alerts') }}
                                                    </label>
                                                </div>
                                            </div>
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                                    <input class="form-check-input" type="checkbox" id="maintenance_alerts_enabled">
                                                    <label class="form-check-label" for="maintenance_alerts_enabled">
                                                        {{ __('Enable Maintenance Alerts') }}
                                                    </label>
                                                </div>
                                            </div>
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                                    <input class="form-check-input" type="checkbox" id="fuel_alerts_enabled">
                                                    <label class="form-check-label" for="fuel_alerts_enabled">
                                                        {{ __('Enable Fuel Level Alerts') }}
                                                    </label>
                                                </div>
                                            </div>
                                            <button type="button" class="btn btn-primary" onclick="saveAlertSettings()">
                                                <i class="icon ti ti-device-floppy me-1"></i>
                                                {{ __('Save Alert Settings') }}
                                            </button>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h3 class="card-title">{{ __('Notification Settings') }}</h3>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                                    <input class="form-check-input" type="checkbox" id="alert_email_enabled">
                                                    <label class="form-check-label" for="alert_email_enabled">
                                                        {{ __('Enable Email Alerts') }}
                                                    </label>
                                                </div>
                                            </div>
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                                    <input class="form-check-input" type="checkbox" id="alert_sms_enabled">
                                                    <label class="form-check-label" for="alert_sms_enabled">
                                                        {{ __('Enable SMS Alerts') }}
                                                    </label>
                                                </div>
                                            </div>
                                            <div class="alert alert-info">
                                                <i class="ti ti-info-circle me-1"></i>
                                                {{ __('Alert settings will be applied to all vehicles in the selected store.') }}
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        @endcanaccess
                        @canaccess('settings_system', 'view')   
                        <!-- System Settings Tab -->
                        <div class="tab-pane" id="system-tab">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-header">
                                            <h3 class="card-title">{{ __('System Configuration') }}</h3>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Timezone') }}</label>
                                                <select class="form-select" id="timezone">
                                                    <option value="Europe/Istanbul">Europe/Istanbul</option>
                                                    <option value="UTC">UTC</option>
                                                    <option value="America/New_York">America/New_York</option>
                                                    <option value="Europe/London">Europe/London</option>
                                                    <option value="Asia/Tokyo">Asia/Tokyo</option>
                                                </select>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Language') }}</label>
                                                <select class="form-select" id="language">
                                                    @foreach(config('app.available_locales') as $name => $code)
                                                        <option value="{{ $code }}">{{ $name }}</option>
                                                    @endforeach
                                                </select>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Date Format') }}</label>
                                                <select class="form-select" id="date_format">
                                                    <option value="d.m.y">{{ __('DD.MM.YYYY') }}</option>
                                                    <option value="y-m-d">{{ __('YYYY-MM-DD') }}</option>
                                                    <option value="m/d/y">{{ __('MM/DD/YYYY') }}</option>
                                                    <option value="d/m/y">{{ __('DD/MM/YYYY') }}</option>
                                                </select>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Time Format') }}</label>
                                                <select class="form-select" id="time_format">
                                                    <option value="24h_colon">{{ __('H:i:s (24-hour)') }}</option>
                                                    <option value="24h_double">{{ __('HH:mm:ss (24-hour)') }}</option>
                                                    <option value="12h_colon">{{ __('h:i:s A (12-hour, AM/PM)') }}</option>
                                                    <option value="12h_double">{{ __('hh:mm:ss A (12-hour, AM/PM)') }}</option>
                                                </select>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Speed Unit') }}</label>
                                                <select class="form-select" id="speed_unit">
                                                    <option value="km/h">km/h</option>
                                                    <option value="mph">mph</option>
                                                </select>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Currency') }}</label>
                                                <select class="form-select" id="currency">
                                                    <option value="TRY">{{ __('TRY (Turkish Lira)') }}</option>
                                                    <option value="USD">{{ __('USD (US Dollar)') }}</option>
                                                    <option value="EUR">{{ __('EUR (Euro)') }}</option>
                                                </select>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Weight Unit') }}</label>
                                                <select class="form-select" id="weight_unit">
                                                    <option value="kg">{{ __('kg (Kilogram)') }}</option>
                                                    <option value="g">{{ __('g (Gram)') }}</option>
                                                    <option value="lb">{{ __('lb (Pound)') }}</option>
                                                    <option value="oz">{{ __('oz (Ounce)') }}</option>
                                                    <option value="ton">{{ __('ton (Metric Ton)') }}</option>
                                                </select>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Liquid Unit') }}</label>
                                                <select class="form-select" id="liquid_unit">
                                                    <option value="lt">{{ __('lt (Liter)') }}</option>
                                                    <option value="ml">{{ __('ml (Milliliter)') }}</option>
                                                    <option value="gal">{{ __('gal (Gallon)') }}</option>
                                                    <option value="fl_oz">{{ __('fl oz (Fluid Ounce)') }}</option>
                                                </select>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Length Unit') }}</label>
                                                <select class="form-select" id="length_unit">
                                                    <option value="mt">{{ __('mt (Meter)') }}</option>
                                                    <option value="cm">{{ __('cm (Centimeter)') }}</option>
                                                    <option value="mm">{{ __('mm (Millimeter)') }}</option>
                                                    <option value="km">{{ __('km (Kilometer)') }}</option>
                                                    <option value="mi">{{ __('mi (Mile)') }}</option>
                                                    <option value="ft">{{ __('ft (Foot)') }}</option>
                                                    <option value="in">{{ __('in (Inch)') }}</option>
                                                </select>
                                            </div>
                                            <button type="button" class="btn btn-primary" onclick="saveSystemSettings()">
                                                <i class="icon ti ti-device-floppy me-1"></i>
                                                {{ __('Save System Settings') }}
                                            </button>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6 d-none">
                                    <div class="card">
                                        <div class="card-header">
                                            <h3 class="card-title">{{ __('Backup Settings') }}</h3>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                                    <input class="form-check-input" type="checkbox" id="auto_backup_enabled">
                                                    <label class="form-check-label" for="auto_backup_enabled">
                                                        {{ __('Enable Automatic Backups') }}
                                                    </label>
                                                </div>
                                            </div>
                                            <div class="mb-3">
                                                <label class="form-label">{{ __('Backup Frequency') }}</label>
                                                <select class="form-select" id="backup_frequency">
                                                    <option value="daily">{{ __('Daily') }}</option>
                                                    <option value="weekly">{{ __('Weekly') }}</option>
                                                    <option value="monthly">{{ __('Monthly') }}</option>
                                                </select>
                                            </div>
                                            <div class="alert alert-warning">
                                                <i class="ti ti-alert-triangle me-1"></i>
                                                {{ __('System settings will be applied globally to all stores.') }}
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        @endcanaccess
                    </div>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">{{ __('Loading...') }}</span>
                </div>
                <div class="mt-2">{{ __('Loading settings...') }}</div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    let settingsData = {};

    onAppReady(function() {
        http.showLoading();
        setTimeout(() => {  
            loadSettings();
        }, 1000);
    });

    // Auto-calculate fuel cost when values change
    $('#calc_fuel_type, #calc_consumption').on('change input', function() {
        calculateFuelCost();
    });

    function showLoading() {
        $('#loadingState').removeClass('d-none');
    }

    function hideLoading() {
        $('#loadingState').addClass('d-none');
    }

    function loadSettings() {
        showLoading();

        // Load all settings for the current store
        http.get({
            url: '/api/settings',
            loading: false
        }).then(function(response) {
            hideLoading();
            if (response.success) {
                settingsData = response.data;

                //console.log(settingsData);
        
                if (Object.keys(settingsData).length === 0) {
                    showSuccessMessage('{{ __("Using default settings") }}');
                }
                populateSettings();
                showSuccessMessage('{{ __("Settings loaded successfully") }}');

                http.hideLoading();
            } else {
                showErrorMessage(response.message || '{{ __("Error loading settings") }}');
            }
        }).catch(function(error) {
            hideLoading();
            showErrorMessage('{{ __("Error loading settings") }}');
            console.error('Error loading settings:', error);
        });
    }

    function populateSettings() {
        // Populate fuel settings
        if (settingsData.fuel) {
            $('#fuel_price_diesel').val(settingsData.fuel.find(item => item.setting_key === 'fuel_price_diesel')?.typed_value || '');
            $('#fuel_price_gasoline').val(settingsData.fuel.find(item => item.setting_key === 'fuel_price_gasoline')?.typed_value || '');
            $('#fuel_price_lpg').val(settingsData.fuel.find(item => item.setting_key === 'fuel_price_lpg')?.typed_value || '');
        }

        // Populate maintenance settings
        if (settingsData.maintenance) {
            $('#maintenance_reminder_days').val(settingsData.maintenance.find(item => item.setting_key === 'maintenance_reminder_days')?.typed_value || '');
            $('#oil_change_km').val(settingsData.maintenance.find(item => item.setting_key === 'oil_change_km')?.typed_value || '');
            $('#tire_change_km').val(settingsData.maintenance.find(item => item.setting_key === 'tire_change_km')?.typed_value || '');
            $('#maintenance_enabled').prop('checked', settingsData.maintenance.find(item => item.setting_key === 'maintenance_enabled')?.typed_value || false);
        }

        // Populate alert settings
        if (settingsData.alerts) {
            $('#speed_limit_default').val(settingsData.alerts.find(item => item.setting_key === 'speed_limit_default')?.typed_value || '');
            $('#geofence_alerts_enabled').prop('checked', settingsData.alerts.find(item => item.setting_key === 'geofence_alerts_enabled')?.typed_value || false);
            $('#maintenance_alerts_enabled').prop('checked', settingsData.alerts.find(item => item.setting_key === 'maintenance_alerts_enabled')?.typed_value || false);
            $('#fuel_alerts_enabled').prop('checked', settingsData.alerts.find(item => item.setting_key === 'fuel_alerts_enabled')?.typed_value || false);
            $('#alert_email_enabled').prop('checked', settingsData.alerts.find(item => item.setting_key === 'alert_email_enabled')?.typed_value || false);
            $('#alert_sms_enabled').prop('checked', settingsData.alerts.find(item => item.setting_key === 'alert_sms_enabled')?.typed_value || false);
        }

        // Populate system settings
        if (settingsData.system) {
            const dateFormat = settingsData.system.find(item => item.setting_key === 'date_format')?.typed_value || 'd.m.y';
            const timeFormat = settingsData.system.find(item => item.setting_key === 'time_format')?.typed_value || 'H:i';
            const speedUnit = settingsData.system.find(item => item.setting_key === 'speed_unit')?.typed_value || 'km/h';
            
            $('#timezone').val(settingsData.system.find(item => item.setting_key === 'timezone')?.typed_value || '{{ $cookieTimezone ?? "Europe/Istanbul" }}');
            $('#language').val(settingsData.system.find(item => item.setting_key === 'language')?.typed_value || '{{ $cookieLocale ?? "tr" }}');
            $('#date_format').val(dateFormat);
            $('#time_format').val(timeFormat);
            $('#speed_unit').val(speedUnit);
            $('#currency').val(settingsData.system.find(item => item.setting_key === 'currency')?.typed_value || 'TRY');
            $('#weight_unit').val(settingsData.system.find(item => item.setting_key === 'weight_unit')?.typed_value || 'kg');
            $('#liquid_unit').val(settingsData.system.find(item => item.setting_key === 'liquid_unit')?.typed_value || 'lt');
            $('#length_unit').val(settingsData.system.find(item => item.setting_key === 'length_unit')?.typed_value || 'mt');
            $('#auto_backup_enabled').prop('checked', settingsData.system.find(item => item.setting_key === 'auto_backup_enabled')?.typed_value || {{ $cookieAutoBackupEnabled ? 'true' : 'false' }});
            $('#backup_frequency').val(settingsData.system.find(item => item.setting_key === 'backup_frequency')?.typed_value || '{{ $cookieBackupFrequency ?? "daily" }}');
        }

        // Update summary displays
        updateMaintenanceSummary();
        updateFuelCalculator();
    }

    function saveFuelSettings() {
        const settings = [
            { key: 'fuel_price_diesel', value: $('#fuel_price_diesel').val() },
            { key: 'fuel_price_gasoline', value: $('#fuel_price_gasoline').val() },
            { key: 'fuel_price_lpg', value: $('#fuel_price_lpg').val() }
        ];

        saveMultipleSettings(settings, 'fuel', '{{ __("Fuel settings saved successfully") }}');
    }

    function saveMaintenanceSettings() {
        const settings = [
            { key: 'maintenance_reminder_days', value: $('#maintenance_reminder_days').val() },
            { key: 'oil_change_km', value: $('#oil_change_km').val() },
            { key: 'tire_change_km', value: $('#tire_change_km').val() },
            { key: 'maintenance_enabled', value: $('#maintenance_enabled').is(':checked'), type: 'boolean' }
        ];

        saveMultipleSettings(settings, 'maintenance', '{{ __("Maintenance settings saved successfully") }}');
    }

    function saveAlertSettings() {
        const settings = [
            { key: 'speed_limit_default', value: $('#speed_limit_default').val() },
            { key: 'geofence_alerts_enabled', value: $('#geofence_alerts_enabled').is(':checked'), type: 'boolean' },
            { key: 'maintenance_alerts_enabled', value: $('#maintenance_alerts_enabled').is(':checked'), type: 'boolean' },
            { key: 'fuel_alerts_enabled', value: $('#fuel_alerts_enabled').is(':checked'), type: 'boolean' },
            { key: 'alert_email_enabled', value: $('#alert_email_enabled').is(':checked'), type: 'boolean' },
            { key: 'alert_sms_enabled', value: $('#alert_sms_enabled').is(':checked'), type: 'boolean' }
        ];

        saveMultipleSettings(settings, 'alerts', '{{ __("Alert settings saved successfully") }}');
    }

    function saveSystemSettings() {
        const settings = [
            { key: 'timezone', value: $('#timezone').val() },
            { key: 'language', value: $('#language').val() },
            { key: 'date_format', value: $('#date_format').val() },
            { key: 'time_format', value: $('#time_format').val() },
            { key: 'speed_unit', value: $('#speed_unit').val() },
            { key: 'currency', value: $('#currency').val() },
            { key: 'weight_unit', value: $('#weight_unit').val() },
            { key: 'liquid_unit', value: $('#liquid_unit').val() },
            { key: 'length_unit', value: $('#length_unit').val() },
            { key: 'auto_backup_enabled', value: $('#auto_backup_enabled').is(':checked'), type: 'boolean' },
            { key: 'backup_frequency', value: $('#backup_frequency').val() }
        ];

        saveMultipleSettings(settings, 'system', '{{ __("System settings saved successfully") }}');
    }

    function saveMultipleSettings(settings, category, successMessage) {
        showLoading();

        http.post({
            url: '/api/settings/store-multiple',
            data: {
                category: category,
                settings: settings
            },
        }).then(function(response) {
            hideLoading();
            if (response.success) {
                showSuccessMessage(successMessage);
                // Reload settings to update the display
                loadSettings();

                localStorage.removeItem("app_settings");
                // Reload session settings to update JWT token
                loadSessionSettings();

                // If language was changed, update the page language
                if (category === 'system') {
                    const languageSetting = settings.find(s => s.key === 'language');
                    if (languageSetting) {
                        const currentLang = $('html').attr('lang');
                        if (languageSetting.value !== currentLang) {
                            // Update HTML lang attribute
                            $('html').attr('lang', languageSetting.value);
                            
                            // Show message about language change
                            showSuccessMessage('{{ __("Language changed. Page will reload in 2 seconds...") }}');
                            
                        }
                    }
                }
            } else {
                showErrorMessage(response.message || '{{ __("Error saving settings") }}');
            }
        }).catch(function(error) {
            hideLoading();
            showErrorMessage('{{ __("Error saving settings") }}');
            console.error('Error saving settings:', error);
        });
    }

    function calculateFuelCost() {
        const fuelType = $('#calc_fuel_type').val();
        const consumption = parseFloat($('#calc_consumption').val()) || 0;
        
        if (consumption > 0 && settingsData.fuel) {
            const fuelPrice = settingsData.fuel.find(item => item.setting_key === `fuel_price_${fuelType}`)?.typed_value || 0;
            const totalCost = consumption * fuelPrice;
            const currency = settingsData.system.find(item => item.setting_key === 'currency')?.typed_value || 'TRY';
            
            $('#calc_total_cost').val(totalCost.toFixed(2));
            $('#calc_currency').text(currency);
        } else {
            $('#calc_total_cost').val('');
        }
    }

    function updateMaintenanceSummary() {
        if (settingsData.maintenance) {
            $('#maintenance_summary_reminder').text(settingsData.maintenance.find(item => item.setting_key === 'maintenance_reminder_days')?.typed_value || '-');
            $('#maintenance_summary_oil').text(settingsData.maintenance.find(item => item.setting_key === 'oil_change_km')?.typed_value || '-');
            $('#maintenance_summary_tire').text(settingsData.maintenance.find(item => item.setting_key === 'tire_change_km')?.typed_value || '-');
            $('#maintenance_summary_status').text(settingsData.maintenance.find(item => item.setting_key === 'maintenance_enabled')?.typed_value ? '{{ __("Enabled") }}' : '{{ __("Disabled") }}');
        }
    }

    function updateFuelCalculator() {
        const currency = settingsData.system.find(item => item.setting_key === 'currency')?.typed_value || 'TRY';
        $('#calc_currency').text(currency);
        calculateFuelCost();
    }

    function showSuccessMessage(message) {
        toastr.success(message);
    }

    function showErrorMessage(message) {
        toastr.error(message);
    }
</script>
@endsection 