<?php $__env->startSection('title', __('Speed Violations Report')); ?>

<?php $__env->startSection('content'); ?>
<div class="page-wrapper" id="speed-violations-report">
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted"><?php echo e(__('Reports')); ?></a> / <?php echo e(__('Security and Compliance
                        Reports')); ?>

                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-gauge me-2"></i>
                        <?php echo e(__('Speed Violations Report')); ?>

                    </h2>
                    <div class="text-muted mt-1">
                        <?php echo e(__('Monitor and analyze vehicle speed violations')); ?>

                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Filters')); ?></h3>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('Vehicle')); ?></label>
                            <div class="input-group">
                                <select class="form-select" id="vehicleSelect">
                                    <option value=""><?php echo e(__('Select Vehicle')); ?></option>
                                </select>
                                <button type="button" class="btn btn-outline-secondary d-none" id="vehicleSettingsLink"
                                    onclick="goToVehicleSettings()" title="<?php echo e(__('Go to Vehicle Settings')); ?>">
                                    <i class="icon ti ti-settings"></i>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-3">

                            <label class="form-label"><?php echo e(__('Start Date')); ?></label>
                            <input type="date" class="form-control" id="startDate"
                                value="<?php echo e(date('Y-m-d', strtotime('-1 week'))); ?>">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('End Date')); ?></label>
                            <input type="date" class="form-control" id="endDate" value="<?php echo e(date('Y-m-d')); ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label"><?php echo e(__('Speed Limit (km/h)')); ?></label>
                            <input type="number" class="form-control" id="speedLimit" value="90" min="10" max="200">
                        </div>
                        <div class="col-md-1 d-flex align-items-end">
                            <button type="button" class="btn btn-primary w-100" onclick="generateReport()">
                                <i class="icon ti ti-search me-1"></i>
                                <?php echo e(__('Search')); ?>

                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="row row-cards mb-3" id="summaryCards">
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Total Violations')); ?></div>
                            <div class="h1 mb-0" id="totalViolations">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Highest Speed')); ?></div>
                            <div class="h1 mb-0" id="highestSpeed">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Average Speed')); ?></div>
                            <div class="h1 mb-0" id="averageSpeed">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Most Violated Vehicle')); ?></div>
                            <div class="h1 mb-0" id="mostViolatedVehicle">-</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Violations Table -->
            <div class="card mb-3" id="violationsTableCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Speed Violations')); ?></h3>
                </div>
                <div class="card-body p-0">
                    <div id="violations-table"></div>
                </div>
            </div>

            <!-- Map Visualization -->
            <div class="card mb-3 d-none" id="mapCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Violation Locations Map')); ?></h3>
                </div>
                <div class="card-body p-0">
                    <div id="violations-map" style="height: 500px;"></div>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden"><?php echo e(__('Loading...')); ?></span>
                </div>
                <div class="mt-2"><?php echo e(__('Generating speed violations report...')); ?></div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-speed text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No speed violations found')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('No speed violations were detected for the selected period and speed limit.')); ?>

                    </p>
                </div>
            </div>

            <!-- No Vehicle Selected State -->
            <div class="text-center py-4 d-none" id="noVehicleState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-car text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No Vehicle Selected')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('Please select a vehicle from the dropdown above to generate the speed violations report.')); ?>

                    </p>
                </div>
            </div>
        </div>
    </div>
</div>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('styles'); ?>
<style>
    #violations-map .map-icon-label {
        padding: 0px !important;
        border-width: 0px !important;
        font-size: 15px !important;
    }
</style>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('scripts'); ?>
<script>
    var speedUnit = "";
    var lengthUnit = "";
    var currency = "";
    
    let reportData = [];
    let violationsTable;
    let vehicleList = [];
    let myMap;
    let violation_groups = [];
    
        $('#summaryCards').addClass('d-none');
        $('#violationsTableCard').addClass('d-none');
        $('#mapCard').addClass('d-none');
        $('#loadingState').addClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').removeClass('d-none');
    
    // Wait for app to be fully initialized (translations + session settings)
    onAppReady(function() {        
        // Initialize the page after app is ready
        initializePage();
    });
    
    function initializePage() {
        loadVehicles()
        .then(initializeViolationsTable)
        .then(generateReport);
        
    }

    async function loadVehicles() {
        http.post({
            url: '/api/vehicles-list-with-alarm',
            data: { alarm_id: 304 }, // Speed alarm ID
            loading: true
        }).then(function(response) {
            if (response.success && response.data.vehicles) {
                const select = $('#vehicleSelect');
                select.empty();
            select.append('<option value=""><?php echo e(__('Please select a vehicle')); ?></option>');
            const vehiclesWithAlarm = [];
            const vehiclesWithoutAlarm = [];
            vehicleList = response.data.vehicles;
                response.data.vehicles.forEach(vehicle => {
                if (vehicle.has_alarm) {
                    vehiclesWithAlarm.push(vehicle);
                    } else {
                    vehiclesWithoutAlarm.push(vehicle);
                }
            });
            if (vehiclesWithAlarm.length > 0) {
                select.append('<optgroup label="<?php echo e(__('Vehicles with Speed Alarm')); ?>">');
                vehiclesWithAlarm.forEach(vehicle => {
                    var speed_limit = JSON.parse(vehicle.alarm_settings).speed_limit;
                    select.append(`<option value="${vehicle.did}" data-speed-limit="${speed_limit}">${vehicle.name} (${speed_limit} km/h) ✓</option>`);
                    });
                    select.append('</optgroup>');
                }
            if (vehiclesWithoutAlarm.length > 0) {
                select.append('<optgroup label="<?php echo e(__('Vehicles without Speed Alarm')); ?>">');
                vehiclesWithoutAlarm.forEach(vehicle => {
                    select.append(`<option value="${vehicle.did}" data-speed-limit="80">${vehicle.name} ✗</option>`);
                    });
                    select.append('</optgroup>');
                }
            if (vehiclesWithoutAlarm.length > 0) {
                    $('#vehicleSettingsLink').removeClass('d-none');
                } else {
                    $('#vehicleSettingsLink').addClass('d-none');
                }
            if (vehiclesWithAlarm.length === 0) {
                    showConfirmDialog({
                        title: '<?php echo e(__('Warning')); ?>',
                        message: '<?php echo e(__('No vehicles have speed alarm enabled. Please enable speed alarm for at least one vehicle to use this report.')); ?>',
                        confirmText: '<?php echo e(__('Go to Settings')); ?>',
                        showCancelButton: true,
                        type: 'warning',
                        onConfirm: function() {
                            window.location.href = '/vehicles-list';
                        },
                        onCancel: function() {}
                    });
                }
            }
        }).catch(function(error) {
            console.error('<?php echo e(__('Error loading vehicles')); ?>:', error);
        });
    }

    function goToVehicleSettings() {
        window.location.href = '/vehicles-list';
    }

    // Auto-populate speed limit when vehicle is selected
    $('#vehicleSelect').on('change', function() {
        const selectedOption = $(this).find('option:selected');
        const speedLimit = selectedOption.data('speed-limit');
        
        if (speedLimit) {
            $('#speedLimit').val(speedLimit);
        }
    });

    async function generateReport() {
        const vehicleId = $('#vehicleSelect').val();
        const startDate = $('#startDate').val();
        const endDate = $('#endDate').val();
        const speedLimit = $('#speedLimit').val();

        if(!vehicleId){
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("Please select a vehicle")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        if (!startDate || !endDate) {
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("Please select start and end dates")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Show loading state
        showLoadingState();

        const requestData = {
            device_id: vehicleId,
            start_date: startDate,
            end_date: endDate
        };

        http.post({
        url: '/api/reports/speed-violations',
        data: requestData,
            loading: false
        })
        .then(data => {
            hideLoadingState();
            
            if (data.success) {
                // Set units from session settings (now guaranteed to be loaded)
                speedUnit = __(window.sessionSettings.speedUnit || 'km/h');
                lengthUnit = __(window.sessionSettings.lengthUnit || 'km');
                currency = __(window.sessionSettings.currency || 'TRY');
                
                reportData = data.data;
                
                var message = __('#name# used #value# records').replace('#name#', __('Speed Violations Report')).replace('#value#', data.data.length);
                showReportDataProcessingInfo(message);
                
                processSpeedViolationsData(data.data.filter( c=> c.speed > 0));
            } else {
                showNoDataState();
                showConfirmDialog({
                    title: '<?php echo e(__("Error")); ?>',
                    message: data.message || '<?php echo e(__("Unknown error occurred")); ?>',
                    confirmText: '<?php echo e(__("OK")); ?>',
                    showCancelButton: false,
                    type: 'danger',
                    onConfirm: function() {},
                    onCancel: function() {}
                });
            }
        })
        .catch(err => {
            hideLoadingState();
            showNoDataState();
        var err = err;
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: __(err.message),
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
    });
}

// Process speed violations data
function processSpeedViolationsData(data) {
    
    if (!data || data.length === 0) {
            showNoDataState();
            return;
        }

    // Filter violations based on speed limit
    const speedLimit = parseFloat($('#speedLimit').val()) || 90;
    const violations = data.filter(log => parseFloat(log.speed) > speedLimit);

    //calculate polylines
    violation_groups = [];
    var current_group = null;
    
    // Group violations by consecutive speed violations using original data
    data.forEach((log, index) => {
        const speed = parseFloat(log.speed);
        
        if (speed > speedLimit) {
            // This is a violation - add to current group or start new group
            if (!current_group) {
                // Start new group
                current_group = {
                    start_lat: parseFloat(log.latitude),
                    start_lng: parseFloat(log.longitude),
                    start_time: log.timestamp,
                    start_speed: speed,
                    violations: [log],
                    group_index: violation_groups.length
                };
            } else {
                // Continue current group
                current_group.violations.push(log);
            }
        } else {
            // This is not a violation - end current group if exists
            if (current_group) {
                current_group.end_lat = parseFloat(log.latitude);
                current_group.end_lng = parseFloat(log.longitude);
                current_group.end_time = log.timestamp;
                current_group.end_speed = speed;
                
                violation_groups.push(current_group);
                current_group = null;
            }
        }
    });
    
    // Don't forget the last group if it ends with violations
    if (current_group) {
        // Use the last violation as end point
        const lastViolation = current_group.violations[current_group.violations.length - 1];
        current_group.end_lat = parseFloat(lastViolation.latitude);
        current_group.end_lng = parseFloat(lastViolation.longitude);
        current_group.end_time = lastViolation.timestamp;
        current_group.end_speed = parseFloat(lastViolation.speed);
        
        violation_groups.push(current_group);
    }

    if (violations.length === 0) {
        showNoDataState();
        return;
    }

    // Calculate summary statistics
    const summary = calculateViolationsSummary(violations, speedLimit);
    
    // Display the processed data
    displaySpeedViolationsReport({
        summary: summary,
        violations: violations,
        speed_limit: speedLimit
    });
}

// Calculate summary statistics for speed violations
function calculateViolationsSummary(violations, speedLimit) {
    const speeds = reportData.filter(c=> c.speed > 0).map(v => parseFloat(v.speed));
    const highestSpeed = Math.round(Math.max(...speeds));
    const averageSpeed = Math.round(speeds.reduce((sum, speed) => sum + speed, 0) / speeds.length);
    
    // Group violations by date
    const dateCounts = {};
    violations.forEach(violation => {
        const date = violation.timestamp.split(' ')[0]; // Get date part
        dateCounts[date] = (dateCounts[date] || 0) + 1;
    });
    
    // Find most violated date
    let mostViolatedDate = null;
    if (Object.keys(dateCounts).length > 0) {
        const maxDate = Object.keys(dateCounts).reduce((a, b) => dateCounts[a] > dateCounts[b] ? a : b);
        mostViolatedDate = {
            date: maxDate,
            violations: dateCounts[maxDate]
        };
    }
    
    return {
        total_violations: violations.length,
        highest_speed: Math.round(highestSpeed),
        average_speed: Math.round(averageSpeed * 10) / 10,
        most_violated_date: mostViolatedDate,
        speed_limit: speedLimit
    };
}

// Display speed violations report
function displaySpeedViolationsReport(data) {
    // Show all sections
    $('#summaryCards').removeClass('d-none');
    $('#violationsTableCard').removeClass('d-none');
    $('#mapCard').removeClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#noVehicleState').addClass('d-none');

    // Update summary cards
    updateSummary(data.summary);

    // Update map with a small delay to ensure the map container is visible
    setTimeout(() => {
        // Update table
        updateTable(data.violations);
        // Update map
        updateMap(data.violations);
    }, 100);
}

function updateSummary(summary) {
    $('#totalViolations').text(summary.total_violations);
    $('#highestSpeed').text(summary.highest_speed + ' ' + speedUnit);
    $('#averageSpeed').text(summary.average_speed + ' ' + speedUnit);
    
    if (summary.most_violated_date) {
        $('#mostViolatedVehicle').text(summary.most_violated_date.violations + ' <?php echo e(__("violations")); ?>');
    } else {
        $('#mostViolatedVehicle').text('-');
    }
}

function updateTable(violations) {
    // Table should already be initialized, but check just in case
    if (!violationsTable) {
        initializeViolationsTable();
    }
    
    const tableData = violations.map((violation, index) => ({
        id: index + 1,
        date: violation.timestamp.split(' ')[0],
        time: violation.timestamp.split(' ')[1],
        vehicle: vehicleList.find(v => v.did === violation.device_id)?.nm || violation.device_id,
        speed: Math.round(parseFloat(violation.speed)),
        speed_limit: parseFloat($('#speedLimit').val()) || 90,
        speed_excess: Math.round(parseFloat(violation.speed) - (parseFloat($('#speedLimit').val()) || 90)),
        location: `${violation.latitude}, ${violation.longitude}`,
        direction: violation.direction || '-',
        latitude: violation.latitude,
        longitude: violation.longitude
    }));

    violationsTable.setData(tableData);
}

function updateMap(violations) {
    if (!myMap) {
        initializeMap();
    }
    
    // Refresh map to ensure proper display when container becomes visible
    if (myMap.mapLet) {
        myMap.mapLet.invalidateSize();
    }
    
    // Clear existing markers and polylines
    myMap.removeMarkers();
    myMap.clearAllPolygons();
    myMap.removeCircles();
    myMap.removePolyline();
    
    if (!violation_groups || violation_groups.length === 0) {
        return;
    }
    
    // Add regions and places to the map
    addRegionsAndPlacesToMap();
    
    // Generate distinct colors for different groups    
    const groupColors = generateDistinctColors(violation_groups.length);
    
    // Create markers and polylines for each violation group
    const markerModels = [];
    
    violation_groups.forEach((group, groupIndex) => {
        if (group.violations && group.violations.length > 0) {
            const groupColor = groupColors[groupIndex % groupColors.length];
            
            // Create markers for each violation in the group
            group.violations.forEach((violation, violationIndex) => {
                const lat = parseFloat(violation.latitude);
                const lng = parseFloat(violation.longitude);
                
                if (!isNaN(lat) && !isNaN(lng)) {
                    // Determine if this is start or end point
                    const isStart = violationIndex === 0;
                    const isEnd = violationIndex === group.violations.length - 1;
                    
                    let markerIcon = "ti ti-speed";
                    let markerColor = groupColor;
                    let markerLabel = "";
                    
                    if (isStart) {
                        markerIcon = "ti ti-player-play";
                        markerLabel = `<div class="badge text-white" style="background-color: ${groupColor};"><?php echo e(__("START")); ?></div>`;
                    } else if (isEnd) {
                        markerIcon = "ti ti-player-stop";
                        markerLabel = `<div class="badge text-white" style="background-color: ${groupColor};"><?php echo e(__("END")); ?></div>`;
                    } else {
                        markerLabel = `<div class="badge text-white" style="background-color: ${groupColor};">${Math.round(parseFloat(violation.speed))} ${speedUnit}</div>`;
                    }
                    
                    // Create marker
                    markerModels.push({
                        id: `violation-${violation.id}-group-${groupIndex}`,
                        markerClassName: "speed-violation-marker",
                        label: markerLabel,
                        speed: Math.round(parseFloat(violation.speed)),
                        device_id: violation.device_id,
                        popup: `
                            <strong><?php echo e(__("Speed Violation")); ?> - Group #${groupIndex + 1}</strong><br>
                            <?php echo e(__("Date")); ?>: ${dateTimeFormatter(violation.timestamp.split(' ')[0])}<br>
                            <?php echo e(__("Time")); ?>: ${violation.timestamp.split(' ')[1]}<br>
                            <?php echo e(__("Speed")); ?>: ${Math.round(parseFloat(violation.speed))} ${speedUnit}<br>
                            <?php echo e(__("Speed Limit")); ?>: ${parseFloat($('#speedLimit').val()) || 90} ${speedUnit}<br>
                            <?php echo e(__("Excess")); ?>: ${Math.round(parseFloat(violation.speed) - (parseFloat($('#speedLimit').val()) || 90))} ${speedUnit}<br>
                            <?php echo e(__("Direction")); ?>: ${violation.direction || '-'}°<br>
                            ${isStart ? '<strong><?php echo e(__("START POINT")); ?></strong>' : ''}
                            ${isEnd ? '<strong><?php echo e(__("END POINT")); ?></strong>' : ''}
                        `,
                        coord: [lat, lng],
                        icon: {
                            name: markerIcon,
                            color: groupColor,
                            direction: 0,
                            className: "speed-violation-marker",
                        }
                    });
                }
            });
            
            // Create polyline for this group
            const polylineCoords = group.violations.map(violation => [
                parseFloat(violation.latitude),
                parseFloat(violation.longitude)
            ]);
            
            // Use map.js polyline method for each violation group
            myMap.addPolyline({
                coords: polylineCoords,
                color: groupColor,
                weight: 5,
                opacity: 0.8
            });
            
            // Add popup to the polyline group
            setTimeout(() => {
                const totalDistance = calculatePolylineDistance(polylineCoords);
                const maxSpeed = Math.max(...group.violations.map(v => parseFloat(v.speed)));
                const minSpeed = Math.min(...group.violations.map(v => parseFloat(v.speed)));
                
                // Find the polyline layer and add popup
                myMap.mapLet.eachLayer((layer) => {
                    if (layer instanceof L.Polyline && layer.getLatLngs().length === polylineCoords.length) {
                        const layerCoords = layer.getLatLngs();
                        const firstMatch = layerCoords[0].lat === polylineCoords[0][0] && layerCoords[0].lng === polylineCoords[0][1];
                        const lastMatch = layerCoords[layerCoords.length - 1].lat === polylineCoords[polylineCoords.length - 1][0] && 
                                       layerCoords[layerCoords.length - 1].lng === polylineCoords[polylineCoords.length - 1][1];
                        
                        if (firstMatch && lastMatch) {
                            layer.bindPopup(`
                                <strong><?php echo e(__("Violation Group")); ?> #${groupIndex + 1}</strong><br>
                                                            <?php echo e(__("Start")); ?>: ${dateTimeFormatter(group.start_time.split(' ')[0])} ${group.start_time.split(' ')[1]}<br>
                            <?php echo e(__("End")); ?>: ${dateTimeFormatter(group.end_time.split(' ')[0])} ${group.end_time.split(' ')[1]}<br>
                                <?php echo e(__("Distance")); ?>: ${Math.round(totalDistance * 100) / 100} ${lengthUnit}<br>
                                <?php echo e(__("Violations")); ?>: ${group.violations.length}<br>
                                <?php echo e(__("Max Speed")); ?>: ${Math.round(maxSpeed)} ${speedUnit}<br>
                                <?php echo e(__("Min Speed")); ?>: ${Math.round(minSpeed)} ${speedUnit}<br>
                                <?php echo e(__("Speed Limit")); ?>: ${parseFloat($('#speedLimit').val()) || 90} ${speedUnit}
                            `);
                        }
                    }
                });
            }, 100);
        }
    });
    
    // Add markers to map
    if (markerModels.length > 0) {
        myMap.pinMarkers(markerModels);

        // Fit map to show all markers
        setTimeout(() => myMap.fitBounds(), 100);
    }
    }

    // Calculate distance between two GPS coordinates (Haversine formula)
    function calculateDistance(lat1, lon1, lat2, lon2) {
        const lat1Num = parseFloat(lat1);
        const lon1Num = parseFloat(lon1);
        const lat2Num = parseFloat(lat2);
        const lon2Num = parseFloat(lon2);
        const earthRadius = 6371;
        const latDelta = (lat2Num - lat1Num) * Math.PI / 180;
        const lonDelta = (lon2Num - lon1Num) * Math.PI / 180;
        const a = Math.sin(latDelta / 2) * Math.sin(latDelta / 2) +
                 Math.cos(lat1Num * Math.PI / 180) * Math.cos(lat2Num * Math.PI / 180) *
                 Math.sin(lonDelta / 2) * Math.sin(lonDelta / 2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        const distance = earthRadius * c;
        return distance;
    }

// Calculate distance of polyline
function calculatePolylineDistance(points) {
    let totalDistance = 0;
    for (let i = 1; i < points.length; i++) {
        const prevPoint = points[i - 1];
        const currentPoint = points[i];
        totalDistance += calculateDistance(prevPoint[0], prevPoint[1], currentPoint[0], currentPoint[1]);
    }
    return totalDistance;
}

// Initialize map
function initializeMap() {
    myMap = new LeafletMap();
    myMap.initMap("violations-map", {
        // Default zoom level
    });
}
    
    // Refresh map function
    function refreshMap() {
        if (myMap && myMap.mapLet) {
            myMap.mapLet.invalidateSize();
        }
    }

// Initialize table
function initializeViolationsTable() {
    violationsTable = createTabulator("#violations-table", {
        initialSort: [
            {column: "date", dir: "desc"},
            {column: "time", dir: "desc"}
        ],
        columns: [
            {title: "#", field: "id", width: 50, hozAlign: "center"},
            {
                title: "<?php echo e(__('Date')); ?>", 
                field: "date",
                formatter: function(cell) {
                    return dateTimeFormatter(cell.getValue());
                }
            },
            {
                title: "<?php echo e(__('Time')); ?>", 
                field: "time",
                formatter: function(cell) {
                    return formatTime(cell.getValue());
                }
            },
            {
                title: "<?php echo e(__('Vehicle')); ?>", 
                field: "vehicle"
            },
            {
                title: "<?php echo e(__('Speed')); ?>", 
                field: "speed", 
                hozAlign: "center",
                formatter: function(cell) {
                    const speed = cell.getValue();
                    const speedLimit = cell.getData().speed_limit;
                    const excess = cell.getData().speed_excess;
                    if (excess > 20) {
                        return `<span class="badge text-white bg-danger">${speed} ${speedUnit}</span>`;
                    } else if (excess > 10) {
                        return `<span class="badge text-white bg-warning">${speed} ${speedUnit}</span>`;
                    } else {
                        return `<span class="badge text-white bg-info">${speed} ${speedUnit}</span>`;
                    }
                }
            },
            {
                title: "<?php echo e(__('Speed Limit')); ?>", 
                field: "speed_limit", 
                hozAlign: "center",
                formatter: function(cell) {
                    return `${cell.getValue()} ${speedUnit}`;
                }
            },
            {
                title: "<?php echo e(__('Excess')); ?>", 
                field: "speed_excess", 
                hozAlign: "center",
                formatter: function(cell) {
                    const excess = cell.getValue();
                    if (excess > 20) {
                        return `<span class="badge text-white bg-danger">+${excess} ${speedUnit}</span>`;
                    } else if (excess > 10) {
                        return `<span class="badge text-white bg-warning">+${excess} ${speedUnit}</span>`;
                    } else {
                        return `<span class="badge text-white bg-info">+${excess} ${speedUnit}</span>`;
                    }
                }
            },
            {
                title: "<?php echo e(__('Direction')); ?>", 
                field: "direction", 
                hozAlign: "center",
                formatter: function(cell) {
                    const direction = cell.getValue();
                    if (direction && direction !== '-') {
                        return `${direction}°`;
                    }
                    return '-';
                }
            },
            {
                title: "<?php echo e(__('Location')); ?>", 
                field: "location", 
                formatter: function(cell) {
                    const data = cell.getData();
                    if (data.latitude && data.longitude) {
                        return `<a href="#" class="btn btn-sm btn-outline-primary map-link" onclick="showLocationOnMap(${data.latitude}, ${data.longitude}); return false;"><?php echo e(__('See on map')); ?></a>`;
                    }
                    return '-';
                }
            }
        ]
    });
    }

    // Add regions and places to the map
    function addRegionsAndPlacesToMap() {
        // Load regions data
        http.post({
            url: '/regions-list',
            loading: false
        }).then(function(regionsResponse) {
            if (regionsResponse.success && regionsResponse.data) {
                const regions = regionsResponse.data;
                
                // Add regions to map
                regions.forEach(function(region) {
                    if (region.polygon_json) {
                        myMap.addPolygon({
                            id: region.id,
                            label: region.name,
                            color: "gray",
                            fillColor: "gray",
                            coords: region.polygon_json,
                        });
                    }
                });
            }
        }).catch(function(error) {
            console.error('Error loading regions:', error);
        });

        // Load places data
        http.post({
            url: '/places-list',
            loading: false
        }).then(function(placesResponse) {
            if (placesResponse.success && placesResponse.data) {
                const places = placesResponse.data;
                
                // Add places to map
                var placeMarkerColor = "gray";
                places.forEach(function(place) {
                    if (place.latitude && place.longitude) {
                        var marker = [
                            {
                                id: place.id,
                                markerClassName: "place-marker",
                                device_id: 0,
                                label: place.name,
                                status: place.status_id,
                                popup: place.name,
                                coord: [
                                    parseFloat(place.latitude),
                                    parseFloat(place.longitude),
                                ],
                                icon: {
                                    name: "fa-solid fa-location-dot",
                                    color: placeMarkerColor,
                                    direction: 0,
                                    className: "",
                                },
                            },
                        ];
                        myMap.pinMarkers(marker);
                        
                        // Add circle if radius is defined
                        if (place.radius && place.radius > 0) {
                            var circle = L.circle([place.latitude, place.longitude], {
                                radius: place.radius,
                                color: placeMarkerColor,
                                fillColor: placeMarkerColor,
                                fillOpacity: 0.2,
                            }).addTo(myMap.mapLet);
                        }
                    }
                });
            }
        }).catch(function(error) {
            console.error('Error loading places:', error);
        });
    }

    function showLocationOnMap(lat, lng) {
        if (!lat || !lng || isNaN(lat) || isNaN(lng)) {
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: '<?php echo e(__("Invalid coordinates")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }
        
        // Prevent any default scrolling behavior
        event.preventDefault();
        
        // Ensure map container is visible first
        $('#mapCard').removeClass('d-none');
        
        // Refresh map to ensure proper display
        if (myMap && myMap.mapLet) {
            myMap.mapLet.invalidateSize();
        }
        
        // Center map on the location with smooth animation
        myMap.mapLet.setView([parseFloat(lat), parseFloat(lng)], 16, {
            animate: true,
            duration: 0.5
        });
        
        // Add a temporary marker to highlight the location
        const tempMarker = L.marker([lat, lng], {
            icon: L.divIcon({
                className: 'temp-location-marker',
                html: '<i class="fa-solid fa-crosshairs" style="color: red; font-size: 20px;"></i>',
                iconSize: [20, 20],
                iconAnchor: [10, 10]
            })
        }).addTo(myMap.mapLet);
        
        // Remove the temporary marker after 3 seconds
        setTimeout(() => {
            myMap.mapLet.removeLayer(tempMarker);
        }, 3000);
        
        // Focus on map
    $('#violations-map').focus();
}

// Show loading state
function showLoadingState() {
    $('#loadingState').removeClass('d-none');
    $('#summaryCards').addClass('d-none');
    $('#violationsTableCard').addClass('d-none');
    $('#mapCard').addClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#noVehicleState').addClass('d-none');
}

// Hide loading state
function hideLoadingState() {
    $('#loadingState').addClass('d-none');
}

// Show no data state
function showNoDataState() {
    $('#noDataState').removeClass('d-none');
    $('#summaryCards').addClass('d-none');
    $('#violationsTableCard').addClass('d-none');
    $('#mapCard').addClass('d-none');
    $('#noVehicleState').addClass('d-none');
}

// Utility functions
function formatDate(dateString) {
    return new Date(dateString).toLocaleDateString();
}

function formatTime(timeString) {
    return timeString;
    }
</script>
<?php $__env->stopSection(); ?>
<?php echo $__env->make('layouts.app', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?><?php /**PATH C:\xampp\htdocs\aractakipv1\resources\views/reports/speed-violations.blade.php ENDPATH**/ ?>