<?php $__env->startSection('title', __('Fuel Consumption Analysis')); ?>

<?php $__env->startSection('content'); ?>
<div class="page-wrapper" id="fuel-consumption-report">
    <div class="page-header d-print-none">
        <div class="container-xl">
            <div class="row g-2 align-items-center">
                <div class="col">
                    <div class="page-pretitle">
                        <a href="/reports" class="text-muted"><?php echo e(__('Reports')); ?></a> / <?php echo e(__('Cost and Efficiency Reports')); ?>

                    </div>
                    <h2 class="page-title">
                        <i class="ti ti-gas-station me-2"></i>
                        <?php echo e(__('Fuel Consumption Analysis')); ?>

                    </h2>
                    <div class="text-muted mt-1">
                        <?php echo e(__('Fuel consumption and cost analysis based on vehicle movement data')); ?>

                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="page-body">
        <div class="container-xl">
            <!-- Filters -->
            <div class="card mb-3">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Filters')); ?></h3>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label"><?php echo e(__('Vehicle')); ?></label>
                            <div class="input-group">
                                <select class="form-select" id="vehicleSelect">
                                    <option value=""><?php echo e(__('All Vehicles')); ?></option>
                                </select>
                                <button type="button" class="btn btn-outline-secondary d-none" id="vehicleSettingsLink"
                                    onclick="goToVehicleSettings()" title="<?php echo e(__('Go to Vehicle Settings')); ?>">
                                    <i class="icon ti ti-settings"></i>
                                </button>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label"><?php echo e(__('Fuel Type')); ?></label>
                            <select class="form-select" id="fuelTypeFilter">
                                <option value=""><?php echo e(__('All Types')); ?></option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label"><?php echo e(__('Start Date')); ?></label>
                            <input type="date" class="form-control" id="startDate"
                                value="<?php echo e(date('Y-m-d', strtotime('-1 week'))); ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label"><?php echo e(__('End Date')); ?></label>
                            <input type="date" class="form-control" id="endDate" value="<?php echo e(date('Y-m-d')); ?>">
                        </div>
                        <div class="col-md-2 d-flex align-items-end">
                            <button type="button" class="btn btn-primary w-100" onclick="generateReport()">
                                <i class="icon ti ti-search me-1"></i>
                                <?php echo e(__('Search')); ?>

                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Vehicle Settings Modal -->
            <div class="modal fade" id="vehicleSettingsModal" tabindex="-1" aria-labelledby="vehicleSettingsModalLabel" aria-hidden="true">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title" id="vehicleSettingsModalLabel"><?php echo e(__('Configure Vehicle Fuel Settings')); ?></h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                        </div>
                        <div class="modal-body">
                            <div class="alert alert-info">
                                <i class="ti ti-info-circle me-2"></i>
                                <?php echo e(__('Some vehicles do not have fuel consumption data configured. Please set the fuel type and consumption for accurate reporting.')); ?>

                            </div>
                            
                            <div id="vehicleSettingsList">
                                <!-- Vehicle settings will be populated here -->
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal"><?php echo e(__('Cancel')); ?></button>
                            <button type="button" class="btn btn-primary" onclick="saveVehicleSettings()"><?php echo e(__('Save Settings')); ?></button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="row row-cards mb-3" id="summaryCards">
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Total Distance')); ?></div>
                            <div class="h1 mb-0" id="totalDistance">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Total Fuel Used')); ?></div>
                            <div class="h1 mb-0" id="totalFuelUsed">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Total Cost')); ?></div>
                            <div class="h1 mb-0" id="totalCost">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Average Efficiency')); ?></div>
                            <div class="h1 mb-0" id="averageEfficiency">-</div>
                        </div>
                    </div>
                </div>
                <div class="col-sm-6 col-lg-2">
                    <div class="card">
                        <div class="card-body">
                            <div class="subheader"><?php echo e(__('Days Count')); ?></div>
                            <div class="h1 mb-0" id="daysCount">-</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Color Legend -->
            <div class="card mb-3" id="colorLegendCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Fuel Type Color Guide')); ?></h3>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-2 col-sm-4 col-6">
                            <div class="d-flex align-items-center">
                                <div class="badge text-white me-2" style="background-color: #28a745; width: 20px; height: 20px;"></div>
                                <span><?php echo e(__('Diesel')); ?> <span class="fuel-type-count text-success" data-fuel-type="diesel">0</span><span class="plus-icon" style="opacity:0;">+</span></span>
                            </div>
                        </div>
                        <div class="col-md-2 col-sm-4 col-6">
                            <div class="d-flex align-items-center">
                                <div class="badge text-white me-2" style="background-color: #ffc107; width: 20px; height: 20px;"></div>
                                <span><?php echo e(__('Gasoline')); ?> <span class="fuel-type-count text-warning" data-fuel-type="gasoline">0</span><span class="plus-icon" style="opacity:0;">+</span></span>
                            </div>
                        </div>
                        <div class="col-md-2 col-sm-4 col-6">
                            <div class="d-flex align-items-center">
                                <div class="badge text-white me-2" style="background-color: #dc3545; width: 20px; height: 20px;"></div>
                                <span><?php echo e(__('Gas (LPG/CNG)')); ?> <span class="fuel-type-count text-danger" data-fuel-type="gas">0</span><span class="plus-icon" style="opacity:0;">+</span></span>
                            </div>
                        </div>
                        <div class="col-md-2 col-sm-4 col-6">
                            <div class="d-flex align-items-center">
                                <div class="badge text-white me-2" style="background-color: #6f42c1; width: 20px; height: 20px;"></div>
                                <span><?php echo e(__('Hybrid')); ?> <span class="fuel-type-count text-primary" data-fuel-type="hybrid">0</span><span class="plus-icon" style="opacity:0;">+</span></span>
                            </div>
                        </div>
                        <div class="col-md-2 col-sm-4 col-6">
                            <div class="d-flex align-items-center">
                                <div class="badge text-white me-2" style="background-color: #17a2b8; width: 20px; height: 20px;"></div>
                                <span><?php echo e(__('Electric')); ?> <span class="fuel-type-count text-info" data-fuel-type="electric">0</span><span class="plus-icon" style="opacity:0;">+</span></span>
                            </div>
                        </div>
                        <div class="col-md-2 col-sm-4 col-6">
                            <div class="d-flex align-items-center">
                                <div class="badge text-white me-2" style="background-color: #007bff; width: 20px; height: 20px;"></div>
                                <span><?php echo e(__('Default')); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Consumption Table -->
            <div class="card  mb-3" id="consumptionTableCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Fuel Consumption Details')); ?></h3>
                </div>
                <div class="card-body p-0">
                    <div id="consumption-table"></div>
                </div>
            </div>

            <!-- Charts Section -->
            <div class="row row-cards mb-3 d-none" id="chartsSection">
                <!-- Daily Consumption Chart -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Daily Fuel Consumption')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="dailyConsumptionChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Fuel Type Distribution Chart -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Fuel Type Distribution')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="fuelTypeChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Cost Analysis Chart -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Daily Cost Analysis')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="costAnalysisChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Vehicle Comparison Chart -->
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title"><?php echo e(__('Consumption by Vehicle')); ?></h3>
                        </div>
                        <div class="card-body">
                            <div id="vehicleComparisonChart" style="height: 300px;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Map Visualization -->
            <div class="card mb-3 d-none" id="mapCard">
                <div class="card-header">
                    <h3 class="card-title"><?php echo e(__('Route Visualization')); ?></h3>
                </div>
                <div class="card-body p-0">
                    <div id="route-map" style="height: 500px;"></div>
                </div>
            </div>

            <!-- Loading State -->
            <div class="text-center py-4 d-none" id="loadingState">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden"><?php echo e(__('Loading...')); ?></span>
                </div>
                <div class="mt-2"><?php echo e(__('Generating fuel consumption report...')); ?></div>
            </div>

            <!-- No Data State -->
            <div class="text-center py-4 d-none" id="noDataState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-gas-station text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No fuel consumption data found')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('No fuel consumption data was found for the selected period.')); ?>

                    </p>
                </div>
            </div>

            <!-- No Vehicle Selected State -->
            <div class="text-center py-4 d-none" id="noVehicleState">
                <div class="empty">
                    <div class="empty-img">
                        <i class="ti ti-car text-muted" style="font-size: 3rem;"></i>
                    </div>
                    <p class="empty-title"><?php echo e(__('No Vehicle Selected')); ?></p>
                    <p class="empty-subtitle text-muted">
                        <?php echo e(__('Please select a vehicle from the dropdown above to generate the fuel consumption analysis report.')); ?>

                    </p>
                </div>
            </div>
        </div>
    </div>
</div>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('styles'); ?>
<style>
    #route-map .map-icon-label {
        padding: 0px !important;
        border-width: 0px !important;
        font-size: 15px !important;
    }
    .fuel-type-count {
        font-weight: bold;
        font-size: 1.3em;
        margin-left: 4px;
        text-shadow: 0 1px 4px rgba(0,0,0,0.12);
        display: inline-block;
        transition: transform 0.2s cubic-bezier(.68,-0.55,.27,1.55), color 0.2s;
    }
    .fuel-type-count.pop {
        transform: scale(1.4);
        color: #222;
        text-shadow: 0 2px 8px rgba(0,0,0,0.18);
    }
    .plus-icon {
        font-size: 1em;
        margin-left: 2px;
        color: #28a745;
        opacity: 0;
        transition: opacity 0.4s, transform 0.3s;
        transform: scale(1.2);
        vertical-align: middle;
    }
    .plus-icon.visible {
        opacity: 1;
        transform: scale(1.4);
    }
</style>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('scripts'); ?>
<script>

    let reportData = [];
    let consumptionTable;
    let vehicleList = [];
    let myMap;
    let fuelPrices = {};
    let defaultFuelConsumption = 8; // Default liters per 100km

    var speedUnit = '';
    var lengthUnit = '';
    var liquidUnit = '';
    var currency = '';

    $('#summaryCards').addClass('d-none');
    $('#consumptionTableCard').addClass('d-none');
    $('#mapCard').addClass('d-none');
    $('#loadingState').removeClass('d-none');
    $('#noDataState').addClass('d-none');
    $('#noVehicleState').addClass('d-none');
    
    // Fuel type translation function
    function translateFuelType(fuelType) {
        const fuelTypeTranslations = {
            'diesel': '<?php echo e(__("Diesel")); ?>',
            'gasoline': '<?php echo e(__("Gasoline")); ?>',
            'gas': '<?php echo e(__("Gas (LPG/CNG)")); ?>',
            'hybrid': '<?php echo e(__("Hybrid")); ?>',
            'electric': '<?php echo e(__("Electric")); ?>'
        };
        
        return fuelTypeTranslations[fuelType] || fuelType;
    }

    onAppReady(function() {        
        loadVehicles()
        .then(initializeConsumptionTable)
        .then(generateReport);
    });

    async function loadVehicles() {
        http.post({
            url: '/api/vehicles/list',
            loading: false
        }).then(function(response) {
            if (response.success && response.data.vehicles) {
                const select = $('#vehicleSelect');
                select.empty();
                select.append('<option value=""><?php echo e(__('All Vehicles')); ?></option>');
                
                const vehiclesWithFuelData = [];
                const vehiclesWithoutFuelData = [];
                vehicleList = response.data.vehicles;
                
                // Count vehicles by fuel type
                const fuelTypeCounts = {};
                
                response.data.vehicles.forEach(vehicle => {
                    // Check if vehicle has fuel consumption data
                    const fuelConsumption = vehicle.fuel_consumption || defaultFuelConsumption;
                    const fuelType = vehicle.fuel_type || '';
                    
                    if (vehicle.fuel_consumption && vehicle.fuel_type) {
                        vehiclesWithFuelData.push({
                            ...vehicle,
                            fuelConsumption: fuelConsumption,
                            fuelType: fuelType
                        });
                        
                        // Count vehicles with fuel data
                        fuelTypeCounts[fuelType] = (fuelTypeCounts[fuelType] || 0) + 1;
                    } else {
                        vehiclesWithoutFuelData.push({
                            ...vehicle,
                            fuelConsumption: defaultFuelConsumption,
                            fuelType: fuelType
                        });
                    }
                });
                
                if (vehiclesWithFuelData.length > 0) {
                    select.append('<optgroup label="<?php echo e(__('Vehicles with Fuel Data')); ?>">');
                    vehiclesWithFuelData.forEach(vehicle => {
                        select.append(`<option value="${vehicle.did}" data-fuel-consumption="${vehicle.fuelConsumption}" data-fuel-type="${vehicle.fuelType}">${vehicle.name} (${vehicle.fuelConsumption}L/100km, ${vehicle.fuelType}) ✓</option>`);
                    });
                    select.append('</optgroup>');
                }
                
                if (vehiclesWithoutFuelData.length > 0) {
                    select.append('<optgroup label="<?php echo e(__('Vehicles without Fuel Data')); ?>">');
                    vehiclesWithoutFuelData.forEach(vehicle => {
                        select.append(`<option value="${vehicle.did}" data-fuel-consumption="${vehicle.fuelConsumption}" data-fuel-type="${vehicle.fuelType}">${vehicle.name} (${vehicle.fuelConsumption}L/100km default) ✗</option>`);
                    });
                    select.append('</optgroup>');
                }
                
                // Update fuel type filter with vehicle counts
                updateFuelTypeFilter(fuelTypeCounts);
                
                if (vehiclesWithoutFuelData.length > 0) {
                    $('#vehicleSettingsLink').removeClass('d-none');
                    // Show modal for vehicles without fuel data
                    showVehicleSettingsModal(vehiclesWithoutFuelData);
                } else {
                    $('#vehicleSettingsLink').addClass('d-none');
                }
                
                if (vehiclesWithFuelData.length === 0) {
                    showConfirmDialog({
                        title: '<?php echo e(__('Warning')); ?>',
                        message: '<?php echo e(__('No vehicles have fuel consumption data configured. Please configure fuel consumption settings for at least one vehicle to get accurate reports.')); ?>',
                        confirmText: '<?php echo e(__('Go to Settings')); ?>',
                        cancelText: '<?php echo e(__('Cancel')); ?>',
                        type: 'warning',
                        onConfirm: function() {
                            window.location.href = '/vehicles-list';
                        },
                        onCancel: function() {}
                    });
                }
            }
        }).catch(function(error) {
            console.error('<?php echo e(__('Error loading vehicles')); ?>:', error);
        });
    }

    function updateFuelTypeFilter(fuelTypeCounts) {
        const fuelTypeFilter = $('#fuelTypeFilter');
        const currentValue = fuelTypeFilter.val();
        
        fuelTypeFilter.empty();
        fuelTypeFilter.append('<option value=""><?php echo e(__('All Types')); ?></option>');
        
        const fuelTypes = [
            { value: 'diesel', label: '<?php echo e(__("Diesel")); ?>' },
            { value: 'gasoline', label: '<?php echo e(__("Gasoline")); ?>' },
            { value: 'gas', label: '<?php echo e(__("Gas (LPG/CNG)")); ?>' },
            { value: 'hybrid', label: '<?php echo e(__("Hybrid")); ?>' },
            { value: 'electric', label: '<?php echo e(__("Electric")); ?>' }
        ];
        
        fuelTypes.forEach(fuelType => {
            const count = fuelTypeCounts[fuelType.value] || 0;
            const optionText = count > 0 ? `${fuelType.label} (${count})` : fuelType.label;
            const option = `<option value="${fuelType.value}">${optionText}</option>`;
            fuelTypeFilter.append(option);
        });
        
        // Restore previous selection if it still exists
        if (currentValue) {
            fuelTypeFilter.val(currentValue);
        }
        
        animateFuelTypeCounts(fuelTypeCounts);
    }

    function animateFuelTypeCounts(fuelTypeCounts) {
        Object.keys(fuelTypeCounts).forEach(function(type) {
            const el = document.querySelector('.fuel-type-count[data-fuel-type="' + type + '"]');
            const plus = el ? el.nextElementSibling : null;
            if (el) {
                let start = 0;
                const end = fuelTypeCounts[type];
                const duration = 900;
                const startTime = performance.now();
                function animateCount(now) {
                    const elapsed = now - startTime;
                    const progress = Math.min(elapsed / duration, 1);
                    const value = Math.floor(progress * end);
                    el.textContent = value;
                    if (progress < 1) {
                        requestAnimationFrame(animateCount);
                    } else {
                        el.textContent = end;
                        el.classList.add('pop');
                        setTimeout(() => el.classList.remove('pop'), 350);
                        if (plus && end > 0) {
                            plus.classList.add('visible');
                        }
                    }
                }
                el.textContent = '0';
                if (plus) plus.classList.remove('visible');
                if (end > 0) {
                    requestAnimationFrame(animateCount);
                } else {
                    el.textContent = '0';
                }
            }
        });
    }

    function goToVehicleSettings() {
        window.location.href = '/vehicles-list';
    }

    async function generateReport() {
        const vehicleId = $('#vehicleSelect').val();
        const fuelTypeFilter = $('#fuelTypeFilter').val();
        const startDate = $('#startDate').val();
        const endDate = $('#endDate').val();

        if (!startDate || !endDate) {
            showConfirmDialog({
                title: '<?php echo e(__("Warning")); ?>',
                message: '<?php echo e(__("Please select start and end dates")); ?>',
                confirmText: '<?php echo e(__("OK")); ?>',
                cancelText: '',
                type: 'warning',
                onConfirm: function() {},
                onCancel: function() {}
            });
            return;
        }

        // Show loading state
        showLoadingState();

        const requestData = {
            start_date: startDate,
            end_date: endDate
        };
        
        // Only add device_id if a specific vehicle is selected
        if (vehicleId) {
            requestData.device_id = vehicleId;
        }

        // Add fuel type filter if selected
        if (fuelTypeFilter) {
            requestData.fuel_type = fuelTypeFilter;
        }

        http.post({
            url: '/api/reports/fuel-consumption',
            data: requestData,
            loading: false
        })
        .then(data => {
            hideLoadingState();
            
            if (data.success) {

                speedUnit = __(sessionSettings.speedUnit || 'km/h');
                lengthUnit = sessionSettings.lengthUnit || 'km';
                liquidUnit = sessionSettings.liquidUnit || 'lt';
                currency = sessionSettings.currency || 'TRY';

                reportData = data.data.device_logs;
                fuelPrices = data.data.fuel_prices;

                var message = __('#name# used #value# records').replace('#name#', __('Fuel Consumption Report')).replace('#value#', data.data.device_logs ? data.data.device_logs.length : 0);
                showReportDataProcessingInfo(message);

                processFuelConsumptionData(reportData.filter(c => c.speed > 0));
            } else {
                showNoDataState();
                showConfirmDialog({
                    title: '<?php echo e(__("Error")); ?>',
                    message: data.message || '<?php echo e(__("Unknown error occurred")); ?>',
                    confirmText: '<?php echo e(__("OK")); ?>',
                    showCancelButton: false,
                    type: 'danger',
                    onConfirm: function() {},
                    onCancel: function() {}
                });
            }
        })
        .catch(err => {
            hideLoadingState();
            showNoDataState();
            showConfirmDialog({
                title: '<?php echo e(__("Error")); ?>',
                message: __(err.message),
                confirmText: '<?php echo e(__("OK")); ?>',
                showCancelButton: false,
                type: 'danger',
                onConfirm: function() {},
                onCancel: function() {}
            });
        });
    }

    // Process fuel consumption data
    function processFuelConsumptionData(data) {
        if (!data || data.length === 0) {
            showNoDataState();
            return;
        }

        // Calculate fuel consumption based on distance and vehicle settings
        const consumptionData = calculateFuelConsumption(data);

        if (consumptionData.length === 0) {
            showNoDataState();
            return;
        }

        // Calculate summary statistics
        const summary = calculateConsumptionSummary(consumptionData);
        
        // Display the processed data
        displayFuelConsumptionReport({
            summary: summary,
            consumptionData: consumptionData
        });
    }

    // Calculate fuel consumption from GPS data
    function calculateFuelConsumption(data) {
        const consumptionData = [];
        const vehicleGroups = {};
        const fuelTypeFilter = $('#fuelTypeFilter').val();
        
        // Group data by vehicle
        data.forEach(point => {
            if (!vehicleGroups[point.device_id]) {
                vehicleGroups[point.device_id] = [];
            }
            vehicleGroups[point.device_id].push(point);
        });
        
        // Calculate consumption for each vehicle
        Object.keys(vehicleGroups).forEach(deviceId => {
            const vehiclePoints = vehicleGroups[deviceId];
            const vehicle = vehicleList.find(v => v.did === deviceId);
            
            if (!vehicle) return;
            
            const fuelConsumption = vehicle.fuel_consumption || defaultFuelConsumption; // L/100km
            const fuelType = vehicle.fuel_type || 'gasoline';
            
            // Apply fuel type filter if selected
            if (fuelTypeFilter && fuelType !== fuelTypeFilter) {
                return;
            }
            
            let totalDistance = 0;
            let totalFuelUsed = 0;
            let totalCost = 0;
            
            // Calculate distance between consecutive points
            for (let i = 1; i < vehiclePoints.length; i++) {
                const current = vehiclePoints[i];
                const previous = vehiclePoints[i - 1];
                
                // Calculate distance in km using Haversine formula
                const distance = calculateDistance(
                    previous.latitude, previous.longitude,
                    current.latitude, current.longitude
                );
                
                totalDistance += distance;
            }
            
            // Calculate estimated fuel used based on configured consumption rate
            // This is an estimate since we don't have actual fuel level data
            const estimatedFuelUsed = (fuelConsumption * totalDistance) / 100;
            
            
            // Calculate estimated cost based on fuel type and price
            const fuelPriceKey = `fuel_price_${fuelType}`;
            const fuelPrice = fuelPrices[fuelPriceKey] ? fuelPrices[fuelPriceKey].typed_value : 0;
            const estimatedCost = estimatedFuelUsed * fuelPrice;
            
            consumptionData.push({
                device_id: deviceId,
                vehicle_name: vehicle.name,
                configured_consumption_rate: fuelConsumption, // L/100km - the configured rate
                fuel_type: fuelType,
                total_distance: totalDistance,
                estimated_fuel_used: estimatedFuelUsed, // Estimated based on distance and configured rate
                estimated_cost: estimatedCost,
                actual_efficiency: totalDistance > 0 ? (estimatedFuelUsed / totalDistance) * 100 : 0, // Calculated efficiency
                date: vehiclePoints[0].timestamp.split(' ')[0]
            });
        });
        
        return consumptionData;
    }

    // Calculate distance between two points using Haversine formula
    function calculateDistance(lat1, lon1, lat2, lon2) {
        const R = 6371; // Earth's radius in km
        const dLat = (lat2 - lat1) * Math.PI / 180;
        const dLon = (lon2 - lon1) * Math.PI / 180;
        const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                  Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
                  Math.sin(dLon/2) * Math.sin(dLon/2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
        return R * c;
    }

    // Calculate summary statistics for fuel consumption
    function calculateConsumptionSummary(consumptionData) {
        const totalDistance = consumptionData.reduce((sum, item) => sum + item.total_distance, 0);
        const totalFuelUsed = consumptionData.reduce((sum, item) => sum + item.estimated_fuel_used, 0);
        const totalCost = consumptionData.reduce((sum, item) => sum + item.estimated_cost, 0);
        const averageEfficiency = totalDistance > 0 ? (totalFuelUsed / totalDistance) * 100 : 0;
        
        return {
            total_distance: totalDistance,
            total_fuel_used: totalFuelUsed,
            total_cost: totalCost,
            average_efficiency: averageEfficiency,
            vehicle_count: consumptionData.length
        };
    }

    // Display fuel consumption report
    function displayFuelConsumptionReport(data) {
        // Show all sections
        $('#summaryCards').removeClass('d-none');
        $('#consumptionTableCard').removeClass('d-none');
        $('#chartsSection').removeClass('d-none');
        $('#mapCard').removeClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');

        // Update summary cards
        updateSummary(data.summary);

        // Update table
        updateTable(data.consumptionData);

        // Update charts
        updateCharts(data.consumptionData);

        // Update map
        updateMap(data.consumptionData);
    }

    function updateSummary(summary) {
        $('#totalDistance').text(summary.total_distance.toFixed(2) + ' ' + lengthUnit);
        $('#totalFuelUsed').text(summary.total_fuel_used.toFixed(2) + ' ' + liquidUnit);
        $('#totalCost').text(summary.total_cost.toFixed(2) + ' ' + currency);
        $('#averageEfficiency').text(summary.average_efficiency.toFixed(2) + ' ' + liquidUnit + '/100' + lengthUnit);
        
        // Calculate days count from selected date range
        const startDate = $('#startDate').val();
        const endDate = $('#endDate').val();
        
        if (startDate && endDate) {
            const start = new Date(startDate);
            const end = new Date(endDate);
            const daysDiff = Math.ceil((end - start) / (1000 * 60 * 60 * 24)) + 1; // +1 to include both start and end dates
            $('#daysCount').text(daysDiff + ' <?php echo e(__("Days")); ?>');
        } else {
            $('#daysCount').text('-');
        }
    }

    function updateTable(consumptionData) {
        // Table should already be initialized, but check just in case
        if (!consumptionTable) {
            initializeConsumptionTable();
        }
        
        const tableData = consumptionData.map((item, index) => ({
            id: index + 1,
            vehicle: item.vehicle_name,
            fuel_type: item.fuel_type,
            fuel_consumption: item.configured_consumption_rate + ' ' + liquidUnit,
            distance: item.total_distance.toFixed(2) + ' ' + lengthUnit,
            fuel_used: item.estimated_fuel_used.toFixed(2) + ' ' + liquidUnit,
            cost: item.estimated_cost.toFixed(2) + ' ' + currency,
            date: item.date
        }));

        consumptionTable.setData(tableData);
    }

    function updateMap(consumptionData) {
        if (!myMap) {
            initializeMap();
        }
        
        // Clear existing markers
        myMap.removeMarkers();
        myMap.clearAllPolygons();
        myMap.removeCircles();
        myMap.removePolyline();
        
        if (!consumptionData || consumptionData.length === 0) {
            return;
        }
        
        // Add regions and places to the map
        addRegionsAndPlacesToMap();
        
        // Create markers for each vehicle's consumption
        const markerModels = [];
        
        consumptionData.forEach((item, index) => {
            // Find the vehicle's last known position
            const vehicleData = reportData.filter(d => d.device_id === item.device_id);
            if (vehicleData.length > 0) {
                const lastPoint = vehicleData[vehicleData.length - 1];
                const lat = parseFloat(lastPoint.latitude);
                const lng = parseFloat(lastPoint.longitude);
                
                if (!isNaN(lat) && !isNaN(lng)) {
                    let markerColor = "#007bff"; // Default blue
                    
                    // Color based on fuel type
                    switch(item.fuel_type) {
                        case 'diesel': markerColor = "#28a745"; break;
                        case 'gasoline': markerColor = "#ffc107"; break;
                        case 'gas': markerColor = "#dc3545"; break;
                        case 'hybrid': markerColor = "#6f42c1"; break;
                        case 'electric': markerColor = "#17a2b8"; break;
                    }
                    
                    markerModels.push({
                        id: `consumption-${item.device_id}-${index}`,
                        markerClassName: "fuel-consumption-marker",
                        label: `<div class="badge text-white" style="background-color: ${markerColor};">${item.vehicle_name} # ${item.estimated_fuel_used.toFixed(1)}L</div>`,
                        device_id: item.device_id,
                        popup: `
                            <strong><?php echo e(__("Fuel Consumption")); ?> - ${item.vehicle_name}</strong><br>
                            <?php echo e(__("Fuel Type")); ?>: ${translateFuelType(item.fuel_type)}<br>
                            <?php echo e(__("Distance")); ?>: ${item.total_distance.toFixed(2)} km<br>
                            <?php echo e(__("Estimated Fuel Used")); ?>: ${item.estimated_fuel_used.toFixed(2)} L<br>
                            <?php echo e(__("Estimated Cost")); ?>: ${item.estimated_cost.toFixed(2)} ${currency}<br>
                            <?php echo e(__("Actual Efficiency")); ?>: ${item.actual_efficiency.toFixed(2)} L/100km
                        `,
                        coord: [lat, lng],
                        icon: {
                            name: "ti ti-gas-station",
                            color: markerColor,
                            direction: 0,
                            className: "fuel-consumption-marker",
                        }
                    });
                }
            }
        });
        
        // Add markers to map
        if (markerModels.length > 0) {
            myMap.pinMarkers(markerModels);
            // Fit map to show all markers
            setTimeout(() => myMap.fitBounds(), 100);
        }
    }

    // Initialize map
    function initializeMap() {
        myMap = new LeafletMap();
        myMap.initMap("route-map", {
            // Default zoom level
        });
    }

    // Initialize table
    async function initializeConsumptionTable() {
        consumptionTable = createTabulator("#consumption-table", {
            initialSort: [
                {column: "distance", dir: "desc"}
            ],
            columns: [
                {title: "#", field: "id", width: 50, hozAlign: "center"},
                {
                    title: "<?php echo e(__('Vehicle')); ?>", 
                    field: "vehicle"
                },
                {
                    title: "<?php echo e(__('Fuel Type')); ?>", 
                    field: "fuel_type", 
                    hozAlign: "center",
                    formatter: function(cell) {
                        const type = cell.getValue();
                        const colors = {
                            'diesel': 'bg-success',
                            'gasoline': 'bg-warning',
                            'gas': 'bg-danger',
                            'hybrid': 'bg-primary',
                            'electric': 'bg-info'
                        };
                        return `<span class="badge text-white ${colors[type] || 'bg-secondary'}">${translateFuelType(type)}</span>`;
                    }
                },
                {
                    title: "<?php echo e(__('Configured Consumption')); ?>", 
                    field: "fuel_consumption", 
                    hozAlign: "center"
                },
                {
                    title: "<?php echo e(__('Distance')); ?>", 
                    field: "distance", 
                    hozAlign: "center"
                },
                {
                    title: "<?php echo e(__('Fuel Used')); ?>", 
                    field: "fuel_used", 
                    hozAlign: "center"
                },
                {
                    title: "<?php echo e(__('Cost')); ?>", 
                    field: "cost", 
                    hozAlign: "center"
                }
            ]
        });
    }

    // Add regions and places to the map
    function addRegionsAndPlacesToMap() {
        // Load regions data
        http.post({
            url: '/regions-list',
            loading: false
        }).then(function(regionsResponse) {
            if (regionsResponse.success && regionsResponse.data) {
                const regions = regionsResponse.data;
                
                // Add regions to map
                regions.forEach(function(region) {
                    if (region.polygon_json) {
                        myMap.addPolygon({
                            id: region.id,
                            label: region.name,
                            color: "gray",
                            fillColor: "gray",
                            coords: region.polygon_json,
                        });
                    }
                });
            }
        }).catch(function(error) {
            console.error('Error loading regions:', error);
        });

        // Load places data
        http.post({
            url: '/places-list',
            loading: false
        }).then(function(placesResponse) {
            if (placesResponse.success && placesResponse.data) {
                const places = placesResponse.data;
                
                // Add places to map
                var placeMarkerColor = "gray";
                places.forEach(function(place) {
                    if (place.latitude && place.longitude) {
                        var marker = [
                            {
                                id: place.id,
                                markerClassName: "place-marker",
                                device_id: 0,
                                label: place.name,
                                status: place.status_id,
                                popup: place.name,
                                coord: [
                                    parseFloat(place.latitude),
                                    parseFloat(place.longitude),
                                ],
                                icon: {
                                    name: "fa-solid fa-location-dot",
                                    color: placeMarkerColor,
                                    direction: 0,
                                    className: "",
                                },
                            },
                        ];
                        myMap.pinMarkers(marker);
                        
                        // Add circle if radius is defined
                        if (place.radius && place.radius > 0) {
                            var circle = L.circle([place.latitude, place.longitude], {
                                radius: place.radius,
                                color: placeMarkerColor,
                                fillColor: placeMarkerColor,
                                fillOpacity: 0.2,
                            }).addTo(myMap.mapLet);
                        }
                    }
                });
            }
        }).catch(function(error) {
            console.error('Error loading places:', error);
        });
    }

    // Show loading state
    function showLoadingState() {
        $('#loadingState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#consumptionTableCard').addClass('d-none');
        $('#chartsSection').addClass('d-none');
        $('#mapCard').addClass('d-none');
        $('#noDataState').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
    }

    // Hide loading state
    function hideLoadingState() {
        $('#loadingState').addClass('d-none');
    }

    // Show no data state
    function showNoDataState() {
        $('#noDataState').removeClass('d-none');
        $('#summaryCards').addClass('d-none');
        $('#consumptionTableCard').addClass('d-none');
        $('#chartsSection').addClass('d-none');
        $('#mapCard').addClass('d-none');
        $('#noVehicleState').addClass('d-none');
    }

    // Update charts with consumption data
    function updateCharts(consumptionData) {
        if (!consumptionData || consumptionData.length === 0) return;
        
        // Create daily consumption chart
        createDailyConsumptionChart(consumptionData);
        
        // Create fuel type distribution chart
        createFuelTypeChart(consumptionData);
        
        // Create cost analysis chart
        createCostAnalysisChart(consumptionData);
        
        // Create vehicle comparison chart
        createVehicleComparisonChart(consumptionData);
    }

    // Create daily consumption chart (line chart)
    function createDailyConsumptionChart(consumptionData) {
        if (typeof ApexCharts === 'undefined') {
            console.warn('ApexCharts is not loaded');
            return;
        }

        // Group consumption by date
        const dateGroups = {};
        consumptionData.forEach(item => {
            const date = item.date;
            if (!dateGroups[date]) {
                dateGroups[date] = {
                    fuel: 0,
                    distance: 0
                };
            }
            dateGroups[date].fuel += item.estimated_fuel_used;
            dateGroups[date].distance += item.total_distance;
        });

        const dates = Object.keys(dateGroups).sort();
        const fuelData = dates.map(date => parseFloat(dateGroups[date].fuel.toFixed(2)));
        const distanceData = dates.map(date => parseFloat(dateGroups[date].distance.toFixed(2)));

        const options = {
            series: [
                {
                    name: '<?php echo e(__("Fuel Used (L)")); ?>',
                    data: fuelData
                },
                {
                    name: '<?php echo e(__("Distance (km)")); ?>',
                    data: distanceData
                }
            ],
            chart: {
                type: 'line',
                height: 300,
                toolbar: {
                    show: false
                }
            },
            colors: ['#dc3545', '#007bff'],
            dataLabels: {
                enabled: false
            },
            stroke: {
                curve: 'smooth',
                width: 3
            },
            xaxis: {
                categories: dates.map(date => dateTimeFormatter(date)),
                title: {
                    text: '<?php echo e(__("Date")); ?>'
                }
            },
            yaxis: [
                {
                    title: {
                        text: '<?php echo e(__("Fuel Used (L)")); ?>'
                    },
                    labels: {
                        formatter: function(val) {
                            return val.toFixed(2);
                        }
                    }
                },
                {
                    opposite: true,
                    title: {
                        text: '<?php echo e(__("Distance (km)")); ?>'
                    },
                    labels: {
                        formatter: function(val) {
                            return val.toFixed(2);
                        }
                    }
                }
            ],
            legend: {
                position: 'top'
            },
            tooltip: {
                shared: true,
                intersect: false,
                y: {
                    formatter: function(val) {
                        return val.toFixed(2);
                    }
                }
            }
        };

        if (window.dailyConsumptionChart && typeof window.dailyConsumptionChart.destroy === 'function') {
            window.dailyConsumptionChart.destroy();
        }
        
        const chartElement = document.querySelector("#dailyConsumptionChart");
        if (chartElement) {
            window.dailyConsumptionChart = new ApexCharts(chartElement, options);
            window.dailyConsumptionChart.render();
        }
    }

    // Create fuel type distribution chart (pie chart)
    function createFuelTypeChart(consumptionData) {
        if (typeof ApexCharts === 'undefined') {
            console.warn('ApexCharts is not loaded');
            return;
        }

        const fuelTypeCounts = {};
        consumptionData.forEach(item => {
            if (!fuelTypeCounts[item.fuel_type]) {
                fuelTypeCounts[item.fuel_type] = 0;
            }
            fuelTypeCounts[item.fuel_type] += item.estimated_fuel_used;
        });

        const fuelTypes = Object.keys(fuelTypeCounts);
        const fuelAmounts = Object.values(fuelTypeCounts).map(val => parseFloat(val.toFixed(2)));
        const translatedLabels = fuelTypes.map(fuelType => translateFuelType(fuelType));

        const options = {
            series: fuelAmounts,
            chart: {
                type: 'pie',
                height: 300
            },
            labels: translatedLabels,
            colors: ['#28a745', '#ffc107', '#dc3545', '#6f42c1', '#17a2b8'],
            legend: {
                position: 'bottom'
            },
            tooltip: {
                y: {
                    formatter: function (val) {
                        return val.toFixed(2) + ' L';
                    }
                }
            },
            plotOptions: {
                pie: {
                    donut: {
                        size: '60%'
                    }
                }
            }
        };

        if (window.fuelTypeChart && typeof window.fuelTypeChart.destroy === 'function') {
            window.fuelTypeChart.destroy();
        }
        
        const chartElement = document.querySelector("#fuelTypeChart");
        if (chartElement) {
            window.fuelTypeChart = new ApexCharts(chartElement, options);
            window.fuelTypeChart.render();
        }
    }

    // Create cost analysis chart (bar chart)
    function createCostAnalysisChart(consumptionData) {
        if (typeof ApexCharts === 'undefined') {
            console.warn('ApexCharts is not loaded');
            return;
        }

        // Group by date
        const dateGroups = {};
        consumptionData.forEach(item => {
            const date = item.date;
            if (!dateGroups[date]) {
                dateGroups[date] = 0;
            }
            dateGroups[date] += item.estimated_cost;
        });

        const dates = Object.keys(dateGroups).sort();
        const costs = Object.values(dateGroups).map(val => parseFloat(val.toFixed(2)));

        const options = {
            series: [{
                name: '<?php echo e(__("Cost (TRY)")); ?>',
                data: costs
            }],
            chart: {
                type: 'bar',
                height: 300,
                toolbar: {
                    show: false
                }
            },
            colors: ['#28a745'],
            plotOptions: {
                bar: {
                    horizontal: false,
                    columnWidth: '70%',
                    endingShape: 'rounded'
                }
            },
            dataLabels: {
                enabled: false
            },
            xaxis: {
                categories: dates.map(date => dateTimeFormatter(date)),
                title: {
                    text: '<?php echo e(__("Date")); ?>'
                }
            },
            yaxis: {
                title: {
                    text: '<?php echo e(__("Cost (TRY)")); ?>'
                },
                labels: {
                    formatter: function(val) {
                        return val.toFixed(2);
                    }
                }
            },
            tooltip: {
                y: {
                    formatter: function (val) {
                        return val.toFixed(2) + ' TRY';
                    }
                }
            }
        };

        if (window.costAnalysisChart && typeof window.costAnalysisChart.destroy === 'function') {
            window.costAnalysisChart.destroy();
        }
        
        const chartElement = document.querySelector("#costAnalysisChart");
        if (chartElement) {
            window.costAnalysisChart = new ApexCharts(chartElement, options);
            window.costAnalysisChart.render();
        }
    }

    // Create vehicle comparison chart (horizontal bar chart)
    function createVehicleComparisonChart(consumptionData) {
        if (typeof ApexCharts === 'undefined') {
            console.warn('ApexCharts is not loaded');
            return;
        }

        const vehicleData = {};
        consumptionData.forEach(item => {
            if (!vehicleData[item.vehicle_name]) {
                vehicleData[item.vehicle_name] = {
                    fuel: 0,
                    cost: 0
                };
            }
            vehicleData[item.vehicle_name].fuel += item.estimated_fuel_used;
            vehicleData[item.vehicle_name].cost += item.estimated_cost;
        });

        const vehicles = Object.keys(vehicleData);
        const fuelData = vehicles.map(v => parseFloat(vehicleData[v].fuel.toFixed(2)));
        const costData = vehicles.map(v => parseFloat(vehicleData[v].cost.toFixed(2)));

        const options = {
            series: [
                {
                    name: '<?php echo e(__("Fuel Used (L)")); ?>',
                    data: fuelData
                },
                {
                    name: '<?php echo e(__("Cost (TRY)")); ?>',
                    data: costData
                }
            ],
            chart: {
                type: 'bar',
                height: 300,
                stacked: false,
                toolbar: {
                    show: false
                }
            },
            colors: ['#007bff', '#28a745'],
            plotOptions: {
                bar: {
                    horizontal: true,
                    dataLabels: {
                        total: {
                            enabled: true,
                            offsetX: 0,
                            style: {
                                fontSize: '13px',
                                fontWeight: 900
                            }
                        }
                    }
                }
            },
            stroke: {
                width: 1,
                colors: ['#fff']
            },
            xaxis: {
                categories: vehicles,
                title: {
                    text: '<?php echo e(__("Amount")); ?>'
                },
                labels: {
                    formatter: function(val) {
                        return parseFloat(val).toFixed(2);
                    }
                }
            },
            yaxis: {
                title: {
                    text: '<?php echo e(__("Vehicle")); ?>'
                }
            },
            tooltip: {
                shared: false,
                y: {
                    formatter: function (val) {
                        return val.toFixed(2);
                    }
                }
            },
            legend: {
                position: 'top'
            }
        };

        if (window.vehicleComparisonChart && typeof window.vehicleComparisonChart.destroy === 'function') {
            window.vehicleComparisonChart.destroy();
        }
        
        const chartElement = document.querySelector("#vehicleComparisonChart");
        if (chartElement) {
            window.vehicleComparisonChart = new ApexCharts(chartElement, options);
            window.vehicleComparisonChart.render();
        }
    }

    function showVehicleSettingsModal(vehiclesWithoutFuelData) {
        const modalBody = $('#vehicleSettingsList');
        modalBody.empty();

        console.log(vehiclesWithoutFuelData);
        
        vehiclesWithoutFuelData.forEach(vehicle => {
            const vehicleHtml = `
                <div class="card mb-3">
                    <div class="card-header">
                        <h6 class="card-title mb-0">${vehicle.name}</h6>
                    </div>
                    <div class="card-body">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label"><?php echo e(__('Fuel Type')); ?></label>
                                <select class="form-select vehicle-fuel-type" data-vehicle-id="${vehicle.did}">
                                    <option value=""><?php echo e(__('Select Fuel Type')); ?></option>
                                    <option value="diesel" ${vehicle.fuelType === 'diesel' ? 'selected' : ''}><?php echo e(__('Diesel')); ?></option>
                                    <option value="gasoline" ${vehicle.fuelType === 'gasoline' ? 'selected' : ''}><?php echo e(__('Gasoline')); ?></option>
                                    <option value="gas" ${vehicle.fuelType === 'gas' ? 'selected' : ''}><?php echo e(__('Gas (LPG/CNG)')); ?></option>
                                    <option value="hybrid" ${vehicle.fuelType === 'hybrid' ? 'selected' : ''}><?php echo e(__('Hybrid')); ?></option>
                                    <option value="electric" ${vehicle.fuelType === 'electric' ? 'selected' : ''}><?php echo e(__('Electric')); ?></option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label"><?php echo e(__('Fuel Consumption (L/100km)')); ?></label>
                                <input type="number" class="form-control vehicle-fuel-consumption" 
                                       data-vehicle-id="${vehicle.did}" 
                                       value="${vehicle.fuelConsumption}" 
                                       min="1" max="50" step="0.1">
                            </div>
                        </div>
                    </div>
                </div>
            `;
            modalBody.append(vehicleHtml);
        });
        
        $('#vehicleSettingsModal').modal('show');
    }

    function saveVehicleSettings() {
        const vehicleSettings = [];
        let hasErrors = false;
        
        $('.vehicle-fuel-type').each(function() {
            const vehicleId = $(this).data('vehicle-id');
            const fuelType = $(this).val();
            const fuelConsumption = $(`.vehicle-fuel-consumption[data-vehicle-id="${vehicleId}"]`).val();
            
            if (!fuelType || !fuelConsumption) {
                hasErrors = true;
                return false;
            }
            
            vehicleSettings.push({
                device_id: vehicleId,
                fuel_type: fuelType,
                fuel_consumption: parseFloat(fuelConsumption)
            });
        });
        
        if (hasErrors) {
            showNotification("error", "<?php echo e(__('Please fill in all fuel type and consumption fields')); ?>");
            return;
        }
        
        // Save vehicle settings
        const postData = Base64Helper.encode(vehicleSettings);
        
        http.post({
            url: "/vehicle-update-settings",
            data: {
                data: postData
            }
        })
        .then(function(response) {
            if (response.success) {
                showNotification("success", "<?php echo e(__('Vehicle fuel settings updated successfully')); ?>");
                $('#vehicleSettingsModal').modal('hide');
                // Reload vehicles to reflect changes
                loadVehicles();
            } else {
                showNotification("error", response.message || "<?php echo e(__('Error updating vehicle settings')); ?>");
            }
        })
        .catch(function(error) {
            console.error("Vehicle settings update error:", error);
            showNotification("error", "<?php echo e(__('Error updating vehicle settings')); ?>");
        });
    }
</script>
<?php $__env->stopSection(); ?>
<?php echo $__env->make('layouts.app', \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?><?php /**PATH C:\xampp\htdocs\aractakipv1\resources\views\reports\fuel-consumption.blade.php ENDPATH**/ ?>